/** @package Parallel Skip List implementation in Fork95.
 *  @author  C.W. Kessler    03/1999
 *  References:
 *   [1] W. Pugh, Communications of the ACM 1990.
 *   [2] M. Weiss, Data Structures and Algorithm Analysis, 2nd ed.,
 *                 Benjamin-Cummings, 1994.
 */
#include <fork.h>
#include <stdlib.h>
#include <io.h>
#include <assert.h>

#include "skip.h"

#define DEBUG 0  // 1 if debugging (lots of diagnostic output), 0 otherwise
#define SILENCE 1


/** @method new_PSkipListItem
 *  constructor for skip list nodes. 
 *  @param  int height: the height of the node being allocated
 *  @param  Key key:  the initializer for the key entry.
 *  @return PSkipListItem ret: the new node
 */
PSkipListItem new_PSkipListItem( int height, Key key )
{
  PSkipListItem ret;
  int i;
  if (height <= 0) {
     pprintf("new_PSkipListItem(): height %d is nonsense\n", height);
     return NULL;
  }
  ret = (PSkipListItem) shmalloc( sizeof ( struct skiplistnode) );
  ret->height = height;
  ret->next = (PSkipListItem *) shmalloc( height * sizeof(PSkipListItem)); 
  for (i=0; i<height; i++)
     ret->next[i] = NULL;
  ret->lock = new_rwd_lock();
  ret->key = key;
  ret->inf = NULL;
  return ret;
}


/** @method PSkipListItemFree
 *  destructor for skip list nodes. 
 *  @param  PSkipListItem n: the node being destructed
 */
void PSkipListItemFree( PSkipListItem n )
{
  int i;
  shfree( n->next );
  shfree( n->lock );
  shfree( n );
}


/** @method PSkipListItemPrint
 *  prints a single skip list node n to standard output.
 *  @param  PSkipListItem n
 */
void PSkipListItemPrint( PSkipListItem n )
{
  int i;
  printf("Item %p: height %d, key %d, inf %d\n next: [",
           n, n->height, (int) n->key, (int) n->inf );
  for (i=0; i<n->height; i++) printf(" %p,", n->next[i] );
  printf("]\n");
}


extern PSkipList new_PSkipList( int (*cmp)(Key,Key), int maxh, Key minKey );
extern void PSkipListPrint( PSkipList l );
extern int PSkipListIsEmpty( PSkipList l );
extern int PSkipListGetSize( PSkipList l );
extern PSkipListItem PSkipListLocatePred( PSkipList l, Key key );
extern PSkipListItem PSkipListLocate( PSkipList l, Key key );
extern Inf PSkipListAccess( PSkipList l, Key key );
extern void PSkipListLocateAndWLock( PSkipList l, Key key, int h, PSkipListItem ptr[] );
extern PSkipListItem PSkipListInsert( PSkipList l, Key key, Inf inf );
extern PSkipListItem PSkipListDelete( PSkipList l, Key key );
extern PSkipListItem PSkipListFindMin( PSkipList l );
extern PSkipListItem PSkipListDeleteMin( PSkipList l );
extern sync int PSkipListDeleteMins( PSkipList l, int k, PSkipListItem * );
extern PSkipListItem PSkipListChangeInf( PSkipList l, Key key, Inf newInf );
extern PSkipListItem PSkipListDecreaseKey( PSkipList l, Key oldkey, Key newKey );


/** @method new_PSkipList
 *  constructor for a skip list.
 *  @param  comp: pointer to a function that compares two key values
 *  @param  int maxh: the maximal height of a skip list node 
 *  @param  Key minKey: a minimal key value 
 *  @return ret:  a new, empty skip list
 *  @note   The programmer must ensure that minKey will never be inserted,
 *          looked up, or deleted from the PSkipList.
 */
PSkipList new_PSkipList( int (*comp)(Key,Key), int maxh, Key minKey )
{
  PSkipList ret;
  int i;
  if (!comp) {
     pprintf("new_PSkipList(): bad comp() function\n");
     return NULL;
  }
  ret = (PSkipList) shmalloc( sizeof ( struct skiplist) );
  ret->maxHeight = maxh;
  if (maxh > PSKIPLISTMAXHEIGHT) {
     pprintf("new_PSkipList(): 2nd parameter exceeds PSKIPLISTMAXHEIGHT\n");
     return NULL;
  }
  ret->head = new_PSkipListItem( maxh, minKey );
  ret->lock = new_rwd_lock();
  ret->cmp = comp;
  ret->Size = 0;     // the artificial first node is not counted.
  ret->insert = PSkipListInsert;
  ret->locate = PSkipListLocate;
  ret->pred = PSkipListLocatePred;
  ret->delete = PSkipListDelete;
  ret->print = PSkipListPrint;
  ret->empty = PSkipListIsEmpty;
  ret->size = PSkipListGetSize;
  ret->deleteMin = PSkipListDeleteMin;
  ret->deleteMins = PSkipListDeleteMins;
  ret->changeInf = PSkipListChangeInf;
  ret->decreaseKey = PSkipListDecreaseKey;
  return ret;
}


/** @method PSkipListPrint
 *  prints the entire skip list l to standard output.
 *  @param  PSkipList l
 */
void PSkipListPrint( PSkipList l )
{
  PSkipListItem p;
  printf("\n------ PSkipList (Size %d, maxHeight %d, cmp %p): ------------\n",
          l->Size, l->maxHeight, l->cmp );
  // ...
  p = l->head->next[0];   // do not print the artificial first node.
  while (p) {
     PSkipListItemPrint( p );
     p = p->next[0];
  }
  printf("------- End of PSkipList ---------------\n");
}


/** @method PSkipListIsEmpty
 *  reports whether the skip list l contains an element
 *  @param  PSkipList l
 *  @return int ret: 1 if l is currently empty, 0 otherwise.
 */
int PSkipListIsEmpty( PSkipList l )
{
  int ret = 1;
  while (! rwd_lockup( l->head->lock, RW_READ )) ;
  if (l->head->next[0]) 
     ret = 0;
  rwd_unlock( l->head->lock, RW_READ, 3 );
  return ret;
}


/** @method PSkipListGetSize
 *  reports the current size of skip list l
 *  @param  PSkipList l
 *  @return int ret: the current number of elements in l
 *                   plus #pending inserts minus #pending deletes
 */
int PSkipListGetSize( PSkipList l )
{
  return l->Size;
}


/** @method PSkipListFindMin
 *  reports the PSkipListItem ret=(key,inf) with minimal key value,
 *  i.e. that is currently stored at the head of the skip list l 
 *  @param  PSkipList l
 *  @return PSkipListItem ret
 */
PSkipListItem PSkipListFindMin( PSkipList l )
{
  return l->head->next[0];
  // no collisions possible, since write access to next[0] pointers 
  // (e.g. in insert() and delete()) is always atomic.
}


/** @method PSkipListLocatePred
 *  find an element (k,inf) whose key entry k is maximal but still
 *  less than a given key value
 *  @param  PSkipList l
 *  @param  Key key:  the key to be located.
 *  @return PSkipListItem ret: pointer to that item (k,inf) if it exists in l,
 *                             and NULL otherwise
 *  @note   if several items with same key k exist in l,
 *          the routine returns a pointer to the _first_ of these.
 */
PSkipListItem PSkipListLocatePred( PSkipList l, Key key )
{
  PSkipListItem p, q, oldp;
  int i;
  p = l->head; 
  while (!rwd_lockup( p->lock, RW_READ )) ;
  for (i = l->maxHeight-1; i>=0; i--) { // search in list level i:
     // invariant on entry: p with height >=i is r-locked.
     q = p->next[i];
     while( q ) {
        if ( l->cmp( q->key, key ) >= 0 ) break;
        oldp = p;
        if (!rwd_lockup( q->lock, RW_READ )) continue;
        p = q;
        q = q->next[i];
        rwd_unlock( oldp->lock, RW_READ, 3 );
     }
  }
  rwd_unlock( p->lock, RW_READ, 3 );
  return p;
}
 

/** @method PSkipListLocate
 *  find an element (k,inf) with key entry k equal to given key
 *  @param  PSkipList l
 *  @param  Key key:  the key of the element to be located.
 *  @return PSkipListItem ret: pointer to that item (k,inf) if it exists in l,
 *                             and NULL otherwise
 *  @note   if several items with same key key exist in l,
 *          the routine returns a pointer to the first one.
 */
PSkipListItem PSkipListLocate( PSkipList l, Key key )
{
  PSkipListItem p, q, oldp;
  int i;
  p = l->head; 
  while (!rwd_lockup( p->lock, RW_READ )) ;
  for (i = l->maxHeight-1; i>=0; i--) { // search in list level i:
     // invariant on entry: p with height >=i is r-locked.
     q = p->next[i];
     while( q ) {
        if ( l->cmp( q->key, key ) > 0 ) break;
        if ( l->cmp( q->key, key ) == 0 ) {
           rwd_unlock( p->lock, RW_READ, 3 );
           return q;
        }
        oldp = p;
        if (!rwd_lockup( q->lock, RW_READ )) continue;
        p = q;
        q = q->next[i];
        rwd_unlock( oldp->lock, RW_READ, 3 );
     }
  }
  rwd_unlock( p->lock, RW_READ, 3 );
  if (l->cmp( p->key, key ) != 0)  return NULL;
  return p;
}


/** @method PSkipListAccess
 *  find information for a given key value if stored in skip list l
 *  @param  PSkipList l
 *  @param  Key key:  the key of the element to be located.
 *  @return Inf ret:  the information entry inf of item (key,inf) 
 *                      if it exists in l, and NULL otherwise
 *  @note   if several items with same key key exist in l,
 *          the routine returns the inf entry of the first one.
 */
Inf PSkipListAccess( PSkipList l, Key key )
{
  PSkipListItem p;
  p = PSkipListLocate( l, key );
  if (!p) return NULL;
  return p->inf;
}


/** @method PSkipListLocateAndWLock
 *  w-locks the h predecessor nodes on list level 0,...,h-1
 *  whose key entry is less than key
 *  and whose successors' key entry is greater or equal than key.
 *  The routine iterates until all nodes can be locked successfully.
 *  @param  PSkipList l
 *  @param  Key key: the key of the node being inserted
 *  @param  int h: the height of the node being inserted
 *  @param  PSkipListItem ptr[]: returns the pointers to the predecessors
 *  @note   The same node may occur at several subsequent levels of ptr[].
 */
void PSkipListLocateAndWLock( PSkipList l, Key key, int h,
                              PSkipListItem ptr[] )
{
  PSkipListItem p, q;
  int i;
#if DEBUG
  pprintf("LocateAndWLock( key %d, h %d)\n", (int) key, h );
#endif
  // predecessors in lists leveled 0 to h-1 are locked in write mode.

LOCATE_RETRY:     // jump back to here whenever a lockup fails:
#if DEBUG
  pprintf("LOCATE_RETRY:\n");
#endif
  p = l->head; 
  while (!rwd_lockup( p->lock, RW_READ )) ;
  for (i = l->maxHeight-1; i>=h; i--) { // search in list level i:
     // invariant on entry: p with height >=i is r-locked.
#if DEBUG
     pprintf("Iteration %d\n", i);
#endif
     q = p->next[i];
     while( q ) {
#if DEBUG
        pprintf("iteration %d, cmp next\n", i);
#endif
        if ( l->cmp( q->key, key ) >= 0 ) break;
        else {
           PSkipListItem oldp = p;
           if (!rwd_lockup( q->lock, RW_READ )) continue;
           p = q;
           q = q->next[i];
           rwd_unlock( oldp->lock, RW_READ, 3 );
        }
     }
     ptr[i] = p;
  }
  // p as resulting from the previous computation
  rwd_unlock( p->lock, RW_READ, 3 );  // r-unlock node p in height h
  if (!rwd_lockup( p->lock, RW_WRITE )) {
      pprintf("@@@@@ rwd_lock( %p ) FAILED, RETRY...\n", p );
      goto LOCATE_RETRY;
  }
  for (i = h-1; i>=0; i--) {  // search in list level i:
     // invariant on entry: p with height h-1..i+1 is w-locked.
#if DEBUG
     pprintf("Iteration %d\n", i);
#endif
     q = p->next[i];
     while( q ) {
#if DEBUG
        pprintf("iteration %d, cmp next\n", i);
#endif
        if ( l->cmp( q->key, key ) >= 0 ) break;
        else {
           PSkipListItem oldp = p;
           if (!rwd_lockup( q->lock, RW_WRITE )) continue;
           p = q;
           q = q->next[i];
           if (i==h-1 || oldp != ptr[i+1])
              rwd_unlock( oldp->lock, RW_WRITE, 3 );
        }
     }
     ptr[i] = p;
#if DEBUG
     pprintf("ptr[%d] = %p\n", i, p );
#endif
  }
}


/** @method PSkipListInsert
 *  inserts a new item (key,inf) in the skip list l.
 *  @param  PSkipList l
 *  @param  Key key: insert in order of non-decreasing key values
 *  @param  Inf inf: the non-key data to be associated with this entry
 *  @return pointer to the inserted item (key,inf) if insertion succeeded,
 *          and NULL otherwise.
 */
PSkipListItem PSkipListInsert( PSkipList l, Key key, Inf inf )
{
  int i, j, h;
  PSkipListItem neu, p, first;
  PSkipListItem ptr[PSKIPLISTMAXHEIGHT];  // fast alloc on stack
#if DEBUG
  pprintf("Insert( key=%d, inf=%d)\n", (int)key, (int)inf);
#endif
  if (!l) return NULL;

  // by coin flipping determine height h stochastically:
  h=1;
  while (rand() & 0x8) h++;  // prob(h=H) = 2^(-H), H>1
  if (h>l->maxHeight) h=l->maxHeight;

  neu = new_PSkipListItem( h, key );
  neu->inf = inf; 
  syncadd( &(l->Size), 1 );

  // lock the h predecessor nodes on level 0,...,h-1 that
  // have their key less than neu->key less/equal than their successors' key.
  // The returned ptr array contains the pointers to them.
  // Note that the same node may occur at several levels.

  PSkipListLocateAndWLock( l, key, h, ptr );  
  // this routine iterates until a ptr array can be returned successfully.
#if (!SILENCE)
  pprintf("...inserting %p with %d in level 0..%d\n", neu, (int)key, h-1);
#endif
  for (j=0; j<h; j++) {
     neu->next[j] = ptr[j]->next[j];
     ptr[j]->next[j] = neu;
  }
  rwd_unlock( ptr[0]->lock, RW_WRITE, 3);
  for (j=1; j<h; j++) {
     if (ptr[j]==ptr[j-1]) continue;  // unlock the same node only once
     rwd_unlock( ptr[j]->lock, RW_WRITE, 3);
  }
  return neu;
}


/** @method PSkipListDelete
 *  removes an item (key,inf) from the skip list l, if one exists in l.
 *  @param  PSkipList l
 *  @param  Key key
 *  @return pointer to the removed item if successful, and NULL otherwise.
 *  @note   if several items with same key are in l, only one is removed.
 */
PSkipListItem PSkipListDelete( PSkipList l, Key key )
{
  int j, h;
  PSkipListItem ret;
  PSkipListItem ptr[PSKIPLISTMAXHEIGHT];  // fast alloc on stack
#if DEBUG
  pprintf("Delete( key=%d )\n", (int)key);
#endif
  if (!l) return NULL;

  // We can't know here the exact height of the node to be deleted.
  // Workaround: Try initially with h = l->maxHeight / 2.
  //             This still allows many other operations execute concurrently.
  // Search for ret. If not available, fail.
  // If available, take height from ret and enter the deletion process.
  // If then the actual height is found to be greater than h,
  // something has happened; redo until this case does not occur any more.

TRY_AGAIN_DELETE:
  ret = l->locate( l, key );
  if (!ret) return NULL;
  h = ret->height;

  // lock the h predecessor nodes on level 0,...,h-1 that
  // have their key less than neu->key less/equal than their successors' key.
  // The returned ptr array contains the pointers to them.
  // Note that the same node may occur at several levels.

  PSkipListLocateAndWLock( l, key, h, ptr );  
  // this routine iterates until a ptr array can be returned successfully.

  if ( l->cmp( ptr[0]->next[0]->key, key ) == 0 ) { // deletee found.
     if (! rwd_lockup( ptr[0]->next[0]->lock, RW_DELETE ))
        goto DELETEE_NOT_FOUND;  // another deletor was faster than me
     ret = ptr[0]->next[0];
     if ( l->cmp( ret->key, key ) != 0  // something has been inserted meanwhile
         || ret->height > h      ) {  // or the height was higher than expected:
        // ret will never again be unlocked, to avoid non-consistency
        pprintf("$$$$$ TRY_AGAIN_DELETE, key=%d h=%d->%d\n", key, h, ret->height );
        rwd_unlock( ptr[0]->lock, RW_WRITE, 3);
        for (j=1; j<h; j++) {
           if (ptr[j]==ptr[j-1]) continue;  // unlock the same node only once
           rwd_unlock( ptr[j]->lock, RW_WRITE, 3);
        }
        goto TRY_AGAIN_DELETE; 
     }
#if (!SILENCE)
     pprintf("...deleting %p with %d in level 0..%d\n", ret, (int)key, h-1);
#endif
     syncadd( &(l->Size), -1 );
     // unlink ret == ptr[0]->next[0]:
     for (j=0; j<ret->height; j++)
        ptr[j]->next[j] = ret->next[j];
     // ret will never again be unlocked, to avoid non-consistency
     rwd_unlock( ptr[0]->lock, RW_WRITE, 3);
     for (j=1; j<h; j++) {
        if (ptr[j]==ptr[j-1]) continue;  // unlock the same node only once
        rwd_unlock( ptr[j]->lock, RW_WRITE, 3);
     }
     return ret;
  }
  else {
DELETEE_NOT_FOUND:
#if (!SILENCE)
     pprintf("...could not delete %d\n", (int)key );
#endif
     rwd_unlock( ptr[0]->lock, RW_WRITE, 3);
     for (j=1; j<h; j++) {
        if (ptr[j]==ptr[j-1]) continue;  // unlock the same node only once
        rwd_unlock( ptr[j]->lock, RW_WRITE, 3);
     }
     return NULL;
  }
}


/** @method PSkipListDeleteMin
 *  delete and return the item with minimum key entry from the skip list l.
 *  @param  PSkipList l 
 *  @return PSkipListItem ret: if l is nonempty, the node with minimum key
 *                             or NULL if l is empty.
 *  @note   does not always work strictly in the temporal order of calls.
 *          The minimum key item directly follows the artificial first node.
 */
PSkipListItem PSkipListDeleteMin( PSkipList l )
{
  PSkipListItem ret;
  int i;
  rwd_lockup( l->head->lock, RW_WRITE );
  if (! l->head->next[0]) {
     rwd_unlock( l->head->lock, RW_WRITE, 3 );
     return NULL;
  }
  syncadd( &(l->Size), -1 );
  if (! rwd_lockup( l->head->next[0]->lock, RW_DELETE ))  {
      rwd_unlock( l->head->lock, RW_WRITE, 3 );
      return NULL;             // another deletor was faster than me
  }
  ret = l->head->next[0];
#if (!SILENCE)
  pprintf("...deleteMin %d in level 0..%d\n", (int)ret->key, ret->height-1);
#endif
  for (i=0; i<ret->height; i++)
     l->head->next[i] = ret->next[i];
  rwd_unlock( l->head->lock, RW_WRITE, 3 );
  // ret is never unlocked again, to avoid non-consistency
  // rwd_unlock( ret->lock, RW_DELETE, 3 );
  return ret;
}


/** @method PSkipDeleteMins
 *  extract the k items with minimal key values as a whole from the skip list l.
 *  @param  PSkipList l
 *  @param  int k:  a positive integer
 *  @param  PSkipListItem *ret: an array of >=k pointers to skip list nodes,
 *                              where the found pointers are written to if k>0,
 *                              and NULL otherwise
 *  @return int i: the number of found and assigned min items, i<=k.
 */
sync int PSkipListDeleteMins( PSkipList l, int k, PSkipListItem *ret )
{
  PSkipListItem p;
  int j, m;
  sh int i;
  if (k<1) return 0;

  // I use a simple sequential implementation for the first.
  // If later a sophisticated synchronous implementation is added,
  // it will have to be competitive with respect to this simple variant.
  // Note that a parallel implementation requires information about
  // sublist length. The computation and updating of this information
  // may slow down the insert and delete operations considerably.
  // For k not very large it is unlikely that a parallel implementation
  // would outperform this simple sequential variant.
  seq {
    pprintf("call deleteMins(%d)\n", k );
    assert( rwd_lockup( l->head->lock, RW_WRITE ));  
    i = 0;
    p = l->head->next[0];
    m = 1;
    while (p && (i < k)) {
      if (p->height > m)  m = p->height; 
      if (!rwd_lockup( p->lock, RW_DELETE )) {
#if (!SILENCE)
         pprintf("!!!!! skipping a node in deleteMins(%d)\n", k);
#endif
         p = p->next[0];
         continue;
      }
#if (!SILENCE)
      pprintf("...deleteMins found %p with %d\n", p, (int)(p->key));
#endif
      ret[i++] = p;
      p = p->next[0];
    }
    syncadd( &(l->Size), -i );
    // unlink ret[0]...ret[i-1]:
    for (j=0; j<m; j++)
       l->head->next[j] = ret[i-1]->next[j];
    rwd_unlock( l->head->lock, RW_WRITE, 3 );
  }
  return i;
}


/** @method PSkipListDecreaseKey
 *  changes the key of an existing item (key,inf) to a new value k
 *  @param  PSkipList l
 *  @param  Key oldKey
 *  @param  Key newKey
 *  @result PSkipListItem ret: the new item (k,inf) if (key,inf) existed in l
 *                             and NULL otherwise.
 *  @note: if several items with same key existed in l,
 *         decreaseKey 
 */
PSkipListItem PSkipListDecreaseKey( PSkipList l, Key oldKey, Key newKey)
{
  PSkipListItem oldItem, newItem, delnode;
  PSkipListItem ptr[PSKIPLISTMAXHEIGHT];
  int i;
#if DEBUG
  pprintf("call DecreaseKey( %d->%d )\n", (int)oldKey, (int)newKey);
#endif
  if (l->cmp( newKey, oldKey ) >= 0) {
     pprintf("USER ERROR: new key does not decrease old key\n");
     return NULL;
  } 
  oldItem = PSkipListLocate( l, oldKey );
  if (! oldItem)
     return NULL;
  // mark oldItem as a victim for subsequent deletion:
  //if (mpadd( &(oldItem->lock->deleteflag), 1))
  //   return NULL;  // another processor executing delete or decreaseKey
                     // was faster than me -> oldItem will no longer exist.
  newItem = PSkipListInsert( l, newKey, oldItem->inf );
  while (1) {
#if (!SILENCE)
     pprintf("...decreasingKey of %p from %d to %p with %d\n",
              oldItem, oldKey, newItem, newKey );
#endif
     delnode = PSkipListDelete( l, oldKey );
     if ( !delnode) break;
     if ( delnode == oldItem ) {
        PSkipListItemFree( delnode );
        break;
     }
     pprintf("...decreaseKey from %d to %d to be applied to another item\n",
             oldKey, newKey );
     PSkipListItemFree( delnode );
  }
  return newItem;
}


/** @method PSkipListChangeInf
 *  changes the inf of an existing item (key,inf) to a new value newInf
 *  @param  PSkipList l
 *  @param  Key key
 *  @param  Inf newInf
 *  @result PSkipListItem ret: the new item (key,newInf) if (key,inf) 
 *                             existed in l, and NULL otherwise.
 *  @note: if several items with same key exist in l, only
 *         the inf of the first of them is changed.
 */
PSkipListItem PSkipListChangeInf( PSkipList l, Key key, Inf newInf)
{
  PSkipListItem node;
  node = PSkipListLocate( l, key );
  if (!node)  return NULL;
  node->inf = newInf;
  return node;
}
