/** @package Parallel Skip List implementation in Fork95.
 *  @author  C.W. Kessler    03/1999
 *  References:
 *   [1] W. Pugh, Communications of the ACM 1990.
 *   [2] M. Weiss, Data Structures and Algorithm Analysis, 2nd ed.,
 *                 Benjamin-Cummings, 1994.
 */

#define SILENCE 0  // 0 iff calls to insert, delete etc. should be traced.

typedef void *Key;

typedef void *Inf;


/** @class PSkipListItem
 */
typedef struct skiplistnode {
   int height;                  // the height of a node (length of next array)
   struct skiplistnode **next;  // dynamically allocated array
   RWDLock lock;              // for concurrent access protection
   Key key;                   // points to a key element
   Inf inf;                   // points to a data element
} *PSkipListItem;

// constructor:
extern PSkipListItem new_PSkipListItem( int height, Key key );
// destructor:
extern void PSkipListItemFree( PSkipListItem );
// print state to stdout:
extern void PSkipListItemPrint( PSkipListItem );



/** @class PSkipList
 */
typedef struct skiplist {
#define PSKIPLISTMAXHEIGHT 20    // approx. floor of log_2 of max.#elements
   int maxHeight;  // that many list levels are allocated
   PSkipListItem head;
   int (*cmp)( Key, Key );  // the compare function for keys
   void (*print)( struct skiplist * );
   int (*empty)( struct skiplist * );
   int (*size)( struct skiplist * );
   PSkipListItem (*insert)( struct skiplist *, Key, Inf );
   PSkipListItem (*locate)( struct skiplist *, Key );
   Inf (*access)( struct skiplist *, Key );
   PSkipListItem (*delete)( struct skiplist *, Key );
   PSkipListItem (*deleteMin)( struct skiplist * );
   sync int (*deleteMins)( struct skiplist *, int, PSkipListItem * );
   PSkipListItem (*findMin)( struct skiplist * );
   PSkipListItem (*pred)( struct skiplist *, Key );
   PSkipListItem (*decreaseKey)( struct skiplist *, Key, Key );
   PSkipListItem (*changeInf)( struct skiplist *, Key, Inf );
   int Size;           // #elements currently in the list
                       // plus #pending inserts minus #pending deletes
} *PSkipList;

// constructor:
extern PSkipList new_PSkipList( int (*cmp)(Key,Key), int maxh, Key minKey );


// Left as an exercise:
//
// PSkipListItem (*deleteMax)( struct skiplist * );
// PSkipListItem (*findMax)( struct skiplist * );
// PSkipListItem (*succ)( struct skiplist *, Key );
// PSkipListItem (*locatePred)( struct skiplist * );
// PSkipListItem (*locateSucc)( struct skiplist * );
// void          (*split)( struct skiplist *, struct skiplist **, struct skiplist **);
// void          (*concat)( struct skiplist *, struct skiplist *);
