/* pqueue.c  -  Implementation of a parallel FIFO queue.
 * C.W. Kessler 3/97, 4/99, following J. Roehrig's proposal 
 */

#include <fork.h>
#include <io.h>
#include <assert.h>

#include <../util/pqueue.h>


PQItem new_PQItem( void *dat )
{
  PQItem e; 
  e = (PQItem) shmalloc( sizeof( struct pqitem ));
  e->data = dat;
  e->next = e->prev = 0;
  return e;
}


void PQItemFree( PQItem e )
{
  shfree( e );
}


extern void PQueuePrint( PQueue pq );
extern void *PQueueGet ( PQueue pq );
extern void PQueuePut( PQueue pq, void *dat );
extern int PQueueEmpty( PQueue pq );
extern int PQueueSize( PQueue pq );
extern int tdr ( int *pc, int d, int b );


sync PQueue new_PQueue( sh int k )
{
  sh int p = groupsize();
  sh PQueue ppq;
  int i;
 
  seq {
   ppq = (PQueue) shmalloc( sizeof( struct pqueue ));
   ppq->head = (PQItem *)shmalloc( k * sizeof( PQItem ));
   ppq->tail = (PQItem *)shmalloc( k * sizeof( PQItem ));
   ppq->lock = (FairLock *)shmalloc( k * sizeof( FairLock));
   ppq->listngets = (int *)shmalloc( k * sizeof( int ));
   ppq->listnputs = (int *)shmalloc( k * sizeof( int ));
   ppq->k = k;
  }
  forall (i, 0, k, p) {
      ppq->head[i] = NULL;
      ppq->tail[i] = NULL;
      ppq->lock[i] = new_FairLock();
      ppq->listnputs[i] = i;
      ppq->listngets[i] = i;
  }
  ppq->ngets = ppq->nputs = ppq->Size = 0;
  ppq->print = PQueuePrint;
  ppq->put = PQueuePut;
  ppq->get = PQueueGet;
  ppq->empty = PQueueEmpty;
  ppq->size = PQueueSize;
  return ppq;
}


void PQueuePut( PQueue pq, void *dat )
{
 int myrank, myputidx;
 fair_lock *mylock;
 PQItem e;

 syncadd( &(pq->Size), 1);
 myrank = mpadd( &(pq->nputs), 1);
 myputidx = myrank % (pq->k);
 mylock = pq->lock[myputidx];
 e = new_PQItem( dat );

 while (pq->listnputs[myputidx] != myrank )
    ;        // wait until it is my turn
 fair_lockup( mylock );
   if (! pq->tail[myputidx]) {  // list was empty:
      pq->tail[myputidx] = e; 
      pq->head[myputidx] = e;
   }
   else {
      pq->tail[myputidx]->next = e;
      pq->tail[myputidx] = e;
   }
 fair_unlock( mylock );
 syncadd( &(pq->listnputs[myputidx]), pq->k );
}


int tdr ( int *pc, int d, int b )
{
 if (mpadd( pc, 0 ) - d >= b) {  /*test*/
    if (mpadd(pc,-d) - d >= b)  /*decrement, retest*/
       return 1;           
    else {
       mpadd( pc, d );          /*failed: restore *pc*/
       return 0;
    }
 }
 else return 0;                 /*failed*/
}


void *PQueueGet ( PQueue pq )
{
  int mygetidx, myrank;
  PQItem e;
  if (tdr(&(pq->Size),1,0)) { /*test-decrement-retest; enter if nonempty*/
    myrank = mpadd( &(pq->ngets), 1);
    mygetidx = myrank % pq->k;
    while (pq->listngets[mygetidx] != myrank) ;  /*wait until it's my turn*/
    while (pq->listnputs[mygetidx] == myrank) ;  /*wait until put is ready*/
    fair_lockup( pq->lock[mygetidx] );
      e = pq->head[mygetidx];
      assert(e);
      pq->head[mygetidx] = pq->head[mygetidx]->next;
      if (pq->head[mygetidx] == NULL)
          pq->tail[mygetidx] = NULL;
    fair_unlock( pq->lock[mygetidx] );
    syncadd( &(pq->listngets[mygetidx]), pq->k );
    PQItemFree( e );  
    return e->data;
  }
  else return NULL;
}


int PQueueEmpty( PQueue pq )
{
  if (pq->nputs > pq->ngets) return 0;
  else return 1;
}


int PQueueSize( PQueue pq )
{
  return pq->nputs - pq->ngets;
}

void PQueuePrint( PQueue pq )
{
 int i;
 PQItem e;
 printf("\nPQUEUE %d with K = %d heads:", pq, pq->k);
 for (i=0; i<pq->k; i++) {
   printf("\n sublist[%d](nputs=%d,ngets=%d): ", 
            i, pq->listnputs[i], pq->listngets[i]);
   e = pq->head[i];
   while (e) {
     printf("%d ", e->data);
     e = e->next;
   }
   printf("\nnputs = %d\n", pq->nputs);
   printf("ngets = %d\n", pq->ngets);
   printf("Size = %d\n", pq->Size );
 }
}
