/*************************************************/
/*      Hash.c  -  rehashable hashtable          */
/*      parallel  implementation in fork95 v0.72 */
/*      15.3.1999 Oliver Fritzen                 */
/*      18.3.1999 Christoph W. Kessler           */
/*************************************************/

#include <fork.h>
#include <io.h>
#include "HashTable.h"


HashTable *new_HashTable (int (*hashFn)(void *), int Size) 
{
  /************************************/
  /*          Constructor             */
  /************************************/
  int i;
  HashTable *H;
  H = (HashTable *) shmalloc(sizeof(HashTable));
  H->Head = (HashTableItem **) shmalloc(Size*sizeof(HashTableItem));
  H->Lock = (fair_lock *)      shmalloc(Size*sizeof(fair_lock));
  H->Length = (int *)          shmalloc(Size*sizeof(int));
  for(i=0;i<Size;i++) {
    fair_lock_init(&(H->Lock[i]));
    H->Head[i] = NULL;
    H->Length[i]=0;
  }
  rw_lock_init(&(H->RehashLock));
  H->Size = Size;
  H->hashFn = hashFn;
  H->NumElts = 0;
  return H;
}


void HashTableKill (HashTable *H) {
  /************************************/
  /*           Destructor             */
  /************************************/

  HashTableItem *item,*item_old;
  int i;

  for(i=0; i<(H->Size);i++) { /* deleting Head Array */
    
    fair_lockup(&(H->Lock[i]));  /* ----- enter c.s. ----- */
    item = H->Head[i];
    while(item != NULL) {
      item_old = item;
      item = item->next;
      shfree(item_old);
    }
    fair_unlock(&(H->Lock[i]));  /* ----- leave c.s. ----- */
  }
  shfree(H->Head);
  
  shfree(H->Lock);                     /* deleting Lock Array */
  
  shfree(H);
}


void HashTableEnter (HashTable *H, void *entry) {
  /************************************/
  /*          New Entry               */
  /************************************/

  HashTableItem *item;
  int hashval;
  rw_lockup(&(H->RehashLock),RW_READ);
  hashval = (*(H->hashFn))(entry) % (H->Size);
  item = (HashTableItem *) shmalloc(sizeof(HashTableItem));
  item->data = entry;
  fair_lockup(&(H->Lock[hashval]));     /* enter critical section */
  item->next = H->Head[hashval];
  H->Head[hashval]=item;
  fair_unlock(&(H->Lock[hashval]));     /* leave critical section */
  rw_unlock(&(H->RehashLock),RW_READ,0);
  H->NumElts++;
  (H->Length[hashval])++;
}

void HashTableSingleEnter (HashTable *H, void *entry, 
			   int (*equals)(void *, void *)) {
  /************************************/
  /*  New Entry, allows entry only if */
  /*  it not already exists in hash   */
  /************************************/
  HashTableItem *item,*newItem;
  int hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));           /* enter critical section */
  item = H->Head[hashval];
  while((item!=NULL)&&(!(*equals)(item->data,entry))) item=item->next;
  if(!(*equals)(item->data,entry)) {
    newItem = (HashTableItem *) shmalloc(sizeof(HashTableItem));
    newItem->next = H->Head[hashval];
    newItem->data = entry;
    H->Head[hashval] = newItem;
    (H->NumElts)++;
    (H->Length[hashval])++;
  }
  fair_unlock(&(H->Lock[hashval]));           /* leave critical section */
}

void *HashTableLookup (HashTable *H, void *entry, 
		      int (*equals)(void *, void *)) {
  /************************************/
  /*          Search Entry            */
  /************************************/

  HashTableItem *item;
  void *ret;
  int hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));           /* enter critical section */
  item = H->Head[hashval];
  while(item!=NULL) {
    if((*equals)(item->data,entry)) break; 
    item=item->next;
  }
  if(item == NULL) ret = NULL;
  else ret = item->data;
  fair_unlock(&(H->Lock[hashval]));           /* leave critical section */
  return ret;
}


void *HashTableExtract (HashTable *H, void *entry,
			int (*equals)(void *, void *) ) {
  /************************************/
  /*          Search & Remove Entry   */
  /************************************/

  HashTableItem *item,*item_old;
  void *ret;
  int found=0,
      hashval;

  rw_lockup(&(H->RehashLock),RW_READ);
  hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));     /* enter critical section */
  item = H->Head[hashval];
  
  if(item==NULL) {                   /* if list is empty */
    
    fair_unlock(&(H->Lock[hashval]));   /* leave c.s. (1)         */
    rw_unlock(&(H->RehashLock),RW_READ,0);
    return NULL;
  }            
  else if ((*equals)(item->data,entry)) { /* if first-in-list matches */
    H->Head[hashval] = item->next;
    fair_unlock(&(H->Lock[hashval]));   /* leave c.s. (2)         */
    rw_unlock(&(H->RehashLock),RW_READ,0);
    ret = item->data;
    shfree(item);
    H->NumElts--;
    (H->Length[hashval])--;
    return ret;
  }
  else {
    item_old = item;
    item = item->next;
    while((!found)&&(item!=NULL)) {
      if((*equals)(item->data,entry)) found=1;
      else {
	item_old = item;
	item = item->next;
      }
    }
    if(found) {
      item_old->next = item->next;
      fair_unlock(&(H->Lock[hashval])); /* leave c.s. (3)         */
      rw_unlock(&(H->RehashLock),RW_READ,0);
      ret = item->data;
      shfree(item);
      H->NumElts--;
      (H->Length[hashval])--;
      return ret;
    }
    else {
      fair_unlock(&(H->Lock[hashval])); /* leave c.s. (4)         */
      rw_unlock(&(H->RehashLock),RW_READ,0);
      return NULL;
    }
  }
}

sync void HashTableRehash( HashTable *H, int (*newHashFn)(void *), int newSize ) 
{
  /*************************************/
  /*  Insert elements in new HashTable */
  /*  with new size and new hash func. */
  /*************************************/

  sh HashTableItem ** newHead;
  sh fair_lock *newLock;
  sh int *newLength;
  sh int ps;
  sh int p = groupsize();
  pr int i, hashval;
  pr HashTableItem *item, *old_next;
  
  seq {
     //  new_H = new_HashTable(*newHashFn,newSize);
     int i;
     newHead = (HashTableItem **) shmalloc(newSize*sizeof(HashTableItem));
     newLock = (fair_lock *)      shmalloc(newSize*sizeof(fair_lock));
     newLength = (int *)          shmalloc(newSize*sizeof(int));
  }
  farm
    for( i=$; i<newSize; i+=p) {
      fair_lock_init(&(newLock[i]));
      newHead[i] = NULL;
      newLength[i]=0;
    }
  seq {
    ps = 0;
    rw_lockup(&(H->RehashLock),RW_WRITE);
  }
  farm
    for( i=mpadd(&ps,1); i < H->Size; i=mpadd(&ps,1))
    {
      item = H->Head[i];                    /* Head of old list */
      while(item != NULL) {
	hashval = (*newHashFn)((item->data)) % newSize;
	old_next = item->next;
	syncadd(&(newLength[hashval]),1);
	fair_lockup(&(newLock[hashval]));     /* enter c.s. */
	item->next = newHead[hashval];
	newHead[hashval] = item;
	item = old_next;
	fair_unlock(&(newLock[hashval]));     /* leave c.s. */
      } 
    }
  seq {
    shfree( H->Length );
    shfree( H->Head );
    shfree( H->Lock );
    H->Length = newLength;
    H->Head = newHead;
    H->Lock = newLock;
    H->Size = newSize;
    H->hashFn = newHashFn;
    // numElts remains unchanged.
    rw_unlock(&(H->RehashLock),RW_WRITE,0);
  }
}


int HashTableSize(HashTable *H) 
{ /************************************/
  /*      return actual size of ht    */
  /************************************/
  return H->Size;
}

int HashTableNumElts(HashTable *H) 
{ /************************************/
  /*    return actual # of elements   */
  /************************************/
  return H->NumElts;
}

int HashTableListLength(HashTable *H,int index) 
{ /************************************/
  /*    return actual size of list    */
  /*    at hashposition 'index'       */
  /************************************/
  return H->Length[index % H->Size];
}


/* The following stuff is still buggy. 
 * Please do not use these routines. */

#if 0

/* The iterator is not protected.
 * It is rather to be used for debug or I/O purposes
 * in a purely sequential context. 
 */

HashTableIterator *new_HashTableIterator( HashTable *H ) 
{
  int cnt = 0;
  HashTableIterator *i = 
       (HashTableIterator *) shmalloc(sizeof(HashTableIterator));
  i->HashTable = H;
  
  // find first nonempty Head:
  while (!(H->Head[cnt]) && cnt<H->Size)  cnt++;
  if (cnt >= H->Size) {
     i->currentListHeadIdx = H->Size-1;
     i->currentElemPtr = H->Head[H->Size-1];
     return i;
  }
  // else H is nonempty:
  i->currentListHeadIdx = cnt;
  i->currentElemPtr = H->Head[cnt];
  printf("New Iterator Idx %d, curr %d\n", i->currentListHeadIdx,
          (int) *((int *)i->currentElemPtr->data) );
  return i;
}


void HashTableIteratorReset( HashTableIterator *i ) 
{
  int cnt = 0;
  HashTable *H = i->HashTable;
  while (! H->Head[cnt] && cnt<H->Size)  cnt++;
  if (cnt >= H->Size) {
     i->currentListHeadIdx = H->Size-1;
     i->currentElemPtr = H->Head[H->Size-1];
  }
  else { // H is nonempty:
     i->currentListHeadIdx = cnt;
     i->currentElemPtr = H->Head[cnt];
  }
  return;
}


HashTableItem *HashTableIteratorNextItem( HashTableIterator *i ) 
{
  HashTableItem *item = i->currentElemPtr;
  HashTable *H;
  int n;
  int cnt;
  if (!item) {
      printf("NULL!");
      return NULL;
  }

  // else item exists:
  i->currentElemPtr = item->next;
  if (i->currentElemPtr) {
     printf("Next%d:", n);
     return item;
  }
 
  // end of a list: compute next item:
  // find next nonempty head if there is one:
  H = i->HashTable;
  n = H->Size;
  cnt = i->currentListHeadIdx + 1;
  while (! H->Head[cnt]) {
      printf("[++%d]", n); 
      if (cnt >= n) break;
      cnt++; 
  }
  if (cnt >= n) {
     printf("EOL");
     i->currentListHeadIdx = n-1;
     i->currentElemPtr = H->Head[n-1];
     return item;  // next call produces NULL
  }
  else { 
     printf("newHeader(%d): ",cnt);
     i->currentListHeadIdx = cnt;
     i->currentElemPtr = H->Head[cnt];
     return item;
  }
}

#endif
