/*************************************************/
/*      Hash.c  -  rehashable hashtable          */
/*      parallel  implementation in Fork95       */
/*      15.3.1999 Oliver Fritzen                 */
/*      18.3.1999 Christoph W. Kessler           */
/*      18.6.1999 debugged & streamlined by CWK  */
/*************************************************/

#include <fork.h>
#include <io.h>
#include "HashTable.h"


HashTableItem new_HashTableItem( void *entry )
{
  HashTableItem it = (HashTableItem) shmalloc( sizeof(struct hti));
  it->data = entry;
  it->next = NULL;
  return it;
}


sync HashTable new_HashTable (
  int (*hashFn)(void *),
  int Size ) 
{
  sh HashTable H;
  sh int p = 0;
  int i;
  $ = mpadd( &p, 1 );    // assert consecutive numbering of processors
  seq {
     H = (HashTable) shmalloc( sizeof( struct ht ));
     H->Head = (HashTableItem *) shmalloc( Size * sizeof(HashTableItem) );
     H->Length = (int *) shmalloc( Size * sizeof(int) );
     H->Lock = (FairLock *) shmalloc(Size * sizeof(FairLock)); 
     H->RehashLock = new_RWLock();
  }
  farm
   forall( i, 0, Size, p ) {
     H->Head[i] = NULL;
     H->Length[i] = 0;
     H->Lock[i] = new_FairLock();
   }
  H->Size = Size;
  H->hashFn = hashFn;
  H->NumElts = 0;
  H->size = HashTableSize;
  H->numElts = HashTableNumElts;
  H->length = HashTableListLength;
  //H->print = HashTablePrint;
  H->insert = HashTableInsert;
  H->singleInsert = HashTableSingleInsert;
  H->lookup = HashTableLookup;
  H->remove = HashTableRemove;
  H->rehash = HashTableRehash;
  return H;
}


  /************************************/
  /*           Destructor             */
  /************************************/

sync void free_HashTable( sh HashTable H )
{
  sh int p = 0;
  HashTableItem item, item_old;
  int i;
  $ = mpadd( &p, 1 );  // renumber procs
  farm
   forall(i, 0, H->Size, p ) { /* deleting Head Array */
    item = H->Head[i];
    rw_lockup( H->RehashLock, RW_WRITE ); // wait for other async operations
    while(item != NULL) {
      item_old = item;
      item = item->next;
      shfree(item_old);
    }
    shfree(H->Lock[i]);
   }
  seq {
   shfree(H->Head);
   shfree(H->Lock);
   shfree(H);
  }
}


void HashTableInsert ( HashTable H, void *entry ) 
{
  HashTableItem item;
  int hashval;
  rw_lockup( H->RehashLock, RW_READ );
   hashval = (H->hashFn)(entry) % H->Size;
   item = new_HashTableItem( entry );
   fair_lockup( H->Lock[hashval] );     // enter critical section
    item->next = H->Head[hashval];
    H->Head[hashval]=item;
    H->Length[hashval] ++;
    syncadd( &(H->NumElts), 1);
   fair_unlock( H->Lock[hashval]);    // leave critical section
  rw_unlock( H->RehashLock, RW_READ, 0 );
}


  /************************************/
  /*  New Entry, allows entry only if */
  /*  it not already exists in hash   */
  /************************************/

void HashTableSingleInsert (HashTable H, void *entry, 
			   int (*equals)(void *, void *)) {
  HashTableItem item, newItem;
  int hashval = (H->hashFn)(entry) % (H->Size);
  fair_lockup( H->Lock[hashval] );           /* enter critical section */
  item = H->Head[hashval];
  while((item!=NULL)&&(!(*equals)(item->data,entry))) item=item->next;
  if(!(*equals)(item->data,entry)) {
    newItem = new_HashTableItem( entry );
    newItem->next = H->Head[hashval];
    H->Head[hashval] = newItem;
    (H->NumElts)++;
    (H->Length[hashval])++;
  }
  fair_unlock(H->Lock[hashval]);           /* leave critical section */
}


  /************************************/
  /*          Search Entry            */
  /************************************/

void *HashTableLookup (HashTable H, void *entry, 
		      int (*equals)(void *, void *)) {
  HashTableItem item;
  void *ret;
  int hashval = (H->hashFn)(entry) % (H->Size);
  fair_lockup(H->Lock[hashval]);           /* enter critical section */
  item = H->Head[hashval];
  while(item!=NULL) {
    if((*equals)(item->data,entry)) break; 
    item=item->next;
  }
  if(item == NULL) ret = NULL;
  else ret = item->data;
  fair_unlock(H->Lock[hashval]);           /* leave critical section */
  return ret;
}


  /************************************/
  /*          Search & Remove Entry   */
  /************************************/

void *HashTableRemove ( HashTable H, void *entry,
			int (*equals)(void *, void *) ) 
{
  HashTableItem item, item_old;
  void *ret;
  int found=0, hashval;

  rw_lockup( H->RehashLock, RW_READ );
  hashval = (H->hashFn)(entry) % H->Size;
  fair_lockup( H->Lock[hashval] );   // enter critical section
  item = H->Head[hashval];
  
  if(item==NULL) {                   /* if list is empty */
    fair_unlock( H->Lock[hashval]);   /* leave c.s. (1)         */
    rw_unlock( H->RehashLock, RW_READ, 0 );
    return NULL;
  }            
  else if ((equals)( item->data, entry )) { // if first-in-list matches
    H->Head[hashval] = item->next;
    fair_unlock( H->Lock[hashval]);   /* leave c.s. (2)         */
    rw_unlock( H->RehashLock, RW_READ,0);
    ret = item->data;
    shfree(item);
    H->NumElts--;
    (H->Length[hashval])--;
    return ret;
  }
  else {
    item_old = item;
    item = item->next;
    while((!found)&&(item!=NULL)) {
      if((*equals)(item->data,entry)) found=1;
      else {
	item_old = item;
	item = item->next;
      }
    }
    if(found) {
      item_old->next = item->next;
      fair_unlock( H->Lock[hashval]); /* leave c.s. (3)         */
      rw_unlock( H->RehashLock, RW_READ,0);
      ret = item->data;
      shfree(item);
      H->NumElts--;
      (H->Length[hashval])--;
      return ret;
    }
    else {
      fair_unlock( H->Lock[hashval] ); /* leave c.s. (4)         */
      rw_unlock( H->RehashLock, RW_READ,0);
      return NULL;
    }
  }
}


sync void HashTableRehash(
  sh HashTable H,
  sh int (*newHashFn)(void *),
  sh int newSize ) 
{
  sh HashTableItem *newHead;
  sh FairLock *newLock;
  sh int *newLength;
  sh int ps;
  sh int p = 0;
  int i, hashval;
  HashTableItem item, old_next;
  
  seq pprintf("Rehashing %d->%d\n", H->Size, newSize );
  seq {
     newHead = (HashTableItem *) shmalloc( newSize * sizeof(HashTableItem) );
     newLock = (FairLock *) shmalloc( newSize * sizeof(FairLock) );
     newLength = (int *) shmalloc( newSize * sizeof(int) );
  }
  $ = mpadd( &p, 1 );    // assert consecutive numbering of processors
  farm
    for( i=$; i<newSize; i+=p) {
      newLock[i] = new_FairLock();
      newHead[i] = NULL;
      newLength[i] = 0;
    }
  seq 
    rw_lockup( H->RehashLock, RW_WRITE );
  farm
    FORALL( i, &ps, 0, H->Size, 1 ) {  // redistribute list i:
      item = H->Head[i];
      while ( item ) {
	hashval = (newHashFn(item->data)) % newSize;
	old_next = item->next;
	syncadd( &(newLength[hashval]), 1 );  // is atomic
	fair_lockup( newLock[hashval] );   // enter crit. section
	 item->next = newHead[hashval];
	 newHead[hashval] = item;
	fair_unlock( newLock[hashval] );     // leave crit. section
	item = old_next;
      } 
    }
  seq {
    shfree( H->Length );
    shfree( H->Head );
    shfree( H->Lock );
    H->Length = newLength;
    H->Head = newHead;
    H->Lock = newLock;
    H->Size = newSize;
    H->hashFn = newHashFn;
    // NumElts remains the same.
    rw_unlock( H->RehashLock, RW_WRITE, 0 );
  }
}


int HashTableSize(HashTable H) { return H->Size; }

int HashTableNumElts(HashTable H) { return H->NumElts; }

int HashTableListLength(HashTable H,int index) 
{
  return H->Length[index % H->Size];
}
