/* math.c   mathematical functions for the Fork95 compiler: */

#include <math.h>
#include <assert.h>   /* only for test purposes! throw out if possible */
#include <io.h>

#define DEBUG_MATH 0   /* 1=debug output, 0=quiet */

int EDOM = 0;  /* Flag for Domain Error */
int ERANGE = 0; /* Flag for Range Error */

sh const float PI = 3.1415926536;    /*for more digits, see Num. Recipes p.923*/
sh const float PI2 = 6.2831853072;
sh const float PId2 = 1.5707963;

extern void fperror( const char *);

void fperror( const char *funame )
{
  if (EDOM) {
    pprintf("Floatingpoint domain error in function %s\n", funame);
    exit(1);
  }
  if (ERANGE) {
    pprintf("Floatingpoint range error in function %s\n", funame);
    exit(1);
  }
  pprintf("Unspecified floatingpoint error in function %s\n", funame);
  exit(1);
}


extern float SinLoopX( float );

float sin( float x )   /* compute sine function */
{
  register float ret, left;
  register float sign = 1.0;
  
  /* normalize x to positive: */
  if (x<0) {
     ret = -x;
     sign = -sign;     /* sin(-x) = -sin(x) */
  }
  else ret=x;
  /* normalize ret to [0,2*PI[: */
  if (ret>PI2) {
     left = floor(ret/PI2);
     ret = ret - (float)left * PI2;
  }
  /* now fold to [0,PI]: */
  if (ret>PI) { ret = ret - PI; sign = -sign; }  /* sin(x) = -sin(x-PI) */
  /* now fold to [0,PId2]: */
  if (ret>PId2)  ret = PI - ret;               /* sin(x) = sin(PI-x) */
  /* Algorithm from Book by A. Engel: Elementarmathematik vom algorithmischen
                                      Standpunkt, Klett Verlag, ca. 1980 */
  /*printf("ret=%f\n",ret);*/
  return sign * SinLoopX(ret);
}
  
float cos( float x )   /* compute cosine function */
{
  return sin( x + PId2 );
}

int itento( int n ) /* generates pos. integer powers of ten */
{
  register int ret = 1;
  for (; n>0; n--)  ret *= 10;
  return ret;
}


float tento( int n )     /* generates integer powers of ten */
{
  register float ret;
  register float f;
  if (n==0) return 1.0;
  f = (n>0)? 10.0 : 0.1;
  if (n<0) n = -n;
  ret = f;
  while (--n) ret = ret * f;
  return ret;
} 


/* Mehrzweck-Algorithmus fuer ln(), asin(), acos(), atan()
 * nach dem Buch von A. Engel, s.o. */

#if 0
   schlechte Konvergenz, grosse Fehler
   fuer kleine und grosse Argumente

float _universal_iter( float a, float b, float C)
{
#define _UNIV_ITER_MAX 100    // Notbremse
  register float A = 0.5 * (a + b);
  register float B = sqrt( a * b );
  register int i=0;
  A = 0.5 * (A + B);
  B = sqrt( A * B );
  do {
    A = 0.5 * (A + B);
    B = sqrt( A * B );
    if (++i > _UNIV_ITER_MAX) break;  // emergency break
  } 
  while ( fabs( A-B ) / (fabs(A)+fabs(B)) > 0.00003 );
  return 3.0 * C / ( B+B+A);
}


float ln( float x )
{
 if (x<=0.0) { EDOM = 1; fperror("ln"); }
 return _universal_iter( 0.5 + 0.5 * x, sqrt(x), x - 1.0 );
}

float asin( float x )
{
 if (x>1.0 || x< -1.0) { EDOM = 1; fperror("asin"); } 
 return _universal_iter( sqrt(1.0 - x*x), 1.0, x );
}

float acos( float x )
{
 if (x>1.0 || x<-1.0) { EDOM = 1; fperror("acos"); }
 if (x==-1.0)  return PI; 
 return _universal_iter( x, 1.0, sqrt(1.0 - x*x) );
}

float atan( float x )
{
 return _universal_iter( 1.0, sqrt(1.0+x*x), x );
}
#endif


// This approximation method produces
// much better results. See book by A.Engel, p.73ff.

float _univ_iter_2( float s, float c )
{
  register float S = s;
  register float C = c;
  register float S1;
  register int i;
#define _UNIV_ITER2_MAX 100
  do {
    C = sqrt((1.0+C)*0.5);
    S1 = S;
    if (++i > _UNIV_ITER2_MAX) break;  // emergency break
    S = S / C;
  } 
  while ( fabs( 1.0 - C ) > 0.00001 );
  return (4.0 * S - S1 ) * 0.333333333;
}
 
float ln( float x )
{
 if (x<=0.0) { /*EDOM = 1;*/ fperror("ln"); }
 return _univ_iter_2( 0.5*(x - 1.0/x), 0.5*(x + 1.0/x) );
}                                                           

float asin( float x )
{
 if (x>1.0 || x< -1.0) { EDOM = 1; fperror("asin"); } 
 return _univ_iter_2( x, sqrt(1.0 - x*x) );
}

// ! Arccos nahe bei -1 wird ungenau durch Ausloeschung.
float acos( float x )
{
 if (x>1.0 || x<-1.0) { EDOM = 1; fperror("acos"); }
 if (x<-0.999)  return PI; 
 return _univ_iter_2( sqrt(1.0 - x*x), x );
}

float atan( float x )
{
 return _univ_iter_2( x / sqrt(1.0+x*x), 1.0 / sqrt(1.0+x*x) );
}

float acot( float x )
{
 return _univ_iter_2( 1.0 / sqrt(1.0+x*x), x / sqrt(1.0+x*x) );
}

float arsinh( float x )
{
 return _univ_iter_2( x, sqrt(1.0 + x*x) );
}

float arcosh( float x )
{
 if (x<1.0) { EDOM = 1; fperror("arcosh"); } 
 return _univ_iter_2( sqrt(-1.0 + x*x), x );
}

float artanh( float x )
{
 if (x<-1.0 || x>1.0) { EDOM = 1; fperror("artanh"); } 
 return _univ_iter_2( x / sqrt(1.0-x*x), 1.0 / sqrt(1.0-x*x) );
}

float arcoth( float x )
{
 if (x<-1.0 || x>1.0) { EDOM = 1; fperror("arcoth"); } 
 return _univ_iter_2( 1.0 / sqrt(1.0-x*x), x / sqrt(1.0-x*x) );
}



float log10 ( float x )
{
#define _log_ten_e 0.43429448;
 return ln(x) * _log_ten_e;
}


float blubberipow( float x, int n )
{
 float po = x;
 if (n==0) return 1.0;
 else 
   for (;n>1; n--)  po *= x;
 return po;
}


float fabs( float x )
{
  if (x<0.0) return -x;
  else       return x;
}


float pow( float x, float y )
{
 float iy;

 /* domain error if x is zero and y is negative or zero: */
 if (fabs(x)<0.000001 && y<=(float)0) { EDOM = 1; fperror("pow"); }

 /* x^-a = 1./x^a: */
 if (y < 0.0) return 1.0 / pow( x, -y );

 // hence y is nonnegative:

 if (fabs(x)<0.000001) { /*  x is (near)zero: */
#if DEBUG_MATH
    printf("pow(): x (near-)zero.\n");
#endif
    /* as y is positive, then the result is zero. */
    return 0.0;
 }
 // x is nonzero:
 /* if y is zero, then the result is 1.0. */
 if (fabs(y)<0.000001) return 1.0;

 if (x >= 0.0) {   /* x is positive: */
    /* general case: */
#if DEBUG_MATH
    printf("pow(): general case.\n");
#endif
    return exp( y * ln( x ) );
 }
 // x is negative: domain error unless y is integral
#if DEBUG_MATH
 printf("pow(): x negative.\n");
#endif
 iy = floor(y+0.000002);
 if (fabs(y - iy) < 0.000001)
    /* x is negative and y is an integer: */
    return blubberipow(x, (int)(y));
 /* otherwise: domain error */
 EDOM = 1;
 fperror("pow");
 return -47.11;  // some crazy value - statement not reached.
}


/* nach A. Engel: */
 
float _exp_2( float x )
{
  register float s = x * 0.00001693508;
  register int n = 10;
  for ( ; n>0; n-- ) {
    s = s*(3.0 + 4.0*s*s);
  } 
  return s+sqrt(1+s*s);
}

float _exp_1( float x )      // Reihensumme nach Bronstein. 
                          // empirisch schneller fuer x<0.4
{
 //vormals: return pow( 2.7182818, x );
 float ex = 1.0;
 float xhochi = 1.0;
 float faci = 1.0;
 float bruch;
#define _EXP_MAXITER 100    // Notbremse
 int i;
 if (x<0.0)  return 1.0 / exp( - x );
 for (i=1; i<_EXP_MAXITER; i++) {
    xhochi *= x;
    faci *= (float) i;
    bruch = xhochi / faci;
    ex += bruch;
#if DEBUG_MATH
    printf("i=%d: exp(%f)=%f\n", i, x, ex );
#endif
    if (bruch < 0.000001) break;
 }
 return ex;
}

float exp( float x )
{
  if ( x < -15.0 ) return 0.0;  // underflow
  if ( x >= 20.0 ) { ERANGE = 1; fperror("exp"); }  // overflow
  if (fabs(x) < 0.4) return _exp_1( x );
  else               return _exp_2( x );
}

float sqrt( float x )   // produces 4 or 5 correct decimal digits
                        // for all sizes of x.
{
  register float s, s_old;
  register int i;
#define _NEWTON_SQRT_MAXINT 32
  if (x < 0.0) { EDOM = 1; fperror("sqrt"); }
  if (x == 0.0)  return 0.0;

  // Newton-Iteration:
  s = 0.5 * (x + 1.0);   // Iter. 1
  s = 0.5 * (s + x / s); // Iter. 2
  s = 0.5 * (s + x / s); // Iter. 3
  s = 0.5 * (s + x / s); // Iter. 4
  s = 0.5 * (s + x / s); // Iter. 5
  s_old = s;
  for (i=0; i<_NEWTON_SQRT_MAXINT; i++) {
     s = 0.5 * (s + x / s); // Iter. 6+i
     if (fabs(s - s_old) / (s + s_old) < 0.000001) break;
     s_old = s;
  }
  return s;
}
