/* math.c   mathematical functions for the Fork95 compiler: */

#include <math.h>
#include <assert.h>   /* only for test purposes! throw out if possible */

pr int EDOM = 0;  /* Flag for Domain Error */
pr int ERANGE = 0; /* Flag for Range Error */

#define ABS(a) ((a)>0? (a):(-a))

sh const float PI = 3.1415926536;    /*for more digits, see Num. Recipes p.923*/
sh const float PI2 = 6.2831853072;
sh const float PId2 = 1.5707963;

const float _one = 1.0;

extern void fperror( const char *);

void fperror( const char *funame )
{
  if (EDOM) {
    pprintf("Floatingpoint domain error in function %s\n", funame);
    exit(1);
  }
  if (ERANGE) {
    pprintf("Floatingpoint range error in function %s\n", funame);
    exit(1);
  }
  pprintf("Unspecified floatingpoint error in function %s\n", funame);
  exit(1);
}


extern float SinLoopX( float );

float sin( float x )   /* compute sine function */
{
  register float ret, left;
  register float sign = _one;
  
  /* normalize x to positive: */
  if (x<0) {
     ret = -x;
     sign = -sign;     /* sin(-x) = -sin(x) */
  }
  else ret=x;
  /* normalize ret to [0,2*PI[: */
  if (ret>PI2) {
     left = floor(ret/PI2);
     ret = ret - (float)left * PI2;
  }
  /* now fold to [0,PI]: */
  if (ret>PI) { ret = ret - PI; sign = -sign; }  /* sin(x) = -sin(x-PI) */
  /* now fold to [0,PId2]: */
  if (ret>PId2)  ret = PI - ret;               /* sin(x) = sin(PI-x) */
  /* Algorithm from Book by A. Engel: Elementarmathematik vom algorithmischen
                                      Standpunkt, Klett Verlag, ca. 1980 */
  /*printf("ret=%f\n",ret);*/
  return sign * SinLoopX(ret);
}
  
float cos( float x )   /* compute cosine function */
{
  return sin( x + PId2 );
}

int itento( int n ) /* generates pos. integer powers of ten */
{
  register int ret = 1;
  for (; n>0; n--)  ret *= 10;
  return ret;
}


float tento( int n )     /* generates integer powers of ten */
{
  register float ret;
  register float f;
  if (n==0) return 1.0;
  f = (n>0)? 10.0 : 0.1;
  if (n<0) n = -n;
  ret = f;
  while (--n) ret = ret * f;
  return ret;
} 


/* Mehrzweck-Algorithmus fuer ln(), asin(), acos(), atan()
 * nach dem Buch von A. Engel, s.o. */

extern float _ahalf, _three;
float _accuracy = 0.000003;

#define universal_iter(A,B,C) \
{ \
  do { \
    A = (A + B) * _ahalf; \
    B = sqrt( A * B ); \
  } while ( ABS( A-B ) > _accuracy); \
  C = _three * C / ( B+B + A); \
}

float ln( float x )
{
 register float A = (_one + x) * _ahalf;
 register float B = sqrt(x);
 register float C = x - _one;
 
 if (x<(float)0)
   { EDOM = 1; fperror("ln"); }
 
 universal_iter( A, B, C )
 return C;
}

float asin( float x )
{
 register float A = sqrt(_one - x*x);
 register float B = _one;
 register float C = x;
 
 if (x>_one || x< -_one)
   { EDOM = 1; fperror("asin"); }
 
 universal_iter( A, B, C )
 return C;
}

float acos( float x )
{
 register float A = x;
 register float B = _one;
 register float C = sqrt(_one - x*x);
 
 if (x>_one || x< -_one)
   { EDOM = 1; fperror("acos"); }
 if (x==-_one)  return PI; 

 universal_iter( A, B, C )
 return C;
}

float atan( float x )
{
 register float A = _one;
 register float B = sqrt(_one + x*x);
 register float C = x;
 
 universal_iter( A, B, C )
 return C;
}

const float _log_ten_e = 0.43429448;

float log10 ( float x )
{
 return ln(x) * _log_ten_e;
}
