/* Parallel plotting of Mandelbrot fractal sets
 * C.W. Kessler 3/1997
 * Reference: Becker, Doerfler: Dynamische Systeme und Fraktale.
 *            4. Auflage, Vieweg (Wiesbaden), 1992
 */
#include <fork.h>
#include <assert.h>
#include <io.h>
#include <stdlib.h>
#include <math.h>
#include <graphic.h>

#define XPIXELS 900
#define YPIXELS 900 
#define MAXITER 128

sh float right = 2.0, left = -0.6, 
         top = 1.0, bottom = -1.0;     /* the window in the complex plane */


async int is_in_Mandelbrot( float Cre, float Cim )
{
 /* the points (Cre,Cim) in the complex plane for which the
  * computation diverges, i.e. (0,0) is not in the Julia set of (Cre,Cim).*/
 int iter;
 float x = 0.0;
 float y = 0.0;
 float xto2 = 0.0;
 float yto2 = 0.0;
 float dist2;
 for (iter=0; iter<=MAXITER; iter++) {
    y = x * y;
    y = y + y - Cim;
    x = xto2 - yto2 - Cre;
    xto2 = x * x;  yto2 = y * y;
    dist2 = xto2 + yto2;
    /* test on convergence: */
    if ((int)dist2 >= 100) break;
 }
 return iter;
 /* alternative: return 1 for non-converging points, 0 otherwise: */
 /* alternative: return is_in_zc(x,y) */
}

sh int NextXY = 0;

main()
{
 start {
   sh int p = groupsize();
   pr int X, Y;       /* screen coordinates */
   pr int XY;         /* loop variable ranges over screen */
   pr float x, y;     /* world coordinates */
   pr int color;
   sh float dx = (right - left) / (float)XPIXELS;
   sh float dy = (top - bottom) / (float)YPIXELS;
   seq printf("dx=%f dy=%f\n", dx, dy);
   init_pict( XPIXELS, YPIXELS );
   /* clear_pixels( 0 ); not required if all points are computed */
   farm
    for ( XY = mpadd(&NextXY,1);  XY < XPIXELS*YPIXELS;
          XY = mpadd(&NextXY,1)) {
      Y = XY / XPIXELS;
      X = XY % XPIXELS;
      y = bottom + (float)Y * dy;
      x = left + (float)X * dx;
      color = is_in_Mandelbrot( x, y );
      if (color < MAXITER) {
         color = color % 20;
         color = 0xd3f7 << color;
      }
      else color = 0;
      set_pixel( X, Y, color );
    }
   seq printf("\nDone! Writing pixmap file ...\n");
   write_pixmap( "MANDEL" );
 }
 barrier; exit(0);
}
