/* math.c   mathematical functions for the Fork95 compiler: */

#include <math.h>
#include <assert.h>   /* only for test purposes! throw out if possible */

const float PI = 3.1415926536;    /*for more digits, see Num. Recipes p.923*/
const float PI2 = 6.2831853072;
const float PId2 = 1.5707963;
const float hundredbyPI = 31.830989;

#ifdef FORKHEADER
 sh float _sintab[] = {  /* sine table for [0,PI] with 50 entries */
#include <sintab.h>
 };    /*shared array: save space and startup code time */
#else
 float _sintab[] = {
#include <sintab.h>
 };
#endif

const float _one = 1.0;

float sin( float x )   /* compute sine function */
{
  int left;
  register float ret, exact;
  register float sign = _one;
  
  /* normalize x to positive: */
  if (x<0) {
     ret = -x;
     sign = -sign;     /* sin(-x) = -sin(x) */
  }
  else ret=x;
  /* normalize ret to [0,2*PI[: */
  if (ret>PI2) {
     left = floor(ret/PI2);
     ret = ret - (float)left * PI2;
  }
  /* now fold to [0,PI]: */
  if (ret>PI) { ret = ret - PI; sign = -sign; }  /* sin(x) = -sin(x-PI) */
  /* now fold to [0,PId2]: */
  if (ret>PId2)  ret = PI - ret;               /* sin(x) = sin(PI-x) */
  /* now determine sintab index left of x: */
  exact = ret*hundredbyPI;
  left = floor(exact);
#if 0
  printf("x=%5.2lf  ret=%5.2lf\n", x, ret);
  printf("left=%d  \n", left);
  assert(left>=0);
  assert(exact<=50);
  assert(left<=exact);
  assert(exact<left+1);
#endif
  /* return linear interpolation of adjacent table entries */
  return sign * (_sintab[left]*(left+1-exact)+_sintab[left+1]*(exact-left));
}
  
float cos( float x )   /* compute cosine function */
{
  return sin( x + PId2 );
}

int itento( int n ) /* generates pos. integer powers of ten */
{
  register int ret = 1;
  for (; n>0; n--)  ret *= 10;
  return ret;
}


float tento( int n )     /* generates integer powers of ten */
{
  register float ret;
  register float f;
  if (n==0) return 1.0;
  f = (n>0)? 10.0 : 0.1;
  if (n<0) n = -n;
  ret = f;
  while (--n) ret = ret * f;
  return ret;
} 

