/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Dave Kohr (drk@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SummaryInfo.C Information about fields in summary file records.
 *
 */

#include <assert.h>
#include <math.h>
#include <stream.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>


// #include's from SDDF library.

#include "Attributes.h"
#include "CString.h"
#include "RecordDictionaryPIterator.h"
#include "StructureDescriptorIterator.h"


// #include's from SDDFStatistics.

#include "SummaryInfo.h"
#include "SDDFStats_Misc.h"


/**********************************************************************
Error-checked insertion of key/value pair into an Attributes object.
**********************************************************************/

static void
checkedInsert(Attributes &attributes, const char *key, const char *value)
{
	if (attributes.insert(key, value) != SUCCESS_)
		fatalError(CString("Insertion of key \"") + key +
"\" with value \"" + value + "\" into attributes object failed");
}


/**********************************************************************
Error-checked "set" of the value of a field in a RecordDossier.  The
overloaded versions for values which are character strings and
1-dimensional Array's make dealing with those datatypes easier.
**********************************************************************/

static void
checkedSetValue(RecordDossier &inDossier, const char *fieldName,
		const Value& fieldValue)
{
	if (inDossier.setValue(fieldName, fieldValue) != SUCCESS_)
		fatalError(CString("RecordDossier::setValue failed "
"for field \"") + fieldName + "\"");
}

static void
checkedSetValue(RecordDossier &inDossier, int fieldID,
		const Value& fieldValue)
{
	if (inDossier.setValue(fieldID, fieldValue) != SUCCESS_)
		fatalError(CString("RecordDossier::setValue failed "
"for field #") + int2CString(fieldID));
}

static void
checkedSetValue(RecordDossier &inDossier, const char *fieldName,
		const char *stringValue)
{
	Array *array = inDossier.getArrayP(fieldName);
	assert(array);
	int stringLength = strlen(stringValue) + 1;

	array->setDimSizes(&stringLength);
	assert(array->setCellString(stringValue));
}

static void
checkedSetValue(RecordDossier &inDossier, const char *fieldName,
		const Array &inArray)
{
	Array *array = inDossier.getArrayP(fieldName);
	assert(array);

	*array = inArray;
}


/**********************************************************************
Definition of TagHistogram class (only used internally by PerFileSummary
class).
**********************************************************************/

class TagHistogram {
	struct Node {
		int tag, count;
		Node *next;
		Node(int _tag) : tag(_tag), count(1), next(NULL) {}
	};
	Node *nodes;
public:
	TagHistogram() : nodes(NULL) {}
	void addTag(int tag);
	int dominantTag();
	void clear();
	~TagHistogram() { if (nodes) clear(); }
};

void TagHistogram::addTag(int tag)
{
	Node *np = nodes;

	if (!np) {
		nodes = new Node(tag);
		return;
	}
	while (np->tag != tag)
		if (!np->next) {
			np->next = new Node(tag);
			return;
		} else
			np = np->next;
	np->count++;
}

int TagHistogram::dominantTag()
{
	Node *np = nodes;
	int domTag, domCount = 0;

	if (!np)
		return -1;

	while (np) {
		if (np->count > domCount) {
			domTag = np->tag;
			domCount = np->count;
		}
		np = np->next;
	}

	return domTag;
}

void TagHistogram::clear()
{
	Node *np1, *np2 = nodes;

	nodes = NULL;

	while (np2) {
		np1 = np2->next;
		delete np2;
		np2 = np1;
	}
}


/**********************************************************************
Methods and data for PerFileSummary class.
**********************************************************************/

void PerFileSummary::addTag(int tag)
{
	int newBin = int(double(curRecordNum) / recordsPerBin); 
	assert(newBin >= 0 && newBin < num_bins);
	curRecordNum++;

	if (newBin > curBin) {
		int i, domTag = histogram->dominantTag();

		for (i = curBin; i < newBin; i++)
			dominantType[i] = domTag;

		histogram->clear();
		curBin = newBin;
	}

	histogram->addTag(tag);
}


const char *PerFileSummary::structureName()
{
	return "File Summary";
}

const char *PerFileSummary::structureDescription()
{
	return "Summary data for entire file";
}


const PerFileSummary::FieldDesc
PerFileSummary::fieldDesc[] = {
	{ "original SDDF file name",
	  "name", CHARACTER, 1 },
	{ "file size in bytes", 
	  "size", INTEGER, 0 },
	{ "total number of Descriptor packets", 
	  "descriptor count", INTEGER, 0 },
	{ "total number of Data packets", 
	  "data count", INTEGER, 0 },
	{ "total number of Attribute packets", 
	  "attribute count", INTEGER, 0 },
	{ "total number of Command packets", 
	  "command count", INTEGER, 0 },
	{ "number of Descriptor packets with duplicate tags", 
	  "duplicate descriptor count", INTEGER, 0 },
	{ "tag of dominant record type in each fraction of file", 
	  "dominant type", INTEGER, 1 }
};


int
PerFileSummary::tag()
{
	return 100000;		// A hack: should compute a value
				// guaranteed not to conflict with other
				// tags in the output file.
}


void
PerFileSummary::readDescriptorFirstPass(const StructureDescriptor &inDesc,
					int inputTag)
{
	inRecDict.insert(inputTag, inDesc);
	PerRecordSummary::defineSummaryRecord(inDesc, inputTag, outRecDict,
					      *outWriter);
	descriptorCount++;
}
	
void PerFileSummary::endFirstPass()
{
	curBin = 0;
	curRecordNum = 0;
	recordsPerBin = dataCount / (double) num_bins;

	PerRecordSummary::endFirstPass(outRecDict);
}


void PerFileSummary::endSecondPass()
{
	int i, domTag = histogram->dominantTag();

	for (i = curBin; i < num_bins; i++)
		dominantType[i] = domTag;

	histogram->clear();

	PerRecordSummary::endSecondPass(outRecDict);
}


void PerFileSummary::writeSummaryFile(const char *)
{
	int i;

				// Populate and write out record of
				// per-file summary info.
	checkedSetValue(*perFileRecord, fieldDesc[NAME].name, inFileName);
	checkedSetValue(*perFileRecord, fieldDesc[SIZE].name, size);
	checkedSetValue(*perFileRecord, fieldDesc[DESCRIPTOR_COUNT].name,
			descriptorCount);
	checkedSetValue(*perFileRecord, fieldDesc[DATA_COUNT].name,
			dataCount);
	checkedSetValue(*perFileRecord, fieldDesc[ATTRIBUTE_COUNT].name,
			attributeCount);
	checkedSetValue(*perFileRecord, fieldDesc[COMMAND_COUNT].name,
			commandCount);
	checkedSetValue(*perFileRecord,
			fieldDesc[DUPLICATE_DESCRIPTOR_COUNT].name,
			dupDescCount);

	Array histogram(fieldDesc[DOMINANT_TYPE].dataType, 1);

	histogram.setDimSizes(&num_bins);
	for (i = 0; i < num_bins; i++)
		assert(histogram.setCellValue(dominantType[i], i));

	checkedSetValue(*perFileRecord, fieldDesc[DOMINANT_TYPE].name,
			histogram);

	outWriter->putData(*perFileRecord);

				// Write per-record summary records.
	RecordDictionaryPIterator iter(outRecDict);
	RecordDossier *outDossier;

	while ((outDossier = iter.next()) != NULL)
		outWriter->putData(*outDossier);
}


PerFileSummary::PerFileSummary(const char *_inFileName,
			       const char *_outFileName,
			       int _num_bins) :
//			       int _num_bins = 10) :
	histogram(new TagHistogram),
	outFile(new OutputFileStreamPipe(_outFileName)),
	inFileName(_inFileName),
	descriptorCount(0), dataCount(0), attributeCount(0),
	commandCount(0), dupDescCount(0), num_bins(_num_bins)
{
	assert(histogram);

	int i;

	dominantType = new int[num_bins];
	assert(dominantType);
	for (i = 0; i < num_bins; i++)
		dominantType[i] = -1;

	assert(outFile);
	if (!outFile->successfulOpen())
		fatalError(CString("Could not open output file \"") +
_outFileName + "\" in PerFileSummary::PerFileSummary()");

	outWriter = new AsciiPipeWriter(outFile);

	if (!outWriter->successfulConnection())
		fatalError(CString("Could not attach pipe "
"to output file\"") + _outFileName +
"\" in PerFileSummary::PerFileSummary()");

	struct stat stat_buf;

				// Cast to char * is necessary for the
				// Paragon's <sys/stat.h>.
	if (stat((char *) _inFileName, &stat_buf))
		fatalError(CString("stat(2) failed on input file \"") +
inFileName + "\" in PerFileSummary::PerFileSummary()");

	size = (int) stat_buf.st_size;

/*
Writing to output file is currently done here instead of in
PerFileSummary::writeSummaryFile() because descriptors for per-record
summaries must be written as descriptors are encountered in the input file.
*/

				// Write attributes for the output file.
	Attributes attributes;

	checkedInsert(attributes, "generated by", SDDFStatistics_VERSION);
	outWriter->putAttributes(attributes);

				// Build up descriptor for per-file summary
				// info. record; write it to output file.
	attributes.clearEntries();
	checkedInsert(attributes, "description", structureDescription());

	StructureDescriptor pfsSD(structureName(), attributes);

	for (i = 0; i < numFields; i++) {
		attributes.clearEntries();
		checkedInsert(attributes, "description",
			      fieldDesc[i].description);

		FieldDescriptor f(fieldDesc[i].name, attributes,
				  fieldDesc[i].dataType,
				  fieldDesc[i].dimensions);

		assert(pfsSD.insert(f));
	}
	outWriter->putDescriptor(pfsSD, tag());

				// Create per-file summary record.
	perFileRecord = new RecordDossier(tag(), pfsSD);
}


PerFileSummary::~PerFileSummary()
{
	delete histogram;
	delete outWriter;
	delete outFile;
	delete perFileRecord;
	delete [] dominantType;
}


/**********************************************************************
Methods and data for PerRecordSummary class.
**********************************************************************/

const PerRecordSummary::FieldDesc
PerRecordSummary::fieldDesc[] = {
	{ "number of records of this type in file",
	  "__record count__", INTEGER, 0 },
	{ "record type-tag, for denoting dominant types",
	  "__record tag__", INTEGER, 0 },
	{ "minimum value in entire file",
	  "__min__", UNDEFINED, 0 },
	{ "maximum value in entire file",
	  "__max__", UNDEFINED, 0 },
	{ "count of individual scalar values for this field",
	  "__values count__", INTEGER, 0 },
	{ "average value in entire file",
	  "__mean__", DOUBLE, 0 },
	{ "standard deviation of all values in entire file",
	  "__stddev__", DOUBLE, 0 },
	{ "low cutoff value for histogram",
	  "__underflow__", UNDEFINED, 0 },
	{ "high cutoff value for histogram",
	  "__overflow__", UNDEFINED, 0 },
	{ "number of bins in histogram",
	  "__bin count__", INTEGER, 0 },
	{ "count of elements in each bin of histogram",
	  "__histogram__", INTEGER, 1 },
	{ "minimum size of each dimension",
	  "__min dimension__", INTEGER, 1 },
	{ "maximum size of each dimension",
	  "__max dimension__", INTEGER, 1 },
	{ "average size of each dimension",
	  "__mean dimension__", DOUBLE, 1 },
	{ "standard deviation of size of each dimension",
	  "__stddev dimension__", DOUBLE, 1 },
};


CString
PerRecordSummary::fieldName(const CString &inFieldName,
			    PerRecordSummary::VarFields field)
{
	return inFieldName + "::" + fieldDesc[field].name;
}

int
PerRecordSummary::tag(int inputTag)
{
	return inputTag;
}

int
PerRecordSummary::numHistogramBins()
{
	return 10;
}


void
PerRecordSummary::defineSummaryRecord(const StructureDescriptor &inDesc,
				      int inputTag,
				      RecordDictionary &outRecDict,
				      PipeWriter &outWriter)
{
	const CString inRecName(inDesc.getName());
	CString outRecName;
	CString recDescription;

				// Can't use constructor to assign these,
				// due to creation of temporaries; see ARM
				// sec. 12.2.
	outRecName = inRecName + "::__summary__";
	recDescription = "Summary data for \"" + inRecName + "\" record";

	Attributes attributes;
	int i, numInputFields = 0;
	int outRecTag = tag(inputTag);

	checkedInsert(attributes, "description", recDescription);

	StructureDescriptor outSD(outRecName, attributes);

				// Define the fixed fields.
	for (i = firstFixedField; i <= lastFixedField; i++) {
		attributes.clearEntries();
		checkedInsert(attributes, "description",
			      fieldDesc[i].description);

		FieldDescriptor field(fieldDesc[i].name, attributes,
				      fieldDesc[i].dataType,
				      fieldDesc[i].dimensions);

		assert(outSD.insert(field));
	}

				// Define the variable fields.
	StructureDescriptorIterator inFieldIter(inDesc);
	FieldDescriptor inField;

				// Reduce output file size by inserting
				// "description" attribute for only the
				// first set of "variable" fields.
	int isFirstInputField = 1;

	while ((inField = inFieldIter.next()).notNull()) {
		for (i = firstVarField; i <= lastVarField; i++) {
			CString name = fieldName(inField.getName(),
						 (VarFields) i);
			MachineDataType type = fieldDesc[i].dataType;

			if (type == UNDEFINED) {
				type = inField.getType();
			}

			int dimensions = fieldDesc[i].dimensions;

			attributes.clearEntries();
			if (isFirstInputField)
				checkedInsert(attributes, "description",
					      fieldDesc[i].description);

			FieldDescriptor outField(name, attributes, type,
						 dimensions);

			assert(outSD.insert(outField));
		}
		isFirstInputField = 0;
		numInputFields++;
	}

	outWriter.putDescriptor(outSD, outRecTag);

				// Get RecordDossier for summary record.
	if (!outRecDict.insert(outRecTag, outSD))
		fatalError(CString("RecordDictionary::insert() failed for "
" input record with tag ") + int2CString(inputTag));

	RecordDossier &outRD = outRecDict.fetch(outRecTag);

				// Initialize every field of outRD.
	int outFieldID = 0;

	for (i = firstFixedField; i <= lastFixedField; i++) {
		checkedSetValue(outRD, outFieldID,
				initialValue((FixedFields) i, outRecTag));
		outFieldID++;
	}

	for (inField = inFieldIter.first(); inField.notNull();
	     inField = inFieldIter.next()) {
		for (i = firstVarField; i <= lastVarField; i++) {
			checkedSetValue(outRD, outFieldID,
					initialValue((VarFields) i,
						     inField));
			outFieldID++;
		}
	}
}


Value
PerRecordSummary::initialValue(PerRecordSummary::FixedFields outputField,
			       int inputTag)
{
	switch (outputField) {
	case RECORD_COUNT:
		break;
	case RECORD_TAG:
		return tag(inputTag);
	default:
		assert(0);
	}
	return 0;
}

Value
PerRecordSummary::initialValue(PerRecordSummary::VarFields outputField,
			       const FieldDescriptor &inputField)
{
	switch (outputField) {
	case MIN:
	case MAX:
	case UNDER_FLOW:
	case OVER_FLOW:
					// cast 0 to type of inputField.
					// use char(0) since Value's cannot
					// be down-cast (apparently).
		if (inputField.getType() == CHARACTER)
					// Proper value '\0' not
					// represented correctly in ASCII
					// by SDDF library! (bug in lib.)
			return (Value(CHARACTER, 0) = 'a');
		return (Value(inputField.getType(), 0) = 0);

	case VALUES_COUNT:
		break;			// return int(0).

	case BIN_COUNT:
		return (Value(INTEGER, 0) = numHistogramBins());

	case MEAN:
	case STDDEV:
		return Value(0.0);

	case HISTOGRAM:
		{
			Array histogram(INTEGER, 1);
			int histogramSize = numHistogramBins(), i;

			histogram.setDimSizes(&histogramSize);
			for (i = 0; i < histogramSize; i++)
				assert(histogram.setCellValue(0, i));

			return Value(&histogram);
		}

	case MIN_DIMENSION:
	case MAX_DIMENSION:
	case MEAN_DIMENSION:
	case STDDEV_DIMENSION:
		{
			MachineDataType type =
				(outputField <= MAX_DIMENSION) ? INTEGER :
				DOUBLE;
			Array a(type, 1);
			int arraySize = inputField.getDimension(), i;

			a.setDimSizes(&arraySize);
			for (i = 0; i < arraySize; i++)
				assert(a.setCellValue(0, i));

			return Value(&a);
		}

	default:
		assert(0);
	}

	return 0;
}


void
PerRecordSummary::readDataFirstPass(const RecordDossier &inDossier,
				    RecordDictionary &outRecDict)
{
	RecordDossier &outDossier =
		outRecDict.fetch(tag(inDossier.getTag()));

	int recordCount = (int) outDossier.getValue(RECORD_COUNT);
				// Casting the test (recordCount == 0)
				// directly to Boolean_ does *not* work on
				// the Paragon!
	Boolean_ isFirstRec;

	if (recordCount == 0)
		isFirstRec = TRUE_;
	else
		isFirstRec = FALSE_;

	assert(outDossier.setValue(RECORD_COUNT, ++recordCount));

	int i, vcField, outField = firstVarField;

	for (i = 0, vcField = VALUES_COUNT;
	     i < inDossier.entryCount();
	     i++, vcField += numVarFields) {
		const Value &inValue = inDossier.getValue(i);
		const Array *arrayP = inDossier.getArrayP(i);
		const int valuesCount = (int) outDossier.getValue(vcField);
		int j;
				// Same cast problem as for isFirstRec.
		Boolean_ isFirstVal;

		if (valuesCount == 0)
			isFirstVal = TRUE_;
		else
			isFirstVal = FALSE_;

		for (j = firstVarField; j <= lastVarField; j++) {
			updateFieldFirstPass(inValue, arrayP, outDossier,
					     outField, (VarFields) j,
					     isFirstVal, isFirstRec);
			outField++;
		}
	}
}


void
PerRecordSummary::updateFieldFirstPass(const Value &inValue,
				       const Array *arrayP,
				       RecordDossier &outDossier,
				       int outField, VarFields varField,
				       Boolean_ isFirstVal,
				       Boolean_ isFirstRec)
{
				// Static to avoid constructor overhead.
	static Value minValue, maxValue, nextValue, outValue;
	int i, numInCells, inDimension;
	double inValueD;
	Array *outArrayP;

	static int *dimBounds = NULL;
	static double *dimStats = NULL;
	static int maxDim = 0;

	if (arrayP && maxDim < (inDimension = arrayP->getDimension())) {
		if (dimBounds) {
			delete [] dimBounds;
			delete [] dimStats;
		}
		maxDim = inDimension;
		dimBounds = new int[maxDim];
		dimStats = new double[maxDim];
	}

	switch (varField) {
	case MIN:
		if (arrayP) {
			numInCells = arrayP->getCellCount();
			if (numInCells == 0)
				break;
			arrayP->getTheCellValue(minValue, 0);

			for (i = 1; i < numInCells; i++) {
				arrayP->getTheCellValue(nextValue, i);
				if (nextValue < minValue)
					minValue = nextValue;
			}
		} else
			minValue = inValue;
		if (isFirstVal) {
			assert(outDossier.setValue(outField, minValue));
			break;
		}
		outValue = outDossier.getValue(outField);
		if (minValue < outValue)
			assert(outDossier.setValue(outField, minValue));
		break;

	case MAX:
		if (arrayP) {
			numInCells = arrayP->getCellCount();
			if (numInCells == 0)
				break;
			arrayP->getTheCellValue(maxValue, 0);

			for (i = 1; i < numInCells; i++) {
				arrayP->getTheCellValue(nextValue, i);
				if (nextValue > maxValue)
					maxValue = nextValue;
			}
		} else
			maxValue = inValue;
		if (isFirstVal) {
			assert(outDossier.setValue(outField, maxValue));
			break;
		}
		outValue = outDossier.getValue(outField);
		if (maxValue > outValue)
			assert(outDossier.setValue(outField, maxValue));
		break;

	case VALUES_COUNT:
		outValue = outDossier.getValue(outField);
		outValue = (arrayP ? arrayP->getCellCount() : 1) +
			   (int) outValue;
		assert(outDossier.setValue(outField, outValue));
		break;

	case MEAN:
	case STDDEV:
		if (arrayP) {
			numInCells = arrayP->getCellCount();
			if (numInCells == 0)
				break;
			inValueD = 0.0;
			for (i = 0; i < numInCells; i++) {
				arrayP->getTheCellValue(nextValue, i);
				double valD = (double) nextValue;

				inValueD += (varField == STDDEV) ?
					(valD * valD) : valD;
			}
		} else {
			inValueD = (double) inValue;
			if (varField == STDDEV)
				inValueD *= inValueD;
		}
		outValue = outDossier.getValue(outField);
		outValue = inValueD + (double) outValue;
		assert(outDossier.setValue(outField, outValue));
		break;

	case UNDER_FLOW:	// Done after first pass.
	case OVER_FLOW:		// Done after first pass.
	case BIN_COUNT:		// Set by user.
	case HISTOGRAM:		// Done on second pass.
		break;

	case MIN_DIMENSION:
	case MAX_DIMENSION:
		if (!arrayP)
			break;

		outArrayP = outDossier.getArrayP(outField);

				// Can remove these checks after debugging
				// is done....
		assert(outArrayP);
		assert(outArrayP->getDimension() == 1);
		assert(outArrayP->getDimSizes()[0] == inDimension);
		assert(outArrayP->getType() == INTEGER);
		for (i = 0; i < inDimension; i++) {
			assert(outArrayP->getTheCellValue(outValue,
							  i));
			assert(outValue.isDefined());
		}

		if (isFirstRec) {
			for (i = 0; i < inDimension; i++)
				assert(outArrayP->setTheCellValue(i,
arrayP->getDimSizes()[i]));
			break;
		}

		for (i = 0; i < inDimension; i++)
			dimBounds[i] = (int) outArrayP->getTheCellValue(i);

		if (varField == MIN_DIMENSION) {
			for (i = 0; i < inDimension; i++)
				if (arrayP->getDimSizes()[i] <
				    dimBounds[i])
					dimBounds[i] =
						arrayP->getDimSizes()[i];
		} else {
			for (i = 0; i < inDimension; i++)
				if (arrayP->getDimSizes()[i] >
				    dimBounds[i])
					dimBounds[i] =
						arrayP->getDimSizes()[i];
		}

		for (i = 0; i < inDimension; i++)
			assert(outArrayP->setTheCellValue(i,
							  dimBounds[i]));
		break;

	case MEAN_DIMENSION:	// Process these normally.
	case STDDEV_DIMENSION:
		if (!arrayP)
			break;

		outArrayP = outDossier.getArrayP(outField);

				// Can remove these checks after debugging
				// is done....
		assert(outArrayP);
		assert(outArrayP->getDimension() == 1);
		assert(outArrayP->getDimSizes()[0] == inDimension);
		assert(outArrayP->getType() == DOUBLE);
		for (i = 0; i < inDimension; i++) {
			assert(outArrayP->getTheCellValue(outValue,
							  i));
			assert(outValue.isDefined());
		}

		for (i = 0; i < inDimension; i++)
			dimStats[i] =
				(double) outArrayP->getTheCellValue(i);
		for (i = 0; i < inDimension; i++) {
			double toAdd = arrayP->getDimSizes()[i];

			if (varField == STDDEV_DIMENSION)
				toAdd *= toAdd;
			dimStats[i] += toAdd;
		}
		for (i = 0; i < inDimension; i++)
			assert(outArrayP->setTheCellValue(i, dimStats[i]));
		break;

	default:
		assert(0);
	}
}


void
PerRecordSummary::endFirstPass(RecordDictionary &outRecDict)
{
	RecordDictionaryPIterator iter(outRecDict);
	RecordDossier *outDossier;

	while ((outDossier = iter.next())) {
		int i;
		int recordCount = (int) outDossier->getValue(RECORD_COUNT);

		if (recordCount == 0)
			continue;

		for (i = RECORD_COUNT;
		     i < outDossier->entryCount();
		     i += numVarFields) {
			int valuesCount = (int)
				outDossier->getValue(i + VALUES_COUNT);

			if (valuesCount == 0)
				continue;

			double mean = (double)
				outDossier->getValue(i + MEAN);

			mean /= valuesCount;
			assert(outDossier->setValue(i + MEAN, mean));

			double stddev = (double)
				outDossier->getValue(i + STDDEV);

			stddev = stddev / valuesCount - mean * mean;

			if (stddev < 0) {
				cout << "Round-off error computing "
					"field \""
				     << (const char *) (outDossier->
					getField(i + STDDEV)->getName())
				     << "\" of record with tag "
				     << outDossier->getTag() << ";\n";
				cout << "\tAdjusting value to 0.0."
				     << endl;
				stddev = 0;
			}

			stddev = sqrt(stddev);
			assert(outDossier->setValue(i + STDDEV, stddev));

			const Value low = outDossier->getValue(i + MIN);
			const Value high = outDossier->getValue(i + MAX);

			assert(outDossier->setValue(i + UNDER_FLOW, low));
			assert(outDossier->setValue(i + OVER_FLOW, high));

			Array *avgDimSizes =
				outDossier->getArrayP(i + MEAN_DIMENSION);
			int j;

			assert(avgDimSizes);
			for (j = 0; j < avgDimSizes->getDimSizes()[0];
			     j++) {
			     	double mean = (double)
					avgDimSizes->getTheCellValue(j);

				mean /= recordCount;
				avgDimSizes->setTheCellValue(j, mean);
			}

			Array *stddevDimSizes =
				outDossier->getArrayP(i +
						      STDDEV_DIMENSION);

			assert(stddevDimSizes);
			for (j = 0; j < stddevDimSizes->getDimSizes()[0];
			     j++) {
			     	double stddev = (double)
					stddevDimSizes->getTheCellValue(j);
			     	double mean = (double)
					avgDimSizes->getTheCellValue(j);

				stddev = sqrt(stddev / recordCount -
					      mean * mean);
				stddevDimSizes->setTheCellValue(j, stddev);
			}
		}
	}
}


void
PerRecordSummary::readDataSecondPass(const RecordDossier &inDossier,
				     RecordDictionary &outRecDict)
{
	RecordDossier &outDossier =
		outRecDict.fetch(tag(inDossier.getTag()));
	int i, outField;

	for (i = 0, outField = RECORD_COUNT;
	     i < inDossier.entryCount();
	     i++, outField += numVarFields) {
		const int numBins = (int)
			outDossier.getValue(outField + BIN_COUNT);
		const double underflow = (double)
			outDossier.getValue(outField + UNDER_FLOW);
		const double overflow = (double)
			outDossier.getValue(outField + OVER_FLOW);
		const double binWidth = (overflow - underflow) / numBins;
		Array *histogram =
			outDossier.getArrayP(outField + HISTOGRAM);
		assert(histogram);

		const Array *inData = inDossier.getArrayP(i);

		if (inData) {
			int j;

			for (j = 0; j < inData->getCellCount(); j++) {
				const double v = (double)
					inData->getTheCellValue(j);

				if (v < underflow || v > overflow)
					continue;

				int bin = int((v - underflow) / binWidth);

				if (bin == numBins)
					bin--;

				const int curCount = 1 + (int)
					histogram->getTheCellValue(bin);

				assert(histogram->
				       setTheCellValue(bin, curCount));
			}
		} else {
			const double v = (double) inDossier.getValue(i);

			if (v < underflow || v > overflow)
				continue;

			int bin = int((v - underflow) / binWidth);

			if (bin == numBins)
				bin--;

			const int curCount = 1 + (int)
				histogram->getTheCellValue(bin);

			assert(histogram->setTheCellValue(bin, curCount));
		}
	}
}


void
PerRecordSummary::endSecondPass(RecordDictionary &)
{
}
