/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Dave Kohr (drk@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SDDFStatistics.C - compute summary statistics for an SDDF file.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/SDDFlibrary/SamplePrograms/SDDFStatistics/RCS/SDDFStatistics.C,v 1.1 1994/04/06 22:25:54 aydt Exp $
 */

#include <stream.h>

// #include's from SDDF library.

#include "InitializeStatic.C"

#include "Attributes.h"
#include "CString.h"
#include "InputFileStreamPipe.h"
#include "PacketHeader.h"
#include "PipeReader.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"


// #include's from SDDFStatistics

#include "SummaryInfo.h"
#include "SDDFStats_Misc.h"


#define SDDFStatistics_VERSION "SDDFStatistics pre-alpha version 1.0"


/**********************************************************************
Read one input file, write summary to output file.
**********************************************************************/

static void
summarize(const char *inputFileName, const char *outputFileName)
{
			// Open input file.
	InputFileStreamPipe *In = new InputFileStreamPipe(inputFileName);

	if (!In->successfulOpen())
		fatalError(CString("Could not open input file \"") +
			   inputFileName + "\"");

	PipeReader *InPipe = In->createPipeReader();

	if (InPipe == NULL)
		fatalError(CString("Could not attach pipe to input "
" file \"") + inputFileName + "\"");

	PerFileSummary pfs(inputFileName, outputFileName);

	/****************************************************************
	 *    Process the packets in the files.  Read from the input 
	 *    pipe and process the right fields of the right records.
	 ****************************************************************/

	PacketHeader PH;

				// Perform first pass over the data.
	cout << "First pass." << endl;

	while ((PH = InPipe->getPacketHeader()).type != PIPE_EMPTY) {
		switch (PH.type) {

		case PKT_ATTRIBUTE:
			{
				Attributes attr;

				InPipe->getAttributes(attr);
				pfs.readAttributeFirstPass(attr);
			}
			break;	// CC complains if this is inside the {}'s.

		case PKT_DESCRIPTOR:
			{
				StructureDescriptor desc;

				InPipe->getDescriptor(desc);
				pfs.readDescriptorFirstPass(desc, PH.tag);
			}
			break;

		case PKT_DATA:
			{
				RecordDossier &dossier =
					pfs.getInRecDict().fetch(PH.tag);
				InPipe->getData(dossier);
				pfs.readDataFirstPass(dossier, PH.tag);
			}
			break;

		case PKT_COMMAND:
			pfs.readCommandFirstPass();
			break;

		default:
			fatalError("Unrecognized packet type " +
				   int2CString(PH.type) + "; dying!");
		}
	}
	pfs.endFirstPass();
	delete InPipe;
	delete In;

	In = new InputFileStreamPipe(inputFileName);
	if (!In->successfulOpen())
		fatalError(CString("Could not open input file \"") +
			   inputFileName + "\"");

	InPipe = In->createPipeReader();
	if (InPipe == NULL)
		fatalError(CString("Could not attach pipe to input "
" file \"") + inputFileName + "\"");

				// Perform second pass over the data.
	cout << "Second pass." << endl;

	while ((PH = InPipe->getPacketHeader()).type != PIPE_EMPTY) {
		switch (PH.type) {

		case PKT_ATTRIBUTE:
			{
				Attributes attr;

				InPipe->getAttributes(attr);
				pfs.readAttributeSecondPass(attr);
			}
			break;

		case PKT_DESCRIPTOR:
			{
				StructureDescriptor desc;

				InPipe->getDescriptor(desc);
				pfs.readDescriptorSecondPass(desc, PH.tag);
			}
			break;

		case PKT_DATA:
			{
				RecordDossier &dossier =
					pfs.getInRecDict().fetch(PH.tag);
				InPipe->getData(dossier);
				pfs.readDataSecondPass(dossier, PH.tag);
			}
			break;

		case PKT_COMMAND:
			pfs.readCommandSecondPass();
			break;

		default:
			fatalError("Unrecognized packet type " +
				   int2CString(PH.type) + "; dying!");
		}
	}
	pfs.endSecondPass();
	delete InPipe;
	delete In;

	pfs.writeSummaryFile(outputFileName);
}


static char *progName;

/**********************************************************************
Print a nice usage message and exit with value exitVal.
**********************************************************************/

static void exitWithUsage(int exitVal)
{
	
	CString usageMsg = "Usage: ";

	usageMsg = usageMsg + progName + " inputFile [outputFile]\n"
"\nwhere\n\n"
"\tinputFile is the input SDDF file to summarize, and\n"
"\toutputFile is the output ASCII summary file in SDDF format (default\n"
"\t\tname is \"<inputFile>.summary\").";

	fatalError(usageMsg, exitVal);
}


/**********************************************************************
Parse arguments from command-line, process input file using summarize().
**********************************************************************/

int main(int argc, char **argv)
{
	progName = argv[0];

	if (argc < 2 || argc > 3)
		exitWithUsage(1);

	CString inputFileName = argv[1];

	CString outputFileName;

	if (argc == 3)
		outputFileName = argv[2];
	else
		outputFileName = inputFileName + ".summary";

	summarize(inputFileName, outputFileName);

	return 0;
}
