/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ExpandField.C Program to read a expansion template file and a SDDF data 
 *               file and expand specified data records into multiple 
 *               records in the output SDDF file when matches are found for
 *		 specified fields & values.  Only scalar fields can be
 *	         expanded.
 *
 */
#include <fstream.h>
#include <stdlib.h>
#include <string.h>

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "ConversionPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

extern void initializeExpansionInfo();
extern void processDataPacket( RecordDossier& dossier, PipeWriter *OutPipe );
extern void expandIt( int rIdx, int fIdx, RecordDossier& dossier, 
		      			  PipeWriter *OutPipe );

/*  
 * Structures and global data used to keep track of what records/fields
 * should be expanded.
 */
struct expandField {
	CString		name;
	Value		matchValue;
	int		valueCnt;
	Value		*newValue;
};

struct expandRecord {
	CString		recordName;
	int		fieldCnt;
	expandField 	*field;
};

int	      expandCnt;
expandRecord *expandInfo;
Boolean_     *fieldUnchanged;


/*
 * Main program - one argument, the name of the input file, can be given
 */
main( int argc, char **argv )
{
	initializeExpansionInfo();

 	enum FileType     { UNDEFINED, ASCII, BINARY, CONVERT };
        enum Status       { INVALID, VALID };
        char BUF[512];          

       /****************************************************************
        *    Get the name of the input file, and open the file
        ****************************************************************/
        Status               inputFileStatus;
        InputFileStreamPipe  *In;
        PipeReader           *InPipe;

        if ( argc > 2 ) {
           cerr << form( "Usage: %s [ filename ]\n", argv[0] );
           exit (-1);
        }

        do {
            if ( argc == 2 ) {
                strcpy( BUF, argv[1] );
                argc--;
            } else {
                cerr << "Please enter name of the input SSDF file: ";
                cin >> BUF;
            }

            In = new InputFileStreamPipe( BUF );
            if ( In->successfulOpen() ) {
                inputFileStatus = VALID;
                InPipe = In->createPipeReader();
                if ( InPipe == NULL ) {
                    cerr << "ERROR: Couldn't attach pipe to input file\n";
                    inputFileStatus = INVALID;
                    delete In;
                }
            } else {
                cerr << "ERROR: Problem with input file\n\n";
                inputFileStatus = INVALID;
                delete In;
            }
        } while ( inputFileStatus == INVALID );

 	/****************************************************************
         *    Ask if output file is ascii, binary, or conversion format
         ****************************************************************/
	FileType	outputFileType = UNDEFINED;

        while ( outputFileType == UNDEFINED ) {
            cerr << "\nOutput file in Ascii, Binary or Converted "
		 << "(reverse byte order)  [A, B, or C]: ";
            cin >> BUF;

            if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
                outputFileType = ASCII;
            } else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
                outputFileType = BINARY;
            } else if ( BUF[0] == 'c' || BUF[0] == 'C' ) {
                outputFileType = CONVERT;
            } else {
                cerr << "ERROR: invalid format type.\n";
            }
        }

 	/****************************************************************
         *    Get the name of the output file, and open the file
         ****************************************************************/
	Status 		     outputFileStatus;
	OutputFileStreamPipe *Out;
	PipeWriter           *OutPipe;

	do {
  	    cerr << "Please enter name of the output SDDF file: ";
	    cin >> BUF;

	    Out = new OutputFileStreamPipe( BUF );
	    if ( Out->successfulOpen() ) {
		outputFileStatus = VALID;
		if ( outputFileType == ASCII ) { 
		    OutPipe = new AsciiPipeWriter( Out );
		} else if ( outputFileType == BINARY ) { 
		    OutPipe = new BinaryPipeWriter( Out );
		} else {
		    OutPipe = new ConversionPipeWriter( Out );
		}
	    } else {
		outputFileStatus = INVALID;
		delete Out;
	    }
	} while ( outputFileStatus == INVALID );

 	/****************************************************************
         *    Process the packets in the files.  Read from the input 
	 *    pipe and write to the output pipe.  For Data packets,
	 *    call processDataPacket() to see if any fields need 
	 *    expansion -- writing to output pipe is done in that routine.
         ****************************************************************/
	Attributes *Ap;
	StructureDescriptor *SDp;
	RecordDictionary RD;

	PacketHeader PH = InPipe->getPacketHeader();

	while( PH.type != PIPE_EMPTY ) {

	    switch( PH.type ) {
	      case PKT_ATTRIBUTE:
		  Ap = new Attributes();
		  InPipe->getAttributes( *Ap );
		  OutPipe->putAttributes( *Ap );
		  delete Ap;
		  break;
	      case PKT_DESCRIPTOR:
		  SDp = new StructureDescriptor();
		  InPipe->getDescriptor( *SDp );
		  OutPipe->putDescriptor( *SDp, PH.tag );
		  RD.insert( PH.tag, *SDp );
		  delete SDp;
		  break;
	      case PKT_DATA:
		  {
		  RecordDossier& dossier = RD.fetch( PH.tag );
		  InPipe->getData( dossier );
		  processDataPacket( dossier, OutPipe );
		  }
		  break;
	      case PKT_COMMAND:
		  OutPipe->putCommand( PH.tag );
		  break;
	      }
	      PH = InPipe->getPacketHeader();
	}

	delete InPipe;
	delete In;
	delete OutPipe;
	delete Out;
}

void
processDataPacket( RecordDossier& dossier, PipeWriter *OutPipe )
{
	for ( int i = 0; i < expandCnt; i++ ) {
	    if ( dossier.getName() == expandInfo[i].recordName ) {
		// A record name match;  start the recursive calls to expandIt
		// which will write the output data record(s) as required
		expandIt( i, 0, dossier, OutPipe );
		return;
	    }
	}

	// All done with no record matches!  Write the original dossier.
	OutPipe->putData( dossier );
}


void
expandIt( int rIdx, int fIdx, RecordDossier& dossier, PipeWriter *OutPipe ) 
{
	int i;

	//  If this is the first field we initialize our array of fields
	//  for the dossier to keep track of those which are "currently"
	//  being updated.  This is to make sure that we don't update
	//  a matched value to a replacement value & then update the 
	//  replacement in a future call to expandIt for the same dossier. 
	if ( fIdx == 0 ) {
	    int entryCount = dossier.entryCount();
	    fieldUnchanged = new Boolean_[ entryCount ];

	    for ( i = 0; i < entryCount; i++ ) {
		fieldUnchanged[i] = TRUE_;
	    }
	} 

	int fieldID = dossier.getFieldID( expandInfo[rIdx].field[fIdx].name );
		
	if ( fieldUnchanged[fieldID] && 
	     		( dossier.getValue( fieldID ) == 
	                  expandInfo[rIdx].field[fIdx].matchValue ) ) {

	    //  We have a field to expand;  Mark that we are changing it.
	    //  and update the dossier with new values. 
	    //  If this is our last field then write output record for each
	    //  new value.  If it isn't the last field, then continue calling
	    //  expandIt for the next field for each new value.

	    fieldUnchanged[fieldID] = FALSE_;
	    int k;
	    for ( k = 0; k < expandInfo[rIdx].field[fIdx].valueCnt; k++ ) {
	        dossier.setValue( expandInfo[rIdx].field[fIdx].name, 
		                  expandInfo[rIdx].field[fIdx].newValue[k] );
		if ( fIdx == (expandInfo[rIdx].fieldCnt - 1) ) {
	    	    OutPipe->putData( dossier );
		} else {
		    expandIt( rIdx, fIdx+1, dossier, OutPipe );
		}
	    }
	    //  Reset the dossier field value to original value we matched
	    //  for the current field we are processing.  This is done so
	    //  that other calls to expandIt with the same fIdx will match
	    //  when they should
	    dossier.setValue( expandInfo[rIdx].field[fIdx].name,
			      expandInfo[rIdx].field[fIdx].matchValue );

	    //  Since we are exiting the loop that says we were updating
	    //  a value, reset flag.
	    fieldUnchanged[fieldID] = TRUE_;

	} else {

	    //  We don't have a field to expand;  
	    //  If this is our last field then write output record.
	    //  If it isn't the last field, then continue calling
	    //  expandIt for the next field.
 	    if ( fIdx == (expandInfo[rIdx].fieldCnt - 1) ) {
   	        OutPipe->putData( dossier );
	    } else {
	        expandIt( rIdx, fIdx+1, dossier, OutPipe );
	    }
	}
	if ( fIdx == 0 ) {
	    delete[] fieldUnchanged;
	} 
}


/*
 * Macro to get the next line from "fin" that doesn't begin with '#"
 * into BUF. 
 */
#define GETLINE() do { fin.getline( BUF, 512 ); } while ( BUF[0] == '#' )

void
initializeExpansionInfo()
{
        char            BUF[512];          
        ifstream        fin;

	MachineDataType fieldType;

	do {
            cerr << "File containing expansion template information: ";
            cin >> BUF;

	    fin.open( BUF );
	    if ( !fin ) { 
                cerr << "ERROR: Problem opening expansion template file\n\n";
            }
        } while ( !fin );


	/*
	 * Number of record types to consider expanding
	 */
	GETLINE();
	expandCnt = atoi( BUF );

	expandInfo = new expandRecord[expandCnt];

	int i, j, k;

	/* 
	 * Loop through records
	 */
	for ( i = 0; i < expandCnt; i++ ) {

	    // Name of record
	    GETLINE();
	    expandInfo[i].recordName = BUF;	

	    // Number of fields to check for expansion match
	    GETLINE();
	    expandInfo[i].fieldCnt = atoi( BUF );	

	    expandInfo[i].field = new expandField[ expandInfo[i].fieldCnt ];

	    /* 
	     * Loop through fields
	     */	
	    for ( j = 0; j < expandInfo[i].fieldCnt; j++ ) {

	        // Name of field
	    	GETLINE();
	    	expandInfo[i].field[j].name = BUF;

		// Field type (char, int, float, or double)
		GETLINE();
	        if ( strcmp( BUF, typeNameArray[CHARACTER] ) == 0 ) {
		    fieldType = CHARACTER;
		} else if ( strcmp( BUF, typeNameArray[INTEGER] ) == 0 ) {
		    fieldType = INTEGER;
		} else if ( strcmp( BUF, typeNameArray[FLOAT] ) == 0 ) {
		    fieldType = FLOAT;
		} else if ( strcmp( BUF, typeNameArray[DOUBLE] ) == 0 ) {
		    fieldType = DOUBLE;
		} else {
		    cerr << "Bad field type \"" << BUF << "\" ... aborting\n";
		    exit( -2 );
		}

	        // Value to match for expansion
	    	GETLINE();
		switch( fieldType ) {
		    case CHARACTER:
	            	expandInfo[i].field[j].matchValue = BUF[0];
			break;
		    case INTEGER:
	            	expandInfo[i].field[j].matchValue = atoi( BUF );
			break;
		    case FLOAT:
	            	expandInfo[i].field[j].matchValue = (float) atof( BUF );
			break;
		    case DOUBLE:
	            	expandInfo[i].field[j].matchValue = 
							   (double) atof( BUF );
			break;
		    case UNDEFINED:
		    default:
			cerr << "Unexpected field type - exiting\n";
			exit( -3 );
		}


	        //  Number of resulting records
	    	GETLINE();
	        expandInfo[i].field[j].valueCnt = atoi( BUF );

	        expandInfo[i].field[j].newValue = 
				new Value[ expandInfo[i].field[j].valueCnt ];

	    	/*
		 * Loop through Values to replace matchValue in output records
		 */
		for ( k = 0; k < expandInfo[i].field[j].valueCnt; k++ ) {
		    GETLINE();

		    switch( fieldType ) {
		        case CHARACTER:
	                    expandInfo[i].field[j].newValue[k] = BUF[0];
		    	    break;
		        case INTEGER:
	                    expandInfo[i].field[j].newValue[k] = atoi( BUF );
		    	    break;
		        case FLOAT:
	                    expandInfo[i].field[j].newValue[k] = 
							    (float) atof( BUF );
		    	    break;
		        case DOUBLE:
	                    expandInfo[i].field[j].newValue[k] = 
							   (double) atof( BUF );
		    	    break;
		        case UNDEFINED:
		        default:
			    cerr << "Unexpected field type - exiting\n";
			    exit( -4 );
		    }
		}
	    
	    }
	}
	fin.close();

}
