/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Bill Whitehouse (whotehou@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * DataCharacteristics.C -  methods for the DataCharacteristics class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Primitives/RCS/DataCharacteristics.C,v 1.5 1994/03/15 19:43:34 aydt Exp $
 */

#include <unistd.h>

#include "DataCharacteristics.h"

#include "Assert.h"
#include "InputFileStreamPipe.h"

const char *SupportedError =
		"Only %s supported in this release" ;

const char *ConversionError = 
		"Conversion between %s not supported in this release" ;

DataCharacteristics::DataCharacteristics()
{
	_setClassName( MY_CLASS );

	bytesInHeader = Sddf_BytesInHeader;
	byteOrdering = NativeByteOrdering;
	integerRepresentation = NativeIntegerRepresentation;
	singleFloatRepresentation = NativeSingleFloatRepresentation;
	doubleFloatRepresentation = NativeDoubleFloatRepresentation;
	extraFloatRepresentation = NativeExtraFloatRepresentation;
	charRepresentation = NativeCharRepresentation;
	bytesInChar = NativeBytesInChar;
	bytesInShort = NativeBytesInShort;
	bytesInInteger = NativeBytesInInteger;
	bytesInLong = NativeBytesInLong;
	bytesInFloat = NativeBytesInFloat;
	bytesInDouble = NativeBytesInDouble;
	bytesInLongDouble = NativeBytesInLongDouble;

	inputPipe = NULL;
}

int 		/* static */
DataCharacteristics::writeNativeHeader( char *const bufPtr, int bufLen ) 
{
	if ( bufLen < Sddf_BytesInHeader ) {
	    return -1;
	}

	int i = 0;
	bufPtr[i++] = 'S';
	bufPtr[i++] = 'D';
	bufPtr[i++] = 'D';
	bufPtr[i++] = 'F';
	bufPtr[i++] = 'B';
	bufPtr[i++] = (char)Sddf_BytesInHeader;
	bufPtr[i++] = (char)NativeByteOrdering;
	bufPtr[i++] = (char)NativeIntegerRepresentation;
	bufPtr[i++] = (char)NativeSingleFloatRepresentation;
	bufPtr[i++] = (char)NativeDoubleFloatRepresentation;
	bufPtr[i++] = (char)NativeExtraFloatRepresentation;
	bufPtr[i++] = (char)NativeCharRepresentation;
	bufPtr[i++] = (char)NativeBytesInChar;
	bufPtr[i++] = (char)NativeBytesInShort;
	bufPtr[i++] = (char)NativeBytesInInteger;
	bufPtr[i++] = (char)NativeBytesInLong;
	bufPtr[i++] = (char)NativeBytesInFloat;
	bufPtr[i++] = (char)NativeBytesInDouble;
	bufPtr[i++] = (char)NativeBytesInLongDouble;

	Assert( i == Sddf_BytesInHeader );

	return i;
}

int
DataCharacteristics::getAtomSize( MachineDataType atomType ) const
{
	int size;

	switch ( atomType ) {

	    case CHARACTER:
		size = bytesInChar;
		break;

	    case INTEGER:
		size = bytesInInteger;
		break;

	    case FLOAT:
		size = bytesInFloat;
		break;

	    case DOUBLE:
		size = bytesInDouble;
		break;
	
	    case UNDEFINED:
		size = 0;
		break;
	}

	return size;
}

Boolean_
DataCharacteristics::initializeFromFile( InputFileStreamPipe * pipe,
					 int fileDescr )
{
	/* 
	 * By the time we get here, we've already read the first
	 * 5 bytes of the file which contain "SDDFB".  The next byte
	 * should give us the total number of bytes in the header.
	 */
	inputPipe = pipe;

	char c;
	int bytes = read( fileDescr, &c, 1 );
	if ( bytes != 1 ) {
	    error( "Unable to read SDDF header length from %s",
	            pipe->getObjectName() );
	    return FAILURE_;
	} else {
	    bytesInHeader = (int)c;
	}

	/*
	 * If the size of the header is less than 6 we know it's bogus.
	 * If it's less than we expect, we are probably reading an older file.
	 * If it's greater than we expect, we are probably reading a newer file
	 *  and will get unpredictable results.
	 */
	if ( bytesInHeader < 6 ) {
	    error( "Got header size of %d from %s - expected %d",
		     bytesInHeader, pipe->getObjectName(), Sddf_BytesInHeader );
	    return FAILURE_;
	} else if ( bytesInHeader < Sddf_BytesInHeader ) {
	    warning( "Got header size of %d from %s - expected %d",
		     bytesInHeader, pipe->getObjectName(), Sddf_BytesInHeader );
	} else if ( bytesInHeader > Sddf_BytesInHeader ) {
	    error( "%s header size is %d - expected %d; ignoring extra bytes",
		     pipe->getObjectName(), bytesInHeader, Sddf_BytesInHeader );
	    warning( "Will attempt to read file but results may be incorrect.");
	}

	/*
	 * Allocate space to hold the header & read it in.  If we fail to
	 * get as much as we expect, error.
	 */

	int headerBytesToRead = bytesInHeader - 6;
	char *buf = new char[ headerBytesToRead ];

	bytes = read( fileDescr, buf, headerBytesToRead );

	if ( bytes != headerBytesToRead ) {
	    error( "Unable to read remaining %d bytes of SDDF header from %s",
		    headerBytesToRead, pipe->getObjectName() );
	    delete[] buf;
	    return FAILURE_;
	}

	/*
	 * Now, process those bytes we know how to deal with.
	 */
	int rc = SUCCESS_;
	int irc;
	int i;

	for ( i = 0; i < bytesInHeader; i++ ) {

	    switch ( i ) {

		case 0:			// Byte order
		    irc = setByteOrdering( (Sddf_BYTE_ORDERING)buf[i] );
		    break;

		case 1:			// Integer Representation
		    irc = setIntegerRepresentation( 
					(Sddf_INT_REPRESENT)buf[i] );
		    break;

		case 2:			// Single Precision Float Representation
		    irc = setSingleFloatRepresentation( 
					(Sddf_SNGLP_REPRESENT)buf[i] );
		    break;

		case 3:			// Double Precision Float Representation
		    irc = setDoubleFloatRepresentation( 
					(Sddf_DBLP_REPRESENT)buf[i] );
		    break;

		case 4:			// Extra Precision Float Representation
		    irc = setExtraFloatRepresentation( 
					(Sddf_EXTRA_REPRESENT)buf[i] );
		    break;

		case 5:			// Character Representation
		    irc = setCharRepresentation( (Sddf_CHAR_REPRESENT)buf[i] );
		    break;

		case 6:			// Bytes in Character
		    irc = setBytesInChar( (int)buf[i] );
		    break;

		case 7:			// Bytes in Short
		    irc = setBytesInShort( (int)buf[i] );
		    break;
		    
		case 8:			// Bytes in Integer
		    irc = setBytesInInteger( (int)buf[i] );
		    break;

		case 9:			// Bytes in Long
		    irc = setBytesInLong( (int)buf[i] );
		    break;

		case 10:		// Bytes in Float
		    irc = setBytesInFloat( (int)buf[i] );
		    break;

		case 11:		// Bytes in Double
		    irc = setBytesInDouble( (int)buf[i] );
		    break;

		case 12:		// Bytes in Long Double
		    irc = setBytesInLongDouble( (int)buf[i] );
		    break;

		default:		// Overflow bytes
		    break;

	    }

	    rc = rc && irc;		

	}

	delete[] buf;
	return CnvToBoolean_( rc );
}


Boolean_
DataCharacteristics::notNative() const
{
	if ( byteOrdering != NativeByteOrdering ) {
	    return TRUE_;
	}
	if ( integerRepresentation != NativeIntegerRepresentation ) {
	    return TRUE_;
	}
	if ( singleFloatRepresentation != NativeSingleFloatRepresentation ) {
	    return TRUE_;
	}
	if ( doubleFloatRepresentation != NativeDoubleFloatRepresentation ) {
	    return TRUE_;
	}
	if ( extraFloatRepresentation != NativeExtraFloatRepresentation ) {
	    return TRUE_;
	}
	if ( charRepresentation != NativeCharRepresentation ) {
	    return TRUE_;
	}
	if ( bytesInChar != NativeBytesInChar ) {
	    return TRUE_;
	}
	if ( bytesInShort != NativeBytesInShort ) {
	    return TRUE_;
	}
	if ( bytesInInteger != NativeBytesInInteger ) {
	    return TRUE_;
	}
	if ( bytesInLong != NativeBytesInLong ) {
	    return TRUE_;
	}
	if ( bytesInFloat != NativeBytesInFloat ) {
	    return TRUE_;
	}
	if ( bytesInDouble != NativeBytesInDouble ) {
	    return TRUE_;
	}
	if ( bytesInLongDouble != NativeBytesInLongDouble ) {
	    return TRUE_;
	}
	return FALSE_;
}

int 		
DataCharacteristics::writeHeader( char *const bufPtr, int bufLen ) const
{
	if ( bufLen < Sddf_BytesInHeader ) {
	    return -1;
	}

	int i = 0;
	bufPtr[i++] = 'S';
	bufPtr[i++] = 'D';
	bufPtr[i++] = 'D';
	bufPtr[i++] = 'F';
	bufPtr[i++] = 'B';
	bufPtr[i++] = (char)Sddf_BytesInHeader;
	bufPtr[i++] = (char)byteOrdering;
	bufPtr[i++] = (char)integerRepresentation;
	bufPtr[i++] = (char)singleFloatRepresentation;
	bufPtr[i++] = (char)doubleFloatRepresentation;
	bufPtr[i++] = (char)extraFloatRepresentation;
	bufPtr[i++] = (char)charRepresentation;
	bufPtr[i++] = (char)bytesInChar;
	bufPtr[i++] = (char)bytesInShort;
	bufPtr[i++] = (char)bytesInInteger;
	bufPtr[i++] = (char)bytesInLong;
	bufPtr[i++] = (char)bytesInFloat;
	bufPtr[i++] = (char)bytesInDouble;
	bufPtr[i++] = (char)bytesInLongDouble;

	Assert( i == Sddf_BytesInHeader );

	return i;
}


/*
 *	Getting and Putting data - ordered by type of data
 */


/* 	Characters - only byte order conversions supported for now        */

int 
DataCharacteristics::functionGet( const char *buffer, char *value ) const
{
	Assert( bytesInChar == NativeBytesInChar );

        char *vBytes = value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) { 
	    while( bytes < bytesInChar ) {
                *vBytes++ = *buffer++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInChar - 1;
	    while( bytes < bytesInChar ) {
                *vBytes-- = *buffer++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionGet( char **bufptr, char *value ) const
{
	Assert( bytesInChar == NativeBytesInChar );

        char *vBytes = value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < bytesInChar ) {
                *vBytes++ = *( *bufptr )++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInChar - 1;
	    while( bytes < bytesInChar ) {
                *vBytes-- = *( *bufptr)++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char *buffer, const char *value ) const
{
	Assert( bytesInChar == NativeBytesInChar );

        const char *vBytes = value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < sizeof(int) ) {
                *buffer++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInChar - 1;
            while( bytes < bytesInChar ) {
                *buffer++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char **bufptr, const char *value ) const
{
	Assert( bytesInChar == NativeBytesInChar );

        const char *vBytes = value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering) {
            while( bytes < bytesInChar ) {
                *( *bufptr )++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInChar - 1;
            while( bytes < bytesInChar ) {
                *( *bufptr )++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

/* 	Integers - only byte order conversions supported for now        */

int 
DataCharacteristics::functionGet( const char *buffer, int *value ) const
{
	Assert( bytesInInteger == NativeBytesInInteger );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) { 
	    while( bytes < bytesInInteger ) {
                *vBytes++ = *buffer++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInInteger - 1;
	    while( bytes < bytesInInteger ) {
                *vBytes-- = *buffer++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionGet( char **bufptr, int *value ) const
{
	Assert( bytesInInteger == NativeBytesInInteger );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < bytesInInteger ) {
                *vBytes++ = *( *bufptr )++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInInteger - 1;
	    while( bytes < bytesInInteger ) {
                *vBytes-- = *( *bufptr)++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char *buffer, const int *value ) const
{
	Assert( bytesInInteger == NativeBytesInInteger );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < sizeof(int) ) {
                *buffer++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInInteger - 1;
            while( bytes < bytesInInteger ) {
                *buffer++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char **bufptr, const int *value ) const
{
	Assert( bytesInInteger == NativeBytesInInteger );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering) {
            while( bytes < bytesInInteger ) {
                *( *bufptr )++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInInteger - 1;
            while( bytes < bytesInInteger ) {
                *( *bufptr )++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

/* 	Floats - only byte order conversions supported for now        */

int 
DataCharacteristics::functionGet( const char *buffer, float *value ) const
{
	Assert( bytesInFloat == NativeBytesInFloat );
	Assert( singleFloatRepresentation == NativeSingleFloatRepresentation );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) { 
	    while( bytes < bytesInFloat ) {
                *vBytes++ = *buffer++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInFloat - 1;
	    while( bytes < bytesInFloat ) {
                *vBytes-- = *buffer++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionGet( char **bufptr, float *value ) const
{
	Assert( bytesInFloat == NativeBytesInFloat );
	Assert( singleFloatRepresentation == NativeSingleFloatRepresentation );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < bytesInFloat ) {
                *vBytes++ = *( *bufptr )++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInFloat - 1;
	    while( bytes < bytesInFloat ) {
                *vBytes-- = *( *bufptr)++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char *buffer, const float *value ) const
{
	Assert( bytesInFloat == NativeBytesInFloat );
	Assert( singleFloatRepresentation == NativeSingleFloatRepresentation );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < sizeof(int) ) {
                *buffer++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInFloat - 1;
            while( bytes < bytesInFloat ) {
                *buffer++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char **bufptr, const float *value ) const
{
	Assert( bytesInFloat == NativeBytesInFloat );
	Assert( singleFloatRepresentation == NativeSingleFloatRepresentation );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering) {
            while( bytes < bytesInFloat ) {
                *( *bufptr )++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInFloat - 1;
            while( bytes < bytesInFloat ) {
                *( *bufptr )++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

/* 	Doubles - only byte order conversions supported for now        */

int 
DataCharacteristics::functionGet( const char *buffer, double *value ) const
{
	Assert( bytesInDouble == NativeBytesInDouble );
	Assert( doubleFloatRepresentation == NativeDoubleFloatRepresentation );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) { 
	    while( bytes < bytesInDouble ) {
                *vBytes++ = *buffer++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInDouble - 1;
	    while( bytes < bytesInDouble ) {
                *vBytes-- = *buffer++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionGet( char **bufptr, double *value ) const
{
	Assert( bytesInDouble == NativeBytesInDouble );
	Assert( doubleFloatRepresentation == NativeDoubleFloatRepresentation );

        char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < bytesInDouble ) {
                *vBytes++ = *( *bufptr )++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInDouble - 1;
	    while( bytes < bytesInDouble ) {
                *vBytes-- = *( *bufptr)++;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char *buffer, const double *value ) const
{
	Assert( bytesInDouble == NativeBytesInDouble );
	Assert( doubleFloatRepresentation == NativeDoubleFloatRepresentation );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering ) {
            while( bytes < sizeof(int) ) {
                *buffer++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInDouble - 1;
            while( bytes < bytesInDouble ) {
                *buffer++ = *vBytes--;
                bytes++;
            }
	}
        return ( bytes );
}

int 
DataCharacteristics::functionPut( char **bufptr, const double *value ) const
{
	Assert( bytesInDouble == NativeBytesInDouble );
	Assert( doubleFloatRepresentation == NativeDoubleFloatRepresentation );

        const char *vBytes = (char *)value;
        int bytes = 0;

	if ( byteOrdering == NativeByteOrdering) {
            while( bytes < bytesInDouble ) {
                *( *bufptr )++ = *vBytes++;
                bytes++;
            }
	} else {
	    vBytes = vBytes + bytesInDouble - 1;
            while( bytes < bytesInDouble ) {
                *( *bufptr )++ = *vBytes--;
                bytes++;
            }
	}
	return ( bytes );
}

/* 
 *	Setting Characteristics - order they appear in headers.
 */

Boolean_
DataCharacteristics::setByteOrdering( Sddf_BYTE_ORDERING newByteOrdering )
{
	Boolean_ rc;

	switch ( newByteOrdering ) {

	    case Sddf_BIG_ENDIAN:
	    case Sddf_LITTLE_ENDIAN:
		byteOrdering = newByteOrdering;
		rc = SUCCESS_;
		break;

	    default:
		error( "Unsupported byte ordering: %d",
			newByteOrdering );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setByteOrderingNonNative( )
{
	Boolean_ rc;

	switch ( NativeByteOrdering ) {

	    case Sddf_BIG_ENDIAN:
		byteOrdering = Sddf_LITTLE_ENDIAN;
		rc = SUCCESS_;
		break;

	    case Sddf_LITTLE_ENDIAN:
		byteOrdering = Sddf_BIG_ENDIAN;
		rc = SUCCESS_;
		break;

	    default:
		error( "Unsupported Native byte ordering: %d",
			NativeByteOrdering );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setIntegerRepresentation( 
			Sddf_INT_REPRESENT newIntegerRepresentation )
{
	Boolean_ rc;

	switch ( newIntegerRepresentation ) {

	    case Sddf_TWOS_COMPLEMENT:
		integerRepresentation = newIntegerRepresentation;
		rc = SUCCESS_;
		break;

	    case Sddf_ONES_COMPLEMENT:
	    case Sddf_SIGN_MAGNITUDE:
		error( SupportedError, "two's complement representation" );
		rc = FAILURE_;
		break;

	    default:
		error( "Unrecognized Integer Representation: %d",
			newIntegerRepresentation );
		rc = FAILURE_;
		break;
	}

	return rc;
}
	    
Boolean_
DataCharacteristics::setSingleFloatRepresentation( 
			  Sddf_SNGLP_REPRESENT newSingleFloatRepresentation )
{
	Boolean_ rc;

	switch ( newSingleFloatRepresentation ) {

	    case Sddf_IEEE_S:
		singleFloatRepresentation = newSingleFloatRepresentation;
		rc = SUCCESS_;
		break;

	    case Sddf_VAX_S:
	    case Sddf_IBM_S:
	    case Sddf_CRAY_S:
		error( SupportedError, "IEEE Single-Precision Floating Point" );
		rc = FAILURE_;
		break;

	    default:
		error( "Unrecognized Single-Precision Float Representation: %d",
			newSingleFloatRepresentation );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setDoubleFloatRepresentation( 
			  Sddf_DBLP_REPRESENT newDoubleFloatRepresentation )
{
	Boolean_ rc;

	switch ( newDoubleFloatRepresentation ) {

	    case Sddf_IEEE_D:
		doubleFloatRepresentation = newDoubleFloatRepresentation;
		rc = SUCCESS_;
		break;

	    case Sddf_VAX_D:
	    case Sddf_IBM_D:
	    case Sddf_CRAY_D:
		error( SupportedError, "IEEE Double-Precision Floating Point" );
		rc = FAILURE_;
		break;

	    default:
		error( "Unrecognized Double-Precision Float Representation: %d",
			newDoubleFloatRepresentation );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setExtraFloatRepresentation( 
			  Sddf_EXTRA_REPRESENT newExtraFloatRepresentation )
{
	/*
	 * SDDF currently doesn't use long doubles so anything reconized
	 * is accepted.
	 */
	Boolean_ rc;

	switch ( newExtraFloatRepresentation ) {

	    case Sddf_IEEE_E:
	    case Sddf_VAX_E:
	    case Sddf_IBM_E:
	    case Sddf_CRAY_E:
		extraFloatRepresentation = newExtraFloatRepresentation;
		rc = SUCCESS_;
		break;

	    default:
		error( "Unrecognized Extra-Precision Float Representation: %d",
			newExtraFloatRepresentation );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setCharRepresentation( 
			Sddf_CHAR_REPRESENT newCharRepresentation )
{
	Boolean_ rc;

	switch ( newCharRepresentation ) {

	    case Sddf_ASCII:
	    case Sddf_EBCDIC:
		if ( NativeCharRepresentation != newCharRepresentation ) {
		    error( ConversionError, "ASCII and EBCDIC" );
		    rc = FAILURE_;
		} else {
		    charRepresentation = newCharRepresentation;
		    rc = SUCCESS_;
		}
		break;

	    default:
		error( "Unrecognized Character Representation: %d",
			newCharRepresentation );
		rc = FAILURE_;
		break;
	}

	return rc;
}

Boolean_
DataCharacteristics::setBytesInChar( int newBytesInChar )
{
	Boolean_ rc;

	if ( NativeBytesInChar != newBytesInChar ) {
	    error( ConversionError, "characters of different sizes" );
	    rc = FAILURE_;
	} else {
	    bytesInChar = newBytesInChar;
	    rc = SUCCESS_;
	}

	return rc;
}

Boolean_
DataCharacteristics::setBytesInShort( int newBytesInShort )
{
	/*
	 * The SDDF format currently doesn't support shorts so we
	 * don't worry about them.  Any size is fine for now.
	 */

	bytesInShort = newBytesInShort;
	return SUCCESS_;
}

Boolean_
DataCharacteristics::setBytesInInteger( int newBytesInInteger )
{
	Boolean_ rc;

	if ( NativeBytesInInteger != newBytesInInteger ) {
	    error( ConversionError, "integers of different sizes" );
	    rc = FAILURE_;
	} else {
	    bytesInInteger = newBytesInInteger;
	    rc = SUCCESS_;
	}

	return rc;
}

Boolean_
DataCharacteristics::setBytesInLong( int newBytesInLong )
{
	/*
	 * The SDDF format currently doesn't support longs so we
	 * don't worry about them.  Any size is fine for now.
	 */

	bytesInLong = newBytesInLong;
	return SUCCESS_;
}

Boolean_
DataCharacteristics::setBytesInFloat( int newBytesInFloat )
{
	Boolean_ rc;

	if ( NativeBytesInFloat != newBytesInFloat ) {
	    error( ConversionError, "floats of different sizes" );
	    rc = FAILURE_;
	} else {
	    bytesInFloat = newBytesInFloat;
	    rc = SUCCESS_;
	}

	return rc;
}

Boolean_
DataCharacteristics::setBytesInDouble( int newBytesInDouble )
{
	Boolean_ rc;

	if ( NativeBytesInDouble != newBytesInDouble ) {
	    error( ConversionError, "doubles of different sizes" );
	    rc = FAILURE_;
	} else {
	    bytesInDouble = newBytesInDouble;
	    rc = SUCCESS_;
	}

	return rc;
}

Boolean_
DataCharacteristics::setBytesInLongDouble( int newBytesInLongDouble )
{
	/*
	 * The SDDF format currently doesn't support long doubles so we
	 * don't worry about them.  Any size is fine for now.
	 */

	bytesInLongDouble = newBytesInLongDouble;
	return SUCCESS_;
}


/*
 * Initialize the static data structures
 */

const char *const DataCharacteristics::MY_CLASS = "DataCharacteristics";

#if defined LITTLE__ENDIAN
Sddf_BYTE_ORDERING DataCharacteristics::NativeByteOrdering = Sddf_LITTLE_ENDIAN;
#elif defined BIG__ENDIAN 
Sddf_BYTE_ORDERING DataCharacteristics::NativeByteOrdering = Sddf_BIG_ENDIAN;
#endif

Sddf_INT_REPRESENT DataCharacteristics::NativeIntegerRepresentation = 
							Sddf_TWOS_COMPLEMENT;

Sddf_SNGLP_REPRESENT DataCharacteristics::NativeSingleFloatRepresentation =
							Sddf_IEEE_S;

Sddf_DBLP_REPRESENT DataCharacteristics::NativeDoubleFloatRepresentation =
							Sddf_IEEE_D;

Sddf_EXTRA_REPRESENT DataCharacteristics::NativeExtraFloatRepresentation =
							Sddf_IEEE_E;

Sddf_CHAR_REPRESENT DataCharacteristics::NativeCharRepresentation =
							Sddf_ASCII;

int DataCharacteristics::NativeBytesInChar = sizeof(char);
int DataCharacteristics::NativeBytesInShort = sizeof(short);
int DataCharacteristics::NativeBytesInInteger = sizeof(int);
int DataCharacteristics::NativeBytesInLong = sizeof(long);
int DataCharacteristics::NativeBytesInFloat = sizeof(float);
int DataCharacteristics::NativeBytesInDouble = sizeof(double);
int DataCharacteristics::NativeBytesInLongDouble = sizeof(long double);
