/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987, 1988, 1989, 1990, 1991, 1992
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */
/*
 * Val.h: Implements a parameter for a sonic message.
 *
 */

#pragma once
#ifndef Val_h
#define Val_h


#include "SoundObj.h"
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <strings.h>

#include "Data.h"
#include "DataTransform.h"
#include "PtrList.h"
#include "DataType.h"
#include "CharType.h"
#include "IntType.h"
#include "EnumVal.h"
#include "EnumType.h"
#include "FloatType.h"
#include "DoubleType.h"
#include "BoolType.h"
#include "ArrayType.h"
#include "RemoteFileType.h"

class Val : public Data {
/*
 * Val is the basic type for all values that describe sonic message 
 * parameters.
 *
 */

private:

char *ValTypeName[] = {
  "ARRAY",
  "BOOLEAN",
  "CHAR",
  "DOUBLE",
  "ENUMTYPE",
  "ENUMVAL",
  "FLOAT",
  "INTEGER",
  "LOCALFILE",
  "REMOTEFILE",
  "UNKNOWN"
  } ;

ValType vtype; // default type
char *vname = NULL;

DataTransform *DTF;

DataType *val;
  
int Kind = 0;
int Index = 0;

protected:

public:

  /* Method Val: The Val constructor */
  Val() { val = new DataType; 
	  vtype = UNKNOWN; 
	  DTF = new DataTransform; } ;
  Val(char c)  { vtype = CHAR; val = new CharType(c);
		 DTF = new DataTransform; } ;
  Val(int i)  { vtype = INTEGER; val = new IntType(i); 
		DTF = new DataTransform; } ;
  Val(float f) { vtype = FLOAT; val = new FloatType(f); 
		 DTF = new DataTransform; } ;
  Val(double d) { vtype = DOUBLE; val = new DoubleType(d); 
		  DTF = new DataTransform; } ;
  Val(char* a)  { vtype = ARRAY; val = new ArrayType(a);
		  DTF = new DataTransform; } ;		  

  /* Method ~Val: The Val destructor */
  ~Val() ;

  /* Method type: Return the type of the Val */
  ValType type() { return vtype; } ;

  /* Method kind: Return the kind of Val (configurable, fixed, variable) */
  int kind(int i) { return(Kind = i); } ;
  int kind()  { return Kind; } ;

  /* Method index: Return index of Val */
  int index(int i) { return (Index=i); } ;
  int index() { return Index; } ;

  /* Method InitFromString: Initialize Val from string, as in Default  *
   *                            Parameters file                        */
  void InitFromString(char *str);

 /* Method InitEnumFromFile: Initialize Val as an an enumerated type from
  *                         a file */
  void InitEnumFromFile(char *filename) {
    delete val; val = new EnumType(filename); vtype = ENUMTYPE;
  } ;

  /* Method SetValueFromString: Set the actual value of Val from an an *
   *                            ascii string                           */
  void SetValueFromString(char *s);

  /* Method SetTransformFunction: Set transform function for variable Vals */
  void SetTransformFunction(DataTransform* dtf);

  /* Method GetTransformFunction: Return the transform function */
  DataTransform* GetTransformFunction() const { return DTF; } ;

  /* Method stringToType: Determine type from its name */
  int stringToType(char *s);

  /* Method retname: Return the name of the Val */
  char* retname() { return vname; } ;

  /* Method printVal: Return a char string with the value of Val */
  char* printVal();

  /* Method EnumAsString: Return the name of an enumerated type index */
  char* EnumAsString(const int i) const;

  /* Method name: Set the name of the Val */
  void name(const char *n) { setValName(n); };
  
  /* Method setValName: Set the name of the Val */
  void setValName(const char *n);

  /* Method isbounded: Determine whether Val is bounded in Device 
    Parameters file  */
  Bool_ isbounded() { return val->isBounded(); } ;


  /* Method getbounds: Retrieve the upper and lower bounds for a Val */
  Bool_ getbounds(int &l, int &u) {
    if (! val->isBounded() ) { return False_; } 
    else { l = (int) val->getLowerBound(); 
	   u = (int) val->getUpperBound(); 
	   return True_; 
	 }
  } ;

  /* Method printOn: Stream output function */
  void printOn(ostream &os = cout);

  /* Method recalcluateValue: Recalcluate value of Val using TF */
  void recalculateValue(double dd);

  /* Method binSend: Write the binary representation of Val to sfd */
  void binSend(int sfd);

  void binSend(int sfd, double dd) { 
    recalculateValue(dd);
    binSend(sfd); 
  } ;

  /* Method binSendValues: Write the binary representation of the Val *
   * value to sfd */

  void binSendValues(int sfd);
  void binSendValues(int sfd, double dd) { 
    recalculateValue(dd);
    binSendValues(sfd);
  } ;

  /* Method binReceive: Read the binary representation of Val from sfd */
  void binReceive(int sfd);

  /* Method binReceiveValues: Read the binary representation of Val value *
   *  from sfd */
  void binReceiveValues(int sfd);

  /* Method  SetValueFromDouble: Set the Val value from double value d */
  void SetValueFromDouble(double d);

  /* Method  int: Return Val as int */
  operator int() { return (int) *val; } ;

  /* Method  char: Return Val as char */
  operator char() { return (char) *val; };

  /* Method  float: Return Val as float */
  operator float() { return (float) *val; } ;

  /* Method  double: Return Val as double */
  operator double() { return (double) *val; } ;

  /* Method  char*: Return Val as character string */
  operator char*() { return (char *) *val; } ;

 /* Method operator=: Assignment operator */
  Val& operator= ( const Val& v );

};

inline ostream& operator<<(ostream &os, Val &obj)
{
	obj.printOn(os);
	return os;
}

#endif
