/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987, 1988, 1989, 1990, 1991, 1992
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */

/*
 * SoundControlFile.h: Implements a sound control file.
 *
 */


#pragma once
#ifndef SoundControlFile_h
#define SoundControlFile_h

#include "SoundObj.h"
#include "ValList.h"
#include "MessageList.h"
#include "SoundDevice.h"
#include "TransformFunctions.h"
#include "WclParams.h"

#include "SCFParsing.h"

extern UserInterface UI;

class SoundControlFile : public SCFParsing {
/*
 * SoundControlFile implements a sound control file, a building block
 * for sonic widgets.
 */

private:
  MessageList  SCFMessages, *Messages;
  SoundDevice  SD;
  ValList *DefaultsTable, *SystemParameters;

  const char* NullStr = "";

  int LargestPossibleArray = 20;
  int HighestArrayIndex = 0;

  ValList *VariableVals;


  /* Method initSoundDevice: Initialize sound device */
  Bool_ initSoundDevice(istream &is, const char *hostname, 
			const char *portname);

  /* Method readMessages: Read messages from input stream */
  Bool_ readMessages(istream &is);

  /* Method configureTransformFunction: Configure transform functions from *
   * input stream */
  Bool_ configureTransformFunction(Val &v, istream &is);

  /* Method registerVariableVal: Make note of variable Vals */
  Bool_ registerVariableVal(Val *v, int arrayIndex);


  TransformFunctions myTransformFunctions;
  Bool_ Acknowledge;

  char host[128], port[128];
  
public:
	
  /* Method SoundControlFile: The SoundControlFile constructor. */
  SoundControlFile() { } ;

  /* Method ~SoundControlFile: The SoundControlFile destructor. */
  ~SoundControlFile();

  /* Method setBlocking: Determine whether we do blocking or not */
  void setBlocking(Bool_ ack) { 
				SD.setAcknowledge(ack); 
			      } ;

  /* Method initialize: Initialize an SCF from an input stream  or from *
   *                    a file                                          */
  Bool_ initialize(istream &from, const char *hostname = "", 
		   const char *portname = "");

  Bool_ initialize(char *scf, const char *hostname = "",
		   const char *portname = "");


  /* Method readSoundDevice: Read the sound device from an input stream */
  Bool_ readSoundDevice(istream &is, char *hostname, char *portname);


  /* Method writeSCF: Write the sound control file back to an output stream */
  void writeSCF(ostream &os) const;

  /* Method execute: Execute the SCF */
  void execute(double d);
  void execute(double *a, int length);

  /* Method dumpParamsResourceFile: Write out a resource file */
  Bool_ dumpParamsResourceFile(char *filename, char *widgetName);

  /* Method setDefault: Set a default in the SCF */
  void setDefault(int msgnum, int valnum, char *value);
  void setDefault(int msgnum, int valnum, double d);

  /* Method getDefault: Retrieve a default from a Val in the SCF */
  Val& getDefault(int msgnum, int valnum);
  
  /* Method getTransformFunction: Get a TF from a Val in the SCF */
  DataTransform* getTransformFunction(int msgnum, int valnum);


  /* Method setTransformFunction: Set the TF for a Val in the SCF */
  void setTransformFunction(int msgnum, int valnum, DataTransform *tf);

  /* Method printOn: Stream output function */
  void printOn(ostream &os = cout);
};

inline ostream &operator<<(ostream &os, SoundControlFile &obj)
{
	obj.printOn(os);
	return os;
}

#endif
