/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987, 1988, 1989, 1990, 1991, 1992
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */
/*
 * BaseSonicWidget.h: The base class for Sonic Widgets.
 *
 */

#pragma once
#ifndef BaseSonicWidget_h
#define BaseSonicWidget_h

#include "SCFParsing.h"
#include "SoundControlFile.h"
#include "Val.h"

class BaseSonicWidget : public SCFParsing {
/*
 * BaseSonicWidget is the base class for all Sonic Widgets. It does nothing
 * except define the virtual execute procedures which all derived Sonic
 * widgets must support.
 */


protected:
  char *myWidgetName = NULL;
  DataTransform *TF = NULL;
  PtrList SCFlist;
  
  Val *transformVal;

  /* Method looksValid:   Check that a sound control file ends in .scf. */
  Bool_ looksValid( char *soundControlFile );



public:
  /* Method BaseSonicWidget: The BaseSonicWidget constructor. */
  BaseSonicWidget(char *name = "");

  /* Method ~BaseSonicWidget: The BaseSonicWidget destructor. */
  ~BaseSonicWidget(); 

  /* Method configure: Called when Sonic Widget is configured if
                       any special actions are required.       */
  virtual void configure() { } ;

  /* Method execute: Execute the sonic widget with scalar or vector. */
  virtual void  execute(double d) = 0 ;
  virtual void  execute(double *a, int length) = 0 ;

  
  /* Method getHeader: Get the header of a sound control file. */
  virtual Bool_ getHeader(char *soundControlFile, char *hostname, 
			  char *portname);


  /* Method getName: Return the name of the sonic widget. */
  virtual char* getName() const { return myWidgetName; } ;
  
  /* Method getSCF:  Return one of the constituant sound control files. */
  virtual SoundControlFile* getSCF(int which) { 
    return ((SoundControlFile *)SCFlist.getElement(which));
  } ;

  /* Method myClassName: Return the name of the widget class. */
  virtual const char* myClassName() const { return("BaseSonicWidget"); } ;

  /* Method numSCFs: Return the number of sound control files managed. */
  virtual int numSCFs() const { return(SCFlist.numberArgs()) ; } ;


  /* Method setBlocking: Turn blocking on or off. */
  virtual void setBlocking(int ack);

  /* Method writeParamsResourceFile: Write WCL resource file for SCF child, *
   *                                 returning filename.                     */
  virtual char* writeParamsResourceFile(char *directory, int which = 0 );

  /* Method writeWCF: Write out widget control file. */
  virtual void writeWCF(ostream &os) const;

  /* Method writeSCF: Write out a specific sound control file. */
  virtual void writeSCF(ostream &os, int which = 0 ) const;

  /* Method getTransformFunction: Return the transform function for this *
   *                              sonic widget.                          */
  virtual DataTransform* getTransformFunction() { return TF; } ;

  /* Method setTransformFunction: Set the transform function for this  *
   *                              sonic widget.                         */
  virtual void setTransformFunction(DataTransform *tf) { 
    TF = tf;
    transformVal->SetTransformFunction(tf);
  } ;

  /* Method initialize: Initialize the sonic widget from a widget control   *
   *                    file that has been positioned past the sonic widget *
   *                    class identifier.                                   */
  virtual Bool_ initialize(istream &is) = 0;

  /* Method initialize: Initialize the sonic widget from sound control  *
   *                                  file(s).                           */
  virtual   Bool_ initialize(SoundControlFile *widgetControlFile ...);
  virtual   Bool_ initialize(SoundControlFile *widgetControlFile);
  

  
  /* Method printOn: Stream output function */
  virtual void printOn(ostream &os = cout);

} ;

inline ostream &operator<<(ostream &os, BaseSonicWidget &obj)
{
	obj.printOn(os);
	return os;
}

#endif
