/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987, 1988, 1989, 1990, 1991, 1992
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */

#include "Val.h"


Val::~Val()
{
  delete vname;
  delete DTF;
  delete val;
}

Val& Val::operator= ( const Val& v )
{
  vtype = v.vtype;
  Kind = v.Kind;
  Index = v.Index;
  vname = duplicateString(v.vname);
  delete val;
  switch(vtype) {
  case CHAR:
    val = new CharType;
   *((CharType *)val) = *((CharType *)v.val);
    break;
  case BOOLEAN:
    val = new BoolType;
    *((BoolType *)val) = *((BoolType *)v.val);
    break;
  case ARRAY:
    val = new ArrayType;
    *((ArrayType *)val) = *((ArrayType *)v.val);
    break;
  case DOUBLE:
    val = new DoubleType;
    *((DoubleType *)val) =*((DoubleType *)v.val);
    break;
  case FLOAT:
    val = new FloatType;
    *((FloatType *)val) = *((FloatType *)v.val);
    break;
  case ENUMTYPE:
    val = new EnumType;
    *((EnumType *)val) = *((EnumType *)v.val);
    break;
  case ENUMVAL:
    val = new EnumVal;
    *((EnumVal *) val) = *((EnumVal *) v.val);
    break;
  case INTEGER:
    val = new IntType;
    *((IntType *) val) = *((IntType *) v.val);
    break;
  case LOCALFILE:
    val = new DataType;
    cerr << "Error: Val: Localfile not supported\n";
    break;
  case REMOTEFILE:
    val = new RemoteFileType;
    *((RemoteFileType *) val) = *((RemoteFileType *) v.val);
    break;
  case UNKNOWN:
    val = new DataType;
    break;
  default:
    cerr << "Error: Val::operator=(): got strange type " << vtype << NL;
    break;
  }
  return *this;
}

char* Val::EnumAsString(const int i) const { 
  return(val->enumAsString(i));
}


void Val::recalculateValue(double dd) 
{
  if (DTF != NULL) {
    DTF->calculateValue((void *)&dd, (void *)this); 
  }
}

void Val::SetValueFromString(char *s)
{
  if (vtype == UNKNOWN) {
    cerr << "Error: Val::setValueFromString: Attempt to set value when type is unknown\n";
  } else {
    val->setValueFromString(s);
  }
}

int Val::stringToType(char *s)
{
  int hi = NUMTYPES;
  int lo = 0;
  int i=0, n=0, cmp=0;

  while(True_) {
    n = (hi+lo)/2;
    if (i == n) {return (-1);}
    i = n;
    cmp = strcmp(s, ValTypeName[i]);
    if (cmp == 0) return(i);
    if (cmp < 0) {
      hi = i;
    } else { // cmp > 0
      lo = i;
    }
    continue;
  }
}

void Val::InitFromString(char *str)
{
  char *handle, *s;
  int l = -1;
  int u = -1;
  char n[256], a[256], tmp[256];
  int i;
  float f;
  double d;

  strcpy(tmp, str);

  handle = strtok(str, " \t"); // this is thrown away for now
  if ((vtype = stringToType(s = strtok(NULL, " \t"))) <  0) { 
    cerr << form("Error: Val::InitFromString: Unrecognized type %s\n", s);
    return;
  }
  s = strtok(NULL, "\n");
  delete val;
  switch(vtype) {
  case LOCALFILE:
    cerr << "Warning: Val: localfile not supported yet\n";
    break;
  case CHAR:
    sscanf(s, " \"%[^\"]\" %s %d %d", n, a,  &l, &u);
      val = new CharType((char) atoi(a));
    break;
  case BOOLEAN:
    sscanf(s, " \"%[^\"]\" %i %d %d",  n, &i,  &l, &u);
    if (i != 1 && i != 0) { 
      cerr << form("Warning: Val::initFromString: Non-Boolean value %d\n", i);
    }
    val = new BoolType(i);
    break;
  case REMOTEFILE:
    sscanf(s, " \"%[^\"]\" %i %d %d",  n, &i,  &l, &u);
    val = new RemoteFileType(i);
    break;
  case ENUMVAL:
    sscanf(s, " \"%[^\"]\" %i %d %d",  n, &i,  &l, &u);
    val = new EnumVal(i);
    break;
  case INTEGER:
    sscanf(s, " \"%[^\"]\" %i %d %d",  n, &i,  &l, &u);
    val = new IntType(i);
    break;
  case FLOAT:
    sscanf(s, " \"%[^\"]\" %f %d %d",  n, &f,  &l, &u);
    val = new FloatType(f);
    break;
  case DOUBLE:
    sscanf(s, " \"%[^\"]\" %lf %d %d",  n, &d,  &l, &u);
    val = new DoubleType(d);
    break;
  case ARRAY:
    sscanf(s, " \"%[^\"]\" \"%[^\"]\"  %d %d",  n, a,  &l, &u);
    val = new ArrayType(a);
    break;
  case ENUMTYPE:
    sscanf(s, " \"%[^\"]\" \"%[^\"]\"",  n, a);
    val = new EnumType(a);
    break;
  default:
    cerr << form("Error Val::initFromString: unknown type %d\n", vtype);;
    break;
  }

  setValName(n);

  if (l >= 0) {
    val->setBounds((char)l, (char)u);
  }
}

void Val::setValName(const char *n) 
{
  delete vname;
  vname = duplicateString(n);
}

void Val::binSend(int sfd)
{
  int l, u, b, i;
  struct stat *buf;
  char buffer[256];
  int bytes;

  fd = sfd;
  int lfd;

  if (vtype == LOCALFILE) {
    cerr << "Error: localfile not supported yet.\n";
    return;
  }
  if (vtype == UNKNOWN) {
    cerr << form("Error in Val::binSend, fd = %d: unknown Val type %d\n", fd, vtype);
    return;
  }

  send(vname);        // name 
  send((char) vtype); // type

  val->binSend(sfd);

  send((char) (val->isBounded()));
  if (val->isBounded() ) {
    send( val->getLowerBound() );
    send( val->getUpperBound() );
  } 
}

void Val::binSendValues(int sfd)
{
  int l, u, b, i;
  struct stat *buf;
  char buffer[256];
  int bytes, lfd;

  fd = sfd;
  
  if (vtype == LOCALFILE) {
    cerr << "Error: Val::binSend: localfile not yet supported\n";
    return;
  }
  
  if (vtype == UNKNOWN) {
    cerr << "Error: Val::binSend: attempt to send value of unknown type\n";
    return;
  }

  send((char) vtype); // type
  val->binSend(sfd);
}


void Val::binReceive(int sfd)
{
  char c, l, u, *s, *n;
  Bool_ bounded;
  int i;
  fd = sfd;
  n = recString(); // name
  receive(c);   // type

  if (vtype != UNKNOWN) {
    cerr << form("Warning: Val::binReceive: receiving a known type; deleting old val of type %s\n", ValTypeName[vtype]);
  }
  delete val;

  vtype = (int) c;

  switch(vtype) {
  case LOCALFILE:
    cerr << "Error: Val::binReceive: localfile not yet supported\n";
    break;
  case CHAR:
    val = new CharType;
    break;
  case BOOLEAN:
    val = new BoolType;
    break;
  case ENUMVAL:
    val = new EnumVal;
    break;
  case REMOTEFILE:
    val = new RemoteFileType;
    break;
  case INTEGER:
    val = new IntType;
    break;
  case FLOAT:
    val = new FloatType;
    break;
  case DOUBLE:
    val = new DoubleType;
    break;
  case ARRAY:
    val = new ArrayType;
    break;
  case ENUMTYPE:
    val = new EnumType;
    break;
  default:
    cerr << form("Error in Val::binReceive, fd = %d: unknown Val type %d\n", fd, (int)c);
    break;
  }
  val->binReceive(sfd);

  name(n);
  receive(c);
  bounded = (int) c;
  if (bounded) {
    receive(l); receive(u);
    val->setBounds(l, u);
  }
  delete n;
}

void Val::binReceiveValues(int sfd)
{
  char c, l, u, *s, *n;
  int i;
  fd = sfd;
  receive(c);   // type
  delete val;

  vtype = (int) c;

  switch(vtype) {

  case LOCALFILE:
    cerr << "Error: Val::binReceiveValues: localfile not yet supported\n";
    break;
  case CHAR:
    val = new CharType;
    break;
  case ENUMVAL:
    val = new EnumVal;
    break;
  case BOOLEAN:
    val = new BoolType;
    break;
  case REMOTEFILE:
    val = new RemoteFileType;
    break;
  case INTEGER:
    val = new IntType;
    break;
  case FLOAT:
    val = new FloatType;
    break;
  case DOUBLE:
    val = new DoubleType;
    break;
  case ARRAY:
    val = new ArrayType;
    break;
  case ENUMTYPE:
    cerr << form("Error in Val::binReceiveValues, fd = %d: attempt to receive values of an ENUMTYPE\n", fd);
    return;
    break;
  default:
    cerr << form("Error in Val::binReceiveValues, fd = %d: unknown Val type %d\n", fd, (int)c);
    return;
  }
  val->binReceive(fd);
}


char* Val::printVal()
{

 if (vtype == LOCALFILE) {
   cerr << "Error: Val::printVal: localfile not yet supported\n";
   return( NULL );
 }

 return( val->print() );

}

void Val::printOn(ostream &os)
{
  os << form("%s\t\"%s\"\t", ValTypeName[vtype], vname);
  os << *val;

  if (val->isBounded()) {
    os << form("\t%d\t%d", (int) val->getLowerBound(), 
	       (int) val->getUpperBound());
  }
}

void Val::SetValueFromDouble(double d)
{
  val->setValueFromDouble(d);
}
void Val::SetTransformFunction(DataTransform *dtf)
{
  delete DTF;
  DTF = dtf;
}
