/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987, 1988, 1989, 1990, 1991, 1992
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */

#include "SoundControlFile.h"


Bool_ SoundControlFile::readSoundDevice(istream &is, char *hostname, 
					char *portname)
{

  if (getTok(is, host) != FIXED || getTok(is, port) != FIXED) {
    return(False_);
  } else {
    strcpy(hostname, host);
    strcpy(portname, port);
    return (True_);
  }
}


Bool_ SoundControlFile::initialize(char *scf, const char *hostname, 
				   const char *portname)
{
  filebuf fb;
  Bool_ status;
  
  if (fb.open(scf, input) == 0) {
    cerr << form("Error: SoundControlFile::initialize(): filename = %s,\
                  %s\n", scf, sys_errlist[errno]);
    return(False_);
  }
  istream from(&fb);
  status = initialize(from, hostname, portname);
  fb.close();
  return(status);
}
  

Bool_ SoundControlFile::initialize(istream &from, const char *hostname, 
				   const char *portname)
{
  VariableVals = new ValList[LargestPossibleArray];

  
  if (initSoundDevice(from, hostname, portname) == False_) { 
    return(False_); 
  }
  if (readMessages(from) == False_) { 
    return(False_); 
  }
  return(True_);
}

SoundControlFile::~SoundControlFile()
{
// do something reasonable here.
}

Bool_ SoundControlFile::registerVariableVal(Val* v, int arrayIndex)
{
  if (arrayIndex >= LargestPossibleArray) {
    ValList *tmp = new ValList[arrayIndex * 2];
    for(int i=0; i < LargestPossibleArray; i++) {
      tmp[i] = VariableVals[i];
    }
    delete [LargestPossibleArray] VariableVals;
    LargestPossibleArray = arrayIndex * 2;
    VariableVals = tmp;
  }
  VariableVals[arrayIndex].addArg(v);
  if (arrayIndex > HighestArrayIndex) {
    HighestArrayIndex = arrayIndex;
  }
  return(True_);
}

Bool_ SoundControlFile::initSoundDevice(istream &is, const char *hostname, 
					const char *portname)
{
  
  getTok(is, host);
  getTok(is, port);

  /* override the file defaults if others are specified */
  if (strlen(hostname) > 0) {
    strcpy(host, hostname);
  }
  if (strlen(portname) > 0) {
    strcpy(port, portname);
  }

  if (SD.openConnection(host, port) == False_) {
    return(False_);
  } else {
    DefaultsTable = SD.defaults();
    SystemParameters = SD.sysparams();
    Messages = SD.messages();
  }
  return(True_);
}
    

Bool_ SoundControlFile::readMessages(istream &is)
{
  int msgnum, nargs;
  Message *msg;
  Val *v;
  char token[MAXID];
  int tmp;

  Tok currentTok;

  while((currentTok = peekTok(is)) != SECTIONMARKER && currentTok != IOERR ) {
    currentTok = getTok(is, msgnum);
    msg = new Message;
    *msg = Messages->getArg(msgnum);
      msg->setNumber(msgnum);

      IndexList& ilist = (Messages->getArg(msgnum)).getIndexList();
      nargs = ilist.numberArgs();
    
      for(int i=0; i < nargs; i++) {
	v = new Val;
	int foobar = ilist.getArg(i);
	*v = DefaultsTable->getArg(foobar);
	switch(getTok(is, token)) {
	case VARIABLE:
	  tmp = atoi(token);
	  v->index(tmp);
	  if (registerVariableVal(v, tmp) == False_) {
	    cerr << "Error: SoundControlFile::getMessages(): Unable to\
 register value dependent on data index " << tmp << NL;
	  }
	  v->kind(VARIABLE);
	  configureTransformFunction(*v, is);
	  break;
	case FIXED:
	  // value is already in token
	  v->SetValueFromString(token);
	  v->kind(FIXED);
	  break;
	case CONFIGURABLE:
	  v->SetValueFromString(token);
	  v->kind(CONFIGURABLE);
	  break;
	case IOERR:
	  cerr << "Error: SoundControlFile: Unexpected error reading control\
 file\n";
	  return(False_);
	  break;
	default:
	  cerr << "Error: SoundControlFile: Unexpected token type (probably\
 a parse error)\n";
	  return(False_);
	  break;
	}
	msg->addArgToValList(*v); 
	//msg takes its arguments with it when it goes.
      }
      SCFMessages.addArg(msg);
  }
  return(True_);
}

void SoundControlFile::writeSCF(ostream &os) const
{
  int n, i, j;
  char *tmp;

  os << host << " " << port << NL;
  n = SCFMessages.numberArgs();
  for(i=0; i < n; i++) {
    Message &m = SCFMessages.getArg(i);
    os << form("/* %s */ %d", m.getName(), m.getNumber());
    ValList &vl = m.getValList();
    for(j=0; j < vl.numberArgs(); j++) {
      Val &v = vl.getArg(j);
      switch(v.kind()) {
      case CONFIGURABLE:
	os << "\t" << CONFIG << v.printVal();
	break;
      case VARIABLE:
	os << "\t" << VAR << v.index() << " ( " << (v.GetTransformFunction())->printConfiguration("") << " ) ";
	break;
      case FIXED:
	os << "\t" << v.printVal();
	break;
      default:
	cerr << "Error: SoundControlFile::writeSCF(): unrecognized type of Val\n";
	break;
      }
    }
    os << NL;
  }
}

Bool_ SoundControlFile::dumpParamsResourceFile(char *filename,
						  char *widgetName)
{
  filebuf fb;
  if (fb.open(filename, output) == 0) {
   cerr << form("Error: SoundControlFile::dumpParamsResourceFile: filename = %s, %s\n", filename, sys_errlist[errno]);
   return(False_);
 }

  ostream to(&fb);

  WclParams myParams(widgetName, DefaultsTable, SystemParameters);
  myParams.dumpWclResourceFile(to, SCFMessages);
  to.close();
  fb.close();
  return(True_);
}


void SoundControlFile::execute(double d)
{
  int i, numevents = SCFMessages.numberArgs();

// here we have to recalculate everything  
// if argument were an array we would have to step through it.
  if (VariableVals[0].numberArgs() > 0) {
    ValList &vl = VariableVals[0];
    for(i=0; i < vl.numberArgs(); i++) {
      Val &v = vl.getArg(i);
      v.recalculateValue(d);
    }
  }

  SD.executeMessage(SCFMessages);
}


void SoundControlFile::execute(double *a, int length)
{
  int i,j, nx, numevents = SCFMessages.numberArgs();

  for(i=0; i <= HighestArrayIndex; i++) {
    if (VariableVals[i].numberArgs() > 0) {
      ValList &vl = VariableVals[i]; // the variable vals are kept in lists
                                     // according to the array indexed value
                                     // they are dependent upon.


      for(j=0; j < vl.numberArgs(); j++) { // for each variable val 
	                                   // corresponding to the index
                                           // we are looking at
	Val &v = vl.getArg(j);
	if (i < length) {                  // if the array is long enough
	                                   // to accomodate array index i
                                           // which was specified in the SCF


	  v.recalculateValue((double) a[i]);
	}
      }
    }
  }

  for(i=0; i < numevents; i++) {
    SD.executeMessage(SCFMessages.getArg(i));
  }
}

void SoundControlFile::setTransformFunction(int msgnum, int valnum, 
					    DataTransform *tf)
{
  Val &v = getDefault(msgnum, valnum);
  v.SetTransformFunction(tf);
}

DataTransform* SoundControlFile::getTransformFunction(int msgnum, int valnum)
{
  Val &v = getDefault(msgnum, valnum);
  return(v.GetTransformFunction());
}

void SoundControlFile::setDefault(int msgnum, int valnum, char *value)
{
  Val &v = getDefault(msgnum, valnum);
  v.SetValueFromString(value);
}

void SoundControlFile::setDefault(int msgnum, int valnum, double d)
{
  Val &v = getDefault(msgnum, valnum);
  v.SetValueFromDouble(d);
}

Val& SoundControlFile::getDefault(int msgnum, int valnum)
{
  if (msgnum < 0 || msgnum >= SCFMessages.numberArgs() ) {
    cerr << "Error: SoundControlFile::getDefault: invalid message number " << msgnum <<NL;
    return NULL;
  }
  ValList &vl = (SCFMessages.getArg(msgnum)).getValList();
  if (valnum < 0 || valnum >= vl.numberArgs()) {
    cerr << "Error: SoundControlFile::getDefault: invalid value number " << valnum <<NL;
    return NULL;
  }
  return(vl.getArg(valnum));
}


void SoundControlFile::printOn(ostream &os)
{
	os << "SoundControlFile\n";
}


Bool_ SoundControlFile::configureTransformFunction(Val &v, istream &is)
{
  Tok currentTok;
  int msgnum, valnum, tmp;
  char foo[MAXID];
  int l;

  if ((currentTok = peekTok(is)) != LPAREN) {
    return(False_);
    } else {
    currentTok = getTok(is, foo);
  }
  if (v.kind() != VARIABLE) {
    cerr << form("Warning: SoundControlFile::readConfiguration(): %s is not variable\n", v.retname());
  } else {
    v.SetTransformFunction(myTransformFunctions.readConfiguration(is));
  }
  if ((currentTok = getTok(is, foo)) != RPAREN) {
    cerr << "Error: SoundControlFile::configureTransformFunction: expected\
 right paren, found: " << foo << NL;
    return(False_);
  }
  return(True_);
}

