/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author:  Bradley Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include <values.h>
#include <math.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include "XYGraphP.h"

#define REALLOC_INTERVAL   10

#define fontheight(f)	(f->max_bounds.ascent+f->max_bounds.descent+2*FONTPAD)
#define offset(field)	XtOffset(XYGraphWidget,field)

#define	ScaleX(x)	(floor(FactorX*((x) - w->xygraph.xmin) + 0.5)) + \
                             w->xygraph.offsetx
#define	ScaleY(y)	(w->xygraph.sizey - \
                             floor(FactorY*((y) - w->xygraph.ymin) + 0.5)) \
				+ w->xygraph.offsety1

#define abs(A)		((A)>=0.0?(A):-(A))
#define Min(a,b)        ( (a) < (b) ? (a) : (b) )
#define Max(a,b)        ( (a) > (b) ? (a) : (b) )


static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     offset(core.height), XtRString, "200"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     offset(core.background_pixel), XtRString, "white"},
  /* xygraph resources */
  {XtNtitle, XtCTitle, XtRString, sizeof(char *),
     offset(xygraph.title), XtRString, NULL},
  {XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
     offset(xygraph.grid_color), XtRString, "black"},
  {XtNbaseColors, XtCBaseColors, XtRPointer, sizeof(Pixel *),
     offset(xygraph.baseColors), XtRString, NULL},
  {XtNcolors, XtCColors, XtRPointer, sizeof(Pixel *),
     offset(xygraph.colors), XtRString, NULL},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     offset(xygraph.numcolors), XtRInt, "1"},
  {XtNyticks, XtCYticks, XtRInt, sizeof(int), 
     offset(xygraph.yticks), XtRString, "10" },
  {XtNxticks, XtCXticks, XtRInt, sizeof(int), 
     offset(xygraph.xticks), XtRString, "10" },
  {XtNlogx, XtCLogx, XtRBoolean, sizeof(Boolean),
     offset(xygraph.logx), XtRString, "FALSE"},
  {XtNlogy, XtCLogy, XtRBoolean, sizeof(Boolean),
     offset(xygraph.logy), XtRString, "FALSE"},
  {XtNautoscale, XtCAutoscale, XtRBoolean, sizeof(Boolean),
     offset(xygraph.autoscale), XtRString, "False"},
  {XtNdensityPlot, XtCDensityPlot, XtRBoolean, sizeof(Boolean),
     offset(xygraph.densityPlot), XtRString, "False"},
  {XtNminValues, XtCMinValues, XtRPointer, sizeof(int *),
     offset(xygraph.minValues), XtRString, NULL },
  {XtNmaxValues, XtCMaxValues, XtRPointer, sizeof(int *),
     offset(xygraph.maxValues), XtRString, NULL },
  {XtNminx, XtCMinx, XtRInt, sizeof(int),
     offset(xygraph.minx), XtRString, "0"},
  {XtNmaxx, XtCMaxx, XtRInt, sizeof(int),
     offset(xygraph.maxx), XtRString, "100"},
  {XtNminy, XtCMiny, XtRInt, sizeof(int),
     offset(xygraph.miny), XtRString, "0"},
  {XtNmaxy, XtCMaxy, XtRInt, sizeof(int),
     offset(xygraph.maxy), XtRString, "100"},
  {XtNconnect, XtCConnect, XtRBoolean, sizeof (Boolean),
     offset(xygraph.connect), XtRString, "False"},
  {XtNshowTicks, XtCShowTicks, XtRBoolean, sizeof (Boolean),
     offset(xygraph.show_ticks), XtRString, "FALSE"},
  {XtNshowRange, XtCShowRange, XtRBoolean, sizeof (Boolean),
     offset(xygraph.show_range), XtRString, "FALSE"},
  {XtNmarkerSize, XtCMarkerSize, XtRInt, sizeof(int),
     offset(xygraph.markersize), XtRString, "4" },
  {XtNxygraphCnt, XtCXYGraphCnt, XtRInt, sizeof(int),
     offset(xygraph.xygraph_cnt), XtRString, "1"},
  {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     offset(xygraph.font), XtRString, "fixed"},
  {XtNtextColor, XtCTextColor, XtRPixel, sizeof(Pixel),
     offset(xygraph.text_color), XtRString, "black"},
  {XtNxvalues, XtCXvalues, XtRPointer, sizeof(int *),
     offset(xygraph.xvalues), XtRString, NULL},
  {XtNyvalues, XtCYvalues, XtRPointer, sizeof(int *),
     offset(xygraph.yvalues), XtRString, NULL},
  {XtNdata, XtCData, XtRPointer, sizeof(int **),
     offset(xygraph.data), XtRString, NULL },
  {XtNdataCnt, XtCDataCnt, XtRInt, sizeof(int),
     offset(xygraph.pointCount), XtRString, NULL },
  {XtNdataSize, XtCDataSize, XtRInt, sizeof(int),
     offset(xygraph.datasize), XtRString, "0" },
  {XtNvalueCnt, XtCValueCnt, XtRPointer, sizeof(int *),
     offset(xygraph.valuecnt), XtRString, NULL},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     offset(xygraph.select), XtRCallback, (caddr_t) NULL},
};

#undef offset
#undef goffset

static void	ClassInitialize(), Initialize();
static void	Realize(), Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void     ReallocForValues(), ReallocForValue();
static void	CreateTextGC(), DestroyTextGC();
static void	CreateColorGCs(), CreateBrushColorGCs(), DestroyColorGCs();
static void	DrawGrid(), DrawTitle(), DrawLine(), DrawPoint();
static long     DrawDensityPoint();

static void	DrawXYGraph(), DrawDensityXYGraph(), DrawXYGraphRange();
static void	XYGraphSelect(), XYGraphNotify();


/************************/
/* xygraph translations */
/************************/
static char xygraph_translations[] = "	\
  <Btn1Down>:	XYGraphSelect()\n	\
  <Btn1Up>:	XYGraphNotify()		\
";

/*******************/
/* xygraph actions */
/*******************/
static XtActionsRec	xygraph_actions[] = {
  {"XYGraphSelect",	(XtActionProc) XYGraphSelect	},
  {"XYGraphNotify",	(XtActionProc) XYGraphNotify	},
};


XYGraphClassRec xygraphClassRec = {
  { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"XYGraph",
    /* widget_size		*/	sizeof(XYGraphRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	xygraph_actions,
    /* num_actions		*/	XtNumber(xygraph_actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	xygraph_translations,
    /* query_geometry           */	XtInheritQueryGeometry,
    /* display_accelerator      */	XtInheritDisplayAccelerator,
    /* extension                */	NULL
    }
};

WidgetClass xygraphWidgetClass = (WidgetClass) &xygraphClassRec;

static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/* ARGSUSED */
static void Initialize(request, new)
Widget request, new;
{
  int		i;
  XYGraphWidget	w = (XYGraphWidget)new;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize\n");
#endif
  if (w->core.width == 0)
    w->core.width = MINSIZE;
  if (w->core.height == 0)
    w->core.height = MINSIZE;

  if (w->xygraph.title != NULL)
    w->xygraph.title = strcpy(XtMalloc(strlen(w->xygraph.title) + 1),
			     w->xygraph.title);
  CreateTextGC(w);

  GCmask = GCForeground | GCBackground;
  GCvalues.foreground = w->xygraph.grid_color;
  w->xygraph.gridGC = XtGetGC((Widget)w, GCmask, &GCvalues);

  /* initialize value pointers */
  w->xygraph.valuecnt = NULL;

  w->xygraph.data = NULL;
  w->xygraph.xvalues = NULL;
  w->xygraph.yvalues = NULL;
  w->xygraph.point_coloring = NULL;
 
  w->xygraph.brushColors = NULL;
  w->xygraph.densityColors = NULL;
  w->xygraph.drawImage = NULL;
} /* Initialize */

/* ARGSUSED */
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  int			i;
  register XYGraphWidget	current = (XYGraphWidget) gcurrent;
  register XYGraphWidget	new = (XYGraphWidget) gnew;
  XYGraphWidget                 request = (XYGraphWidget) grequest;
  Boolean		redraw = FALSE, newdata = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  Boolean               realized = XtIsRealized(gcurrent);

  int                   **ourData, *ourMinValue, *ourMaxValue;
  static Pixel          *ourBaseColors;

#ifdef DEBUG
printf("SetValues\n");
#endif

  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    Resize(new);
    redraw = TRUE;
  }

  if ((new->xygraph.font != current->xygraph.font)
      || (new->xygraph.text_color != current->xygraph.text_color)
      || (new->core.background_pixel != current->core.background_pixel)) {
    DestroyTextGC(current);
    CreateTextGC(new);
    redraw = TRUE;
  }
  
  if (current->xygraph.title != new->xygraph.title) {
    new->xygraph.title =
      strcpy(XtMalloc((unsigned) strlen(new->xygraph.title) + 1),
	     new->xygraph.title);
    if (current->xygraph.title != NULL)	
      XtFree((char *)current->xygraph.title);
    redraw = TRUE;
  }

  /* generic base color set */
  if ( ourBaseColors != NULL ) {
     XtFree( ourBaseColors );
  }
  ourBaseColors = (Pixel *)XtMalloc( 2 * sizeof(Pixel) );
  ourBaseColors[1] = -1;
  new->xygraph.baseColors = ourBaseColors;

  /* change grid color */
  if ((new->xygraph.grid_color != current->xygraph.grid_color)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->xygraph.grid_color;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->xygraph.gridGC);
    new->xygraph.gridGC = XtGetGC(current, GCmask, &GCvalues);
    new->xygraph.baseColors[0] = new->xygraph.grid_color;
    redraw = TRUE;
  }

  /* change colors */
  if (new->xygraph.colors != current->xygraph.colors) {
     CreateColorGCs(new);
  }
  
  /* change number of yticks */
  if (new->xygraph.yticks != current->xygraph.yticks)
    if (new->xygraph.yticks > MAXROWS)
      new->xygraph.yticks = MAXROWS;

  /* change number of xticks */
  if (new->xygraph.xticks != current->xygraph.xticks)
    if (new->xygraph.xticks > MAXROWS)
      new->xygraph.xticks = MAXROWS;

  /* min/max change */
  if ( request->xygraph.minValues != NULL ) {
     ourMinValue = (int *)XtMalloc( 2 * sizeof(int) );
     ourMaxValue = (int *)XtMalloc( 2 * sizeof(int) );
     ourMinValue[0] = new->xygraph.minValues[0];
     ourMinValue[1] = new->xygraph.minValues[1];
     ourMaxValue[0] = new->xygraph.maxValues[0];
     ourMaxValue[1] = new->xygraph.maxValues[1];
     new->xygraph.minValues = ourMinValue;
     new->xygraph.maxValues = ourMaxValue;
     new->xygraph.minx = new->xygraph.minValues[0];
     new->xygraph.maxx = new->xygraph.maxValues[0];
     new->xygraph.miny = new->xygraph.minValues[1];
     new->xygraph.maxy = new->xygraph.maxValues[1];
     redraw = TRUE;
  }

  /* data */
  if ( ( current == NULL ) && ( new->xygraph.data != NULL ) ) {
     newdata = TRUE;
  } else if ( new->xygraph.data != current->xygraph.data ) {
     newdata = TRUE;
  }
  if ( ( newdata ) && ( new->xygraph.pointCount > 0 ) ) {
     new->xygraph.datasize = new->xygraph.pointCount * sizeof(int);
     new->xygraph.xvalues = (int *)XtMalloc( new->xygraph.datasize );
     (void)memcpy( new->xygraph.xvalues, new->xygraph.data[0],
		   new->xygraph.pointCount * sizeof(int) );
     new->xygraph.yvalues = (int *)XtMalloc( new->xygraph.datasize );
     (void)memcpy( new->xygraph.yvalues, new->xygraph.data[1],
		   new->xygraph.pointCount * sizeof(int) );

     ourData = (int **)XtMalloc( 3 * sizeof(int *) );
     new->xygraph.data = ourData;
     new->xygraph.data[0] = &(new->xygraph.xvalues[0] );
     new->xygraph.data[1] = &(new->xygraph.yvalues[0] );
     new->xygraph.data[2] = NULL;

     new->xygraph.xpixels =
                   (int *)XtMalloc( new->xygraph.datasize );
     (void)memcpy( new->xygraph.xpixels, new->xygraph.xvalues,
	           new->xygraph.datasize );
     new->xygraph.ypixels =
                   (int *)XtMalloc( new->xygraph.datasize );
     (void)memcpy( new->xygraph.ypixels, new->xygraph.yvalues,
		   new->xygraph.datasize );

     new->xygraph.valuecnt = (int *)XtMalloc( sizeof(int) );
     new->xygraph.valuecnt[0] = new->xygraph.pointCount;
     new->xygraph.point_coloring = (int *)XtMalloc( new->xygraph.datasize );
     redraw = TRUE;
  }

  if (redraw && realized) {
     Redisplay(new);
  }
  return(FALSE);
  
} /* SetValues */

static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  XYGraphWidget	w = (XYGraphWidget) gw;
  XtGCMask      pixmapGCmask;
  XGCValues     pixmapGCvalues;

#ifdef DEBUG
printf("Realize\n");
#endif
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->xygraph.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->xygraph.backing_store;
    break;
  }
  /* Create the widget window */
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);

  /* Create the drawing pixmap */
  pixmapGCmask = GCForeground;
  pixmapGCvalues.foreground = w->core.background_pixel;
  w->xygraph.pixmapGC = XtGetGC((Widget)w, pixmapGCmask, &pixmapGCvalues);
  w->xygraph.drawPixmap = XCreatePixmap( XtDisplay(w), XtWindow(w),
					 w->core.width, w->core.height,
				         DefaultDepth( XtDisplay(w),
				              DefaultScreen(XtDisplay(w)) ) );

  Resize(gw);
} /* Realize */



static void Destroy(gw)
Widget gw;
{
  int		i;
  XYGraphWidget	w = (XYGraphWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->xygraph.gridGC);
  DestroyColorGCs(w);
  if (w->xygraph.title != NULL)
    XtFree(w->xygraph.title);

} /* Destroy */



    
static void Resize(gw) 
Widget	gw;
{
  XYGraphWidget	w = (XYGraphWidget) gw;
  int		width;
  int		height;
  int		eleh, elew, deltax, deltay;
 
  Display       *display = XtDisplay(gw);
  int           screenNumber = DefaultScreen( XtDisplay(gw) );

#ifdef DEBUG
printf("Resize\n");
#endif

  if (XtIsRealized(gw)) {
      /* Size the drawing pixmap */
      display = XtDisplay(gw);
      XFreePixmap( display, w->xygraph.drawPixmap );
      w->xygraph.drawPixmap = XCreatePixmap( display, XtWindow(gw),
					     w->core.width, w->core.height,
					     DefaultDepth( display,
					                   screenNumber ) );
      w->xygraph.sizex = w->core.width - 2*PADDING;
      w->xygraph.offsetx = PADDING;

      if (w->xygraph.title != NULL)
	height = fontheight(w->xygraph.font)+TITLESPACE+PADDING;
      else 
	height = PADDING;

      w->xygraph.offsety1 = height;
      w->xygraph.offsety2 = PADDING;
      w->xygraph.sizey = w->core.height - height - w->xygraph.offsety2;

      if (w->xygraph.show_range) {
         w->xygraph.sizey -= 2*(fontheight(w->xygraph.font)+PADDING);
	 w->xygraph.offsety2 += 2*(fontheight(w->xygraph.font)+PADDING);
      }

      eleh = w->xygraph.sizey / w->xygraph.yticks;
      elew = w->xygraph.sizex / w->xygraph.xticks;

      deltax = w->xygraph.sizex - elew * w->xygraph.xticks;
      deltay = w->xygraph.sizey - eleh * w->xygraph.yticks;

      w->xygraph.offsetx += deltax/2;
      w->xygraph.offsety1 += deltay/2;
      w->xygraph.offsety2 += deltay/2;

      w->xygraph.sizex -= deltax;
      w->xygraph.sizey -= deltay;

      /* redisplay values */
      if (w->xygraph.xvalues != NULL) {
         XYGraphSetValues(w, w->xygraph.xvalues, 
			     w->xygraph.yvalues,
			     w->xygraph.valuecnt, w->xygraph.xygraph_cnt);
      }
    }
} /* Resize */

/* ARGSUSED */
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  XYGraphWidget	w = (XYGraphWidget) gw;
  int i, j;
  int dataColorIndex;
  Pixel currentPixel;

#ifdef DEBUG
printf("Redisplay\n");
#endif

  XFillRectangle( XtDisplay(w), w->xygraph.drawPixmap, 
		  w->xygraph.pixmapGC,
		  0,0,  w->core.width, w->core.height );
  DrawTitle(w);
  DrawGrid(w);
  if ( w->xygraph.densityPlot ) {
     DrawDensityXYGraph(w);
  } else {
     DrawXYGraph(w);
  }
  DrawXYGraphRange(w);

  XCopyArea( XtDisplay(w), w->xygraph.drawPixmap, XtWindow(w),
	     w->xygraph.pixmapGC,
	     0,0, w->core.width, w->core.height, 0,0 );
} /* Redisplay */


static void ReallocForValues(w, xvalues, yvalues,
			     numlines, numpoints, pointCounts )
XYGraphWidget   w;
int             *xvalues, *yvalues;
int             numlines, numpoints;
int             *pointCounts;
{
  int i, j, pts=0;

  if (w->xygraph.xvalues == NULL) {
     w->xygraph.valuecnt = (int *) XtMalloc( numlines * sizeof(int) );
     w->xygraph.xygraph_cnt = 0;
     w->xygraph.xvalues = (int *)XtMalloc( numpoints * sizeof(int) );
     w->xygraph.yvalues = (int *)XtMalloc( numpoints * sizeof(int) );
     w->xygraph.data = (int **)XtMalloc( 3 * sizeof(int *) );
     w->xygraph.data[2] = NULL;
     w->xygraph.xpixels = (int *) XtMalloc( numpoints * sizeof(int) );
     w->xygraph.ypixels = (int *) XtMalloc( numpoints * sizeof(int) );

     w->xygraph.minValues = (int *)XtMalloc( 2 * sizeof(int) );
     w->xygraph.maxValues = (int *)XtMalloc( 2 * sizeof(int) );

     w->xygraph.minValues[0] = w->xygraph.minx;
     w->xygraph.maxValues[0] = w->xygraph.maxx;
     w->xygraph.minValues[1] = w->xygraph.miny;
     w->xygraph.maxValues[1] = w->xygraph.maxy;

  } else {
     if ( numlines > w->xygraph.xygraph_cnt ) {
        w->xygraph.valuecnt = (int *) XtRealloc( (int *)w->xygraph.valuecnt,
 				                 numlines * sizeof(int) );
     }
     w->xygraph.xvalues = (int *)XtRealloc( w->xygraph.xvalues,
					    numpoints * sizeof(int) );
     w->xygraph.yvalues = (int *)XtRealloc( w->xygraph.yvalues,
					    numpoints * sizeof(int) );
     w->xygraph.xpixels = (int *)XtRealloc( w->xygraph.xpixels,
					    numpoints * sizeof(int) );
     w->xygraph.ypixels = (int *)XtRealloc( w->xygraph.ypixels,
					    numpoints * sizeof(int) );
  } 
  w->xygraph.data[0] = &(w->xygraph.xvalues[0] );
  w->xygraph.data[1] = &(w->xygraph.yvalues[0] );
  w->xygraph.datasize = numpoints * sizeof(int);

  if (w->xygraph.point_coloring == NULL ) {
     w->xygraph.point_coloring = (int *)XtMalloc( numpoints* sizeof(int) );
  } else {
     w->xygraph.point_coloring = (int *)XtRealloc(w->xygraph.point_coloring,
						  numpoints* sizeof(int) );
  }
}


static void ReallocForValue( w, xvalue, yvalue )
XYGraphWidget   w;
int             xvalue, yvalue;
{
  int i, j, pts=0;

  if (w->xygraph.xvalues == NULL) {
     w->xygraph.data = (int **)XtMalloc( 3 * sizeof(int) );
     w->xygraph.data[2] = NULL;

     w->xygraph.xvalues = (int *)XtMalloc( REALLOC_INTERVAL * sizeof(int) );
     w->xygraph.yvalues = (int *)XtMalloc( REALLOC_INTERVAL * sizeof(int) );
     w->xygraph.xpixels = (int *)XtMalloc(REALLOC_INTERVAL*sizeof(int));
     w->xygraph.ypixels = (int *)XtMalloc(REALLOC_INTERVAL*sizeof(int));
     w->xygraph.point_coloring = (int *)XtMalloc(REALLOC_INTERVAL*sizeof(int));
     w->xygraph.valuecnt = (int *) XtMalloc( sizeof(int) );

     w->xygraph.minValues = (int *)XtMalloc( 2 * sizeof(int) );
     w->xygraph.maxValues = (int *)XtMalloc( 2 * sizeof(int) );
     w->xygraph.minValues[0] = w->xygraph.minx;
     w->xygraph.maxValues[0] = w->xygraph.maxx;
     w->xygraph.minValues[1] = w->xygraph.miny;
     w->xygraph.maxValues[1] = w->xygraph.maxy;

     w->xygraph.valuecnt[0] = 0;
     w->xygraph.pointCount = 0;
     w->xygraph.xygraph_cnt = 1;
     w->xygraph.datasize = REALLOC_INTERVAL * sizeof(int);

  } else if ( w->xygraph.pointCount % REALLOC_INTERVAL == 0 ) {
     w->xygraph.xvalues = (int *)XtRealloc( (int *)w->xygraph.xvalues,
	    ((w->xygraph.pointCount+REALLOC_INTERVAL)) * sizeof(int));
     w->xygraph.yvalues = (int *)XtRealloc( (int *)w->xygraph.yvalues,
	    ((w->xygraph.pointCount+REALLOC_INTERVAL)) * sizeof(int));
     w->xygraph.xpixels = (int *)XtRealloc( (int *)w->xygraph.xpixels,
		  (w->xygraph.pointCount+REALLOC_INTERVAL) * sizeof(int) );
     w->xygraph.ypixels = (int *) XtRealloc( (int *) w->xygraph.ypixels,
		  (w->xygraph.pointCount+ REALLOC_INTERVAL) * sizeof(int) );
     w->xygraph.point_coloring = (int *)XtRealloc(
				        (int *)w->xygraph.point_coloring,
		  (w->xygraph.pointCount+ REALLOC_INTERVAL) * sizeof(int) );
  } 
  w->xygraph.data[0] = &(w->xygraph.xvalues[0]);
  w->xygraph.data[1] = &(w->xygraph.yvalues[0]);
  w->xygraph.datasize = (w->xygraph.pointCount + REALLOC_INTERVAL) *
                        sizeof(int);
}
 


static void CreateTextGC(w)
XYGraphWidget	w;
{
  XtGCMask	GCmask;
  XGCValues	GCvalues;

  GCmask = GCForeground | GCBackground | GCFont | GCLineWidth;
  GCvalues.foreground = w->xygraph.text_color;
  GCvalues.background = w->core.background_pixel;
  if (w->xygraph.font != NULL)
    GCvalues.font = w->xygraph.font->fid;
  else
    GCmask &= ~GCFont;		/* use server default font */
  GCvalues.line_width = 0;
  w->xygraph.textGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
} /* CreateTextGC */

static void CreateColorGCs(w)
XYGraphWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;

  for (i=0; i<w->xygraph.numcolors; i++) {
    gcValues.foreground = w->xygraph.colors[i];
    w->xygraph.colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }
} /* CreateColorGCs */


static void CreateBrushColorGCs(w)
XYGraphWidget w;
{
int         i;
XGCValues   gcValues;
XtGCMask    gcMask;

     gcValues.fill_style = FillSolid;
     gcMask = GCForeground; 

     for (i=0; i<w->xygraph.numBrushColors; i++) {
         gcValues.foreground = w->xygraph.brushColors[i];
         w->xygraph.brushColorGCs[i] = XtGetGC(w, gcMask, &gcValues );
     }
} /* CreateBrushColorGCs */


static void DestroyTextGC(w)
XYGraphWidget	w;
{
#ifdef DEBUG
printf("DestroyTextGC\n");
#endif
  if (w->xygraph.font != NULL)
    XtDestroyGC(w->xygraph.textGC);

} /* DestroyTextGC */

static void DestroyColorGCs(w)
XYGraphWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif
  if (w->xygraph.colors != NULL)
    for (i=0; i<w->xygraph.numcolors; i++)
      XtDestroyGC(w->xygraph.colorGCs[i]);

} /* DestroyColorGCs */


static void DrawTitle(w)
XYGraphWidget	w;
{
  int		x, y, l;

#ifdef DEBUG
printf("DrawTitle\n");
#endif

  if (w->xygraph.title != NULL) {
    l = XTextWidth(w->xygraph.font, w->xygraph.title, strlen(w->xygraph.title));
    x = (w->core.width - l)/2;
    y = FONTPAD + w->xygraph.font->max_bounds.ascent + PADDING;
    XDrawString(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.textGC, x, y,
		w->xygraph.title, strlen(w->xygraph.title));
  }
} /* DrawTitle */

static void DrawGrid(w)
XYGraphWidget	w;
{
  int		i;
  int		eleh;
  int		elew;
  int   	offx = w->xygraph.offsetx;
  int   	offy = w->xygraph.offsety1;
  int		yticks = w->xygraph.yticks;
  int		xticks = w->xygraph.xticks;
	
#ifdef DEBUG
printf("DrawGrid\n");
#endif

    eleh = w->xygraph.sizey / yticks;
    elew = w->xygraph.sizex / xticks;

    XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx, offy, offx, offy + yticks*eleh);

    XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx, offy, offx + xticks * elew, offy);

    XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx + xticks*elew, offy,
		offx + xticks * elew, offy + yticks*eleh);

    XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx, offy + yticks*eleh,
		offx + xticks*elew, offy + yticks*eleh);

  if (w->xygraph.show_ticks) {

    for(i=0; i<yticks+1; i++) {
      XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx, offy + i*eleh,
		offx + YTICK, offy + i*eleh);

      XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx + xticks*elew, offy+i*eleh,
		offx + xticks*elew - YTICK, offy + i*eleh);
    }
    for(i=0; i<xticks+1; i++) {
      XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx + i*elew, offy,
		offx + i*elew, offy + XTICK);

     XDrawLine(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.gridGC,
		offx + i*elew, offy + yticks*eleh,
		offx + i*elew, offy + yticks*eleh - XTICK);
    }
  }

} /* DrawGrid */



static void DrawLine(w, linenum)
XYGraphWidget	w;
int		linenum;
{
int	x1 = w->xygraph.xmove;
int	y1 = w->xygraph.ymove;
int	x2 = w->xygraph.xloc;
int	y2 = w->xygraph.yloc;

int	ColorLine;

#ifdef DEBUG
printf("DrawLine\n");
#endif

      if (w->xygraph.xvalues == NULL)
	return;

      if (w->xygraph.colors != NULL) {

	 if (linenum >= w->xygraph.numcolors)
	    ColorLine = w->xygraph.numcolors - 1;
	 else
	    ColorLine = linenum;

         XDrawLine(XtDisplay(w), w->xygraph.drawPixmap,
                   w->xygraph.colorGCs[ColorLine],
		   x1, y1, x2, y2);
	}

       w->xygraph.xmove = w->xygraph.xloc;
       w->xygraph.ymove = w->xygraph.yloc;

} /* DrawLine */

static void DrawPoint(w, linenum, pointnum)
XYGraphWidget	w;
int		linenum;
int             pointnum;
{
int	x1 = w->xygraph.xloc;
int	y1 = w->xygraph.yloc;
int     x, y;

int pointColor;
GC  gcToUse;

#ifdef DEBUG
printf("DrawPoint\n");
#endif
      if (w->xygraph.xvalues == NULL) {
	return;
      }

      if (w->xygraph.colors != NULL) {
         if ( w->xygraph.point_coloring[pointnum] > 0 ) {
            gcToUse = w->xygraph.brushColorGCs[
			   w->xygraph.point_coloring[pointnum] ];
	 } else {
	     pointColor = Min( linenum, w->xygraph.numcolors - 1);
             gcToUse = w->xygraph.colorGCs[pointColor];
	 }

	 if ( w->xygraph.markersize == 1 ) {
            XDrawPoint(XtDisplay(w), w->xygraph.drawPixmap, gcToUse,
                       x1, y1);
	 } else {
            x = x1 - w->xygraph.markersize/2;
 	    y = y1 - w->xygraph.markersize/2;
 	    XFillArc(XtDisplay(w), w->xygraph.drawPixmap, gcToUse,
                     x, y, w->xygraph.markersize, w->xygraph.markersize, 
                     0, 360*64);
	 }
     }
} /* DrawPoint */


static long DrawDensityPoint(w, linenum, pointnum)
XYGraphWidget	w;
int		linenum;
int             pointnum;
{
int	x1 = w->xygraph.xloc;
int	y1 = w->xygraph.yloc;
unsigned long currentPixel;

      if (w->xygraph.xvalues == NULL)
	return;

      currentPixel = XGetPixel( w->xygraph.drawImage, x1, y1) + 1;
      XPutPixel( w->xygraph.drawImage, x1, y1, currentPixel );
      return( currentPixel );
} 


extern void XYGraphSetValues(w, xvalues, yvalues, pointCounts, numlines)
XYGraphWidget    w;
int		*xvalues;
int		*yvalues;
int		*pointCounts;
int		numlines;
{
  register int	i, j;
  int		pts;
  float		FactorX, FactorY;
  double	x;
  int           numpoints = 0;
  Boolean       reAlloc = False;
  int           *newXValues, *newYValues;

#ifdef DEBUG
printf("XYGraphSetValues\n");
#endif


  /* determine the total number of points */
  for (i=0; i< numlines; i++) {
      numpoints += pointCounts[i];
  }

  /* determine if memory reallocations are necessary */
  if ( w->xygraph.valuecnt == NULL ) {
      reAlloc = True;
  } else {
      for (i=0; i<numlines; i++) {
         if ( pointCounts[i] != w->xygraph.valuecnt[i] ) {
            reAlloc = True;
	    break;
	 }
      }
  }

  /* reallocate memory */
  if ( reAlloc ) {
     ReallocForValues( w, xvalues, yvalues, numlines, numpoints, pointCounts );
  }

  pts = 0;
  for (i=0; i<numlines; i++) {
     for (j=0; j<pointCounts[i]; j++) {
         w->xygraph.xvalues[pts+j] = xvalues[pts+j];
         w->xygraph.xpixels[pts+j] = xvalues[pts+j];
         w->xygraph.yvalues[pts+j] = yvalues[pts+j];
         w->xygraph.ypixels[pts+j] = yvalues[pts+j];
         w->xygraph.point_coloring[pts+j] = 0;
     }
     w->xygraph.valuecnt[i] = pointCounts[i];
     pts += pointCounts[i];
  }
  w->xygraph.pointCount = pts;
  w->xygraph.xygraph_cnt = numlines;

  if (w->xygraph.autoscale != TRUE) {
    w->xygraph.xmin = w->xygraph.minx;
    w->xygraph.xmax = w->xygraph.maxx;
    w->xygraph.ymin = w->xygraph.miny;
    w->xygraph.ymax = w->xygraph.maxy;
  } else {
    w->xygraph.xmin = MAXINT;
    w->xygraph.ymin = MAXINT;
    w->xygraph.xmax = -MAXINT;
    w->xygraph.ymax = -MAXINT;
  }

  pts = 0;

  /* Here is where we start the data -> pixels xforms */
  for (i = 0; i < w->xygraph.xygraph_cnt; i++) {
     for (j = 0; j < w->xygraph.valuecnt[i]; j++) {
	if (w->xygraph.autoscale == TRUE) {
          if (w->xygraph.xvalues[pts+j] < w->xygraph.xmin)
             w->xygraph.xmin = w->xygraph.xvalues[pts+j]; 

          if (w->xygraph.xvalues[pts+j] > w->xygraph.xmax)
	      w->xygraph.xmax = w->xygraph.xvalues[pts+j];

          if (w->xygraph.yvalues[pts+j] < w->xygraph.ymin)
	      w->xygraph.ymin = w->xygraph.yvalues[pts+j];

          if (w->xygraph.yvalues[pts+j] > w->xygraph.ymax)
	      w->xygraph.ymax = w->xygraph.yvalues[pts+j];
        }
          if (w->xygraph.logx) {
             x = w->xygraph.xvalues[pts+j];
             x = log10(x)+0.5;
             w->xygraph.xpixels[pts+j] = x;
          }

          if (w->xygraph.logy) {
             x = w->xygraph.yvalues[pts+j];
             x = log10(x)+0.5;
	     w->xygraph.ypixels[pts+j] = x;
          }
       }
       pts += w->xygraph.valuecnt[i];
    }

  if (w->xygraph.logx) {
     x = w->xygraph.xmin;
     x = log10(x)+0.5;
     w->xygraph.xmin = x;
     x = w->xygraph.xmax;
     x = log10(x)+0.5;
     w->xygraph.xmax = x;
  }

  if (w->xygraph.logy) {
     x = w->xygraph.ymin;
     x = log10(x)+0.5;
     w->xygraph.ymin = x;
     x = w->xygraph.ymax;
     x = log10(x)+0.5;
     w->xygraph.ymax = x;
  }
  FactorX = w->xygraph.sizex/(float)(w->xygraph.xmax-w->xygraph.xmin);
  FactorY = w->xygraph.sizey/(float)(w->xygraph.ymax-w->xygraph.ymin);

  pts = 0;
  for (i = 0; i < w->xygraph.xygraph_cnt; i++) {
     for (j = 0; j < w->xygraph.valuecnt[i]; j++) {
        w->xygraph.xpixels[pts + j] = ScaleX(w->xygraph.xpixels[pts + j]);
        w->xygraph.ypixels[pts + j] = ScaleY(w->xygraph.ypixels[pts + j]);
     }
     pts += w->xygraph.valuecnt[i];
  }

  Redisplay(w);
} /* XYGraphSetValues */



extern void XYGraphSetValue( w, xvalue, yvalue )
XYGraphWidget    w;
int	        xvalue;
int		yvalue;
{
     int       x, y;  
     float     FactorX, FactorY;
     Boolean   rangeChange = False;
     int       newPointIndex;

/* ASSUME whichLine === 0 */

     w->xygraph.pointCount++;
  
     /* allocate new memory */
     ReallocForValue( w, xvalue, yvalue );
     newPointIndex = w->xygraph.valuecnt[0];
     w->xygraph.valuecnt[0] += 1;

     /* update data values */
     w->xygraph.xvalues[newPointIndex] = xvalue;
     w->xygraph.xpixels[newPointIndex] = xvalue;
     w->xygraph.yvalues[newPointIndex] = yvalue;
     w->xygraph.ypixels[newPointIndex] = yvalue;

     w->xygraph.point_coloring[newPointIndex] = 0;

     /* update extrema */
     if ( w->xygraph.autoscale == TRUE ) {
        if ( xvalue < w->xygraph.xmin ) {
           w->xygraph.xmin = xvalue;
	   rangeChange = True;
        } else if ( xvalue > w->xygraph.xmax ) {
           w->xygraph.xmax = xvalue;
	   rangeChange = True;
        }
        if ( yvalue < w->xygraph.ymin ) {
           w->xygraph.ymin = yvalue;
	   rangeChange = True;
        } else if ( yvalue > w->xygraph.ymax ) {
           w->xygraph.ymax = yvalue;
	   rangeChange = True;
        }
     } else {
        w->xygraph.xmin = w->xygraph.minx;
	w->xygraph.xmax = w->xygraph.maxx;
	w->xygraph.ymin = w->xygraph.miny;
	w->xygraph.ymax = w->xygraph.maxy;
     }

     /* logarithmically scale this point and also the extrema in case they
        have changed */
     if (w->xygraph.logx) {
        x = xvalue;                                /* value */
        x = log10(x)+0.5;
        w->xygraph.xpixels[newPointIndex] = x;
        if ( rangeChange ) {
           x = w->xygraph.xmin;                    /* min */
           x = log10(x)+0.5;
           w->xygraph.xmin = x;  
           x = w->xygraph.xmax;                    /* max */
           x = log10(x)+0.5;
           w->xygraph.xmax = x;
	}
     }
     if (w->xygraph.logy) { 
        y = yvalue;                                 /* value */
        y = log10(y)+0.5;
        w->xygraph.ypixels[newPointIndex]= y;
        if ( rangeChange ) {
           y = w->xygraph.ymin;                     /* min */
           y = log10(y)+0.5;
           w->xygraph.ymin = y;    
           y = w->xygraph.ymax;                     /* max */
           y = log10(y)+0.5;
           w->xygraph.ymax = y;
	}
     }  

     /* recalculate scaling factors and rescale if necessary */
     FactorX = w->xygraph.sizex/(float)(w->xygraph.xmax-w->xygraph.xmin);
     FactorY = w->xygraph.sizey/(float)(w->xygraph.ymax-w->xygraph.ymin); 
     w->xygraph.xpixels[newPointIndex] =
                                  ScaleX(w->xygraph.xpixels[newPointIndex]);
     w->xygraph.ypixels[newPointIndex] =
                                  ScaleY(w->xygraph.ypixels[newPointIndex]);

     Redisplay(w);
}



extern void XYGraphSetColors(w, numcolors, colors)
XYGraphWidget	w;
int		numcolors;
Pixel		*colors;
{

#ifdef DEBUG
printf("XYGraphSetColors\n");
#endif
DestroyColorGCs(w);

w->xygraph.numcolors = numcolors;
w->xygraph.colors = colors;

CreateColorGCs(w);
Redisplay(w);
} /* XYGraphSetColors */


extern void XYGraphSetDensityColors(w, numcolors, colors)
XYGraphWidget w;
int           numcolors;
Pixel         *colors;
{
int i;
     if ( w->xygraph.densityColors != NULL ) {
        XtFree( w->xygraph.densityColors );
     }
     w->xygraph.densityColors = (Pixel *)XtMalloc( numcolors*sizeof(Pixel));

     for (i=0; i<numcolors; i++) {
         w->xygraph.densityColors[i] = colors[i];
     }
     w->xygraph.numDensityColors = numcolors;
} /* XYGraphSetDensityColors */



extern void XYGraphSetBrushColors(w, numcolors, colors)
XYGraphWidget w;
int           numcolors;
Pixel         *colors;
{
int i;
     if ( w->xygraph.brushColors != NULL ) {
        XtFree( w->xygraph.brushColors );
     } 
     w->xygraph.brushColors = (Pixel *)XtMalloc( numcolors*sizeof(Pixel) );
     
     for (i=0; i<numcolors; i++) {
         w->xygraph.brushColors[i] = colors[i];
     }
     w->xygraph.numBrushColors = numcolors;

     CreateBrushColorGCs(w);     
}


static void DrawXYGraph(w)
XYGraphWidget	w;
{

int	i, j;
int	ptcnt;

#ifdef DEBUG
printf ("DrawXYGraph\n");
#endif

     if (w->xygraph.xvalues == NULL) {
        return;
     }

     ptcnt = 0;

     for (i = 0; i < w->xygraph.xygraph_cnt; i++) {
           /* The next two lines apply to DrawLine() only */
	   w->xygraph.xmove = w->xygraph.xpixels[ptcnt];
	   w->xygraph.ymove = w->xygraph.ypixels[ptcnt];

	   for (j = 0; j < w->xygraph.valuecnt[i]; j++) {
	      w->xygraph.xloc = w->xygraph.xpixels[ptcnt + j];
	      w->xygraph.yloc = w->xygraph.ypixels[ptcnt + j];
/*
printf(" Location %d = %d %d\n", ptcnt+j, w->xygraph.xpixels[ptcnt + j], w->xygraph.ypixels[ptcnt+j]);
*/

/*
	      if (w->xygraph.connect) {
	          DrawLine(w, i);
              }    
	      else {
*/
		 DrawPoint(w, i, ptcnt+j);
/*
              }
*/
	   }

	   ptcnt += w->xygraph.valuecnt[i];
	}
}


static void DrawDensityXYGraph(w)
XYGraphWidget	w;
{
int	i, j;
int	ptcnt;
long        current_overplot;
static long max_overplot = 2;
int         densityColorIndex;

       if (w->xygraph.xvalues == NULL)
          return;

       ptcnt = 0;
       for (i = 0; i < w->xygraph.xygraph_cnt; i++) {
 	   for (j = 0; j < w->xygraph.valuecnt[i]; j++) {
	      w->xygraph.xloc = w->xygraph.xvalues[ptcnt + j];
	      w->xygraph.yloc = w->xygraph.yvalues[ptcnt + j];
	      current_overplot = DrawDensityPoint(w, i, ptcnt+j);
              max_overplot = Max( max_overplot, current_overplot );
	   }
	   ptcnt += w->xygraph.valuecnt[i];
	}

        for( i=0; i < w->core.height; i++) {
           for ( j=0; j < w->core.width; j++) {
               current_overplot = XGetPixel(w->xygraph.drawImage, j, i );
	       if (current_overplot > 0 ) {
/*
if (current_overplot > 1 )
 printf("Pixel (%d,%d) has overplot %d\n", j, i, current_overplot );
*/
                  densityColorIndex = ((float)current_overplot /
		      (float)max_overplot) * w->xygraph.numDensityColors - 1;
  	          XPutPixel( w->xygraph.drawImage, j, i, 
		             w->xygraph.densityColors[densityColorIndex] );
	       } else {
                  XPutPixel( w->xygraph.drawImage, j, i,
			     w->core.background_pixel );
	       }
	   }
        }
        XPutImage( XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.pixmapGC,
	           w->xygraph.drawImage,  
	           0,0, 0,0, w->core.width, w->core.height );
}


static void DrawXYGraphRange(w)
XYGraphWidget w;
{
int	i,x,y,l;
char	label[128];

#ifdef DEBUG
	printf ("DrawXYGraphRange\n");
#endif

  if (w->xygraph.xvalues == NULL)
     return;

  if (w->xygraph.show_range) {

     if (w->xygraph.logx)
        sprintf (label, "X Min: %d  X Max: %d (Log X Axis)",
		w->xygraph.xmin, w->xygraph.xmax);
     else
        sprintf (label, "X Min: %d  X Max: %d", w->xygraph.xmin,
		   w->xygraph.xmax);

     l = XTextWidth(w->xygraph.font,label, strlen(label));
     x = w->xygraph.offsetx;
     y = w->core.height - 
	  (w->xygraph.font->max_bounds.ascent+PADDING/2);

     XDrawString(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.textGC,
		 x,y, 
		 label, strlen(label));

   if (w->xygraph.logy)
      sprintf (label, "Y Min: %d  Y Max: %d (Log Y Axis)",
		w->xygraph.ymin, w->xygraph.ymax);
   else
     sprintf (label, "Y Min: %d  Y Max: %d", w->xygraph.ymin,
		w->xygraph.ymax);

   l = XTextWidth(w->xygraph.font,label, strlen(label));
   x = w->xygraph.offsetx;
   y = w->core.height - 
	2*(w->xygraph.font->max_bounds.ascent+PADDING/2);

   XDrawString(XtDisplay(w), w->xygraph.drawPixmap, w->xygraph.textGC, x,y, 
		label, strlen(label));
  }
}

/************************************************************************/
/* XYGraphNotify() processes button up event.				*/
/************************************************************************/
static void XYGraphNotify(w, event)
XYGraphWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("XYGraphNotify\n");
#endif

  XtCallCallbacks(w, XtNselect, &(w->xygraph.location));

} /* XYGraphNotify */

/************************************************************************/
/* XYGraphSelect() processes button down event.				*/
/************************************************************************/
static void XYGraphSelect(w, event)
XYGraphWidget		w;
XButtonPressedEvent	*event;
{
	w->xygraph.location.x = event ->x;
	w->xygraph.location.y = event ->y;

} /* XYGraphSelect */

/************************************************************************/
/*	Euclid() finds the Euclidean distance between a pair of points.	*/ 
/************************************************************************/
static int Euclid(x1, y1, x2, y2)
int		x1, y1, x2, y2;
{
	int	Idist;
	double	Dist, dist;

#ifdef DEBUG
	printf("Euclid x1: %d y1: %d x2: %d y2: %d\n", x1, y1, x2, y2);
#endif

	dist = (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2);
if (dist == 0.0) printf("HOLY COW!!!!!!!!!!!!!\n");
	Dist = sqrt (dist);
	Idist = (int) Dist;

#ifdef DEBUG
	printf("Euclid dist: %f Idist: %d\n", dist, Idist);
#endif

	return (Idist);
}


/************************************************************************/
/*   	XYGraphPlotValues()  plots existing data values, either         */
/*      point-by-point or by density                                    */
/************************************************************************/
extern void XYGraphPlotValues( w )
XYGraphWidget  w;
{
        if ( w->xygraph.densityPlot ) {
           if ( w->xygraph.drawImage != NULL ) {
               XDestroyImage( w->xygraph.drawImage );
           }
           XFillRectangle( XtDisplay(w), w->xygraph.drawPixmap, 
	      	           w->xygraph.pixmapGC,
		           0,0,  w->core.width, w->core.height );
           w->xygraph.drawImage = XGetImage(XtDisplay(w),w->xygraph.drawPixmap,
				    0,0,  w->core.width, w->core.height,
				    AllPlanes, ZPixmap );
	}
        Redisplay(w);
}


/************************************************************************/
/*   	XYGraphGetNearestPoint() finds the graph point nearest the 	*/
/*      mouse select position.						*/
/************************************************************************/
extern void XYGraphGetNearestPoint(w, xNearest, yNearest, index)
XYGraphWidget	w;
int		*xNearest, *yNearest;
int		*index;
{
        int pts, i, j;
        int currentDistance, 
            closestDistance = w->core.height * w->core.width;

        if (w->xygraph.valuecnt == NULL) {
	    *index = -1;
            return;
        }
        pts = 0;
        for (i = 0; i < w->xygraph.xygraph_cnt; i++) {
            for (j = 0; j < w->xygraph.valuecnt[i]; j++) {
               currentDistance = Euclid(
	          w->xygraph.location.x, w->xygraph.location.y,
		  w->xygraph.xpixels[pts+j],w->xygraph.ypixels[pts+j]);
             if ( currentDistance < closestDistance ) {
                closestDistance = currentDistance;
                *xNearest = w->xygraph.xpixels[pts+j];
	        *yNearest = w->xygraph.ypixels[pts+j];
		*index = pts + j;
             }
           }
        }
}


/************************************************************************/
/*   	XYGraphDrawPointMarker() places a marker at the indicated 	*/
/*      graph (pixel) position.						*/
/************************************************************************/
extern void XYGraphDrawPointMarker(w, index)
XYGraphWidget	w;
int		index;
{
	int xLoc, yLoc;

	xLoc = w->xygraph.xvalues[index];
	yLoc = w->xygraph.yvalues[index];
	XDrawArc( XtDisplay(w), XtWindow(w), w->xygraph.textGC,
		  xLoc - MARKERRADIUS, yLoc - MARKERRADIUS, 
		  2*MARKERRADIUS, 2*MARKERRADIUS,
		  0, FULLARC);
}



/************************************************************************/
/*   	XYGraphSetPointColors() defines the color of all points         */
/*      THIS ROUTINE ASSUMES THAT ONLY ONE DATA SET (ONE "LINE") IS     */
/*         INVOLVED.                                                    */
/************************************************************************/
extern void XYGraphSetPointColors(w, whichColors)
XYGraphWidget	w;
int             *whichColors;
{
         int i;
	 for (i=0; i<w->xygraph.valuecnt[0]; i++) {
             w->xygraph.point_coloring[i] = whichColors[i] + 1;
	 }
	 Redisplay(w);
}

/************************************************************************/
/*   	XYGraphSetPointColor() defines the color of a particular point  */
/*      THIS ROUTINE ASSUMES THAT ONLY ONE DATA SET (ONE "LINE") IS     */
/*         INVOLVED.                                                    */
/************************************************************************/
extern void XYGraphSetPointColor(w, whichPoint, whichColor)
XYGraphWidget	w;
int		whichPoint;
int		whichColor;
{
         w->xygraph.point_coloring[whichPoint] = whichColor + 1;
	 Redisplay(w);
}



/************************************************************************/
/*   	XYGraphRemovePoint() deletes a particular data point from the   */
/*         XYGraph.                                                     */
/*      THIS ROUTINE ASSUMES THAT ONLY ONE DATA SET (ONE "LINE") IS     */
/*         INVOLVED.                                                    */
/************************************************************************/
extern void XYGraphRemovePoint(w, whichPoint)
XYGraphWidget w;
int           whichPoint;
{
     int pointCount = w->xygraph.valuecnt[0];
/*
     w->xygraph.xvalues[whichPoint] = w->xygraph.xvalues[pointCount-1];
     w->xygraph.yvalues[whichPoint] = w->xygraph.yvalues[pointCount-1];
     w->xygraph.xvalues_unscaled = w->xygraph.xvalues_unscaled[pointCount-1];
     w->xygraph.yvalues_unscaled = w->xygraph.yvalues_unscaled[pointCount-1];
     w->xygraph.valuecnt[0] = pointCount - 1;
*/
     Redisplay(w);
}     



