/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/led/RCS/Led.c,v 1.11 1994/02/25 04:46:40 aydt Exp $
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Xresource.h>
#include <X11/StringDefs.h>
#include "LedP.h"

#define Offset(field)	XtOffset(LedWidget, field)
#define Norm(w, v)	(((float)v-(float)w->led.min_value) /	\
			 ((float)w->led.max_value -		\
			  (float)w->led.min_value + 1))

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Resize(), Redisplay(), Notify(), Destroy();
static Boolean	SetValues();

static void	CreateColorGCs(), CreatePixmapGCs();
static void	DestroyColorGCs(), DestroyPixmapGCs();
static void	ResizeLed(), ResizeLedLevels();
static void	ClearLeds(), FillColorLeds(), FillPixmapLeds();
static void	DrawLeds(), DrawOutline(), DrawSticky();
static void	LedSelect(), LedNotify();

/**********************/
/* public procedures */
/**********************/
extern void	LedSetValue(), LedSetColors(), LedSetPixmaps();
extern int	LedEnqueue(), LedDequeue();
extern void	LedSetColors(), LedSetPixmaps();

/********************/
/* led translations */
/********************/
static char led_translations[] = "	\
  <Btn1Down>:	LedSelect()\n		\
  <Btn1Up>:	LedNotify()		\
";

/***************/
/* led actions */
/***************/
static XtActionsRec	led_actions[] = {
  {"LedSelect",	(XtActionProc) LedSelect	},
  {"LedNotify",	(XtActionProc) LedNotify	},
};

/*****************/
/* led resources */
/*****************/
static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRDimension, "100"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRDimension, "200"},
  /* led resources */
  {XtNstickyHold, XtCStickyHold, XtRInt, sizeof(int),
     Offset(led.sticky_hold), XtRString, "5"},
  {XtNlevels, XtCLevels, XtRInt, sizeof(int),
     Offset(led.levels), XtRInt, "10"},
  {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
     Offset(led.orientation), XtRString, "vertical"},
  {XtNshowOutline, XtCShowOutline, XtRBoolean, sizeof(Boolean),
     Offset(led.show_outline), XtRString, "TRUE"},
  {XtNoutlineColor, XtCOutlineColor, XtRPixel, sizeof(Pixel),
     Offset(led.outline_color), XtRString, "black"},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     Offset(led.numcolors), XtRInt, "10"},
  {XtNcolors, XtCColors, XtRPointer, sizeof(Pixel *),
     Offset(led.colors), XtRString, NULL},
  {XtNnumPixmaps, XtCNumPixmaps, XtRInt, sizeof(int),
     Offset(led.numpixmaps), XtRInt, "0"},
  {XtNpixmaps, XtCPixmaps, XtRPointer, sizeof(Pixmap *),
     Offset(led.pixmaps), XtRString, NULL},
  {XtNvalue, XtCValue, XtRInt, sizeof(int),
     Offset(led.value), XtRString, "0"},
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     Offset(led.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     Offset(led.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(led.select), XtRCallback, (caddr_t) NULL},
};
#undef Offset

/********************/
/* led class record */
/********************/
LedClassRec ledClassRec = {
  (WidgetClass) &widgetClassRec,	/* superclass		  */	
  "Led",				/* class_name		  */
  sizeof(LedRec),			/* size			  */
  ClassInitialize,			/* class_initialize	  */
  NULL,					/* class_part_initialize  */
  FALSE,				/* class_inited		  */
  Initialize,				/* initialize		  */
  NULL,					/* initialize_hook	  */
  XtInheritRealize,			/* realize		  */
  led_actions,				/* actions		  */
  XtNumber(led_actions),		/* num_actions		  */
  resources,				/* resources		  */
  XtNumber(resources),			/* resource_count	  */
  NULLQUARK,				/* xrm_class		  */
  TRUE,					/* compress_motion	  */
  TRUE,					/* compress_exposure	  */
  TRUE,					/* compress_enterleave    */
  FALSE,				/* visible_interest	  */
  Destroy,				/* destroy		  */
  Resize,				/* resize		  */
  Redisplay,				/* expose		  */
  SetValues,				/* set_values		  */
  NULL,					/* set_values_hook	  */
  XtInheritSetValuesAlmost,		/* set_values_almost	  */
  NULL,					/* get_values_hook	  */
  NULL,					/* accept_focus		  */
  XtVersion,				/* version		  */
  NULL,					/* callback_private	  */
  led_translations,			/* tm_table		  */
  XtInheritQueryGeometry,		/* query_geometry         */
  XtInheritDisplayAccelerator,		/* display_accelerator    */
};

WidgetClass ledWidgetClass = (WidgetClass)&ledClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(bgw)
Widget bgw;
{
  int		i;
  LedWidget	w = (LedWidget) bgw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->led.outlineGC);
  DestroyColorGCs(w);
  DestroyPixmapGCs(w);
  XtRemoveAllCallbacks(w, XtNselect);

} /* destroy */

/************************************************************************/
/* CvtStringToOrientation()						*/
/************************************************************************/
static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr	*args;		/* unused */
Cardinal	*num_args;	/* unused */
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XtOrientation	orient;
  XrmQuark		q;
  char			lowerName[1000];

  strcpy( lowerName, fromVal->addr );
  q = XrmStringToQuark(lowerName);
  if (q == XtQEhorizontal) {
    orient = XtorientHorizontal;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }
  if (q == XtQEvertical) {
    orient = XtorientVertical;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }

} /* CvtStringToOrientation */

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
  XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
  XtQEvertical   = XrmStringToQuark(XtEvertical);
  XtAddConverter(XtRString, XtROrientation, CvtStringToOrientation,
		 NULL, (Cardinal)0);

} /* ClassInitialize */

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget	request;
Widget	new;
{
  LedWidget	w = (LedWidget) new;
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("Initialize\n");
#endif
  /* initialize outline GC */
  gcValues.foreground = w->led.outline_color;
  gcMask = GCForeground;
  w->led.outlineGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize color GCs */
  if (w->led.numcolors > 0)
    if (w->led.colors != NULL)
      CreateColorGCs(w);
    else {
      w->led.numcolors = 1;
      w->led.colors = &w->led.outline_color;
      CreateColorGCs(w);
    }

  /* initialize pixmap GCs */
  if ((w->led.numpixmaps > 0) && (w->led.pixmaps != NULL))
    CreatePixmapGCs(w);

  w->led.value = w->led.min_value;
  w->led.oldvalue = w->led.min_value;
  w->led.sticky_value = w->led.min_value;
  w->led.sticky_held = 0;
  ResizeLed(w);
  ResizeLedLevels(w);

} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent;
Widget	grequest;
Widget	gnew;
{
  register LedWidget	current = (LedWidget) gcurrent;
  register LedWidget	new = (LedWidget) gnew;
  XGCValues		gcValues;
  XtGCMask		gcMask;
  Boolean		redraw = FALSE;

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    ResizeLed(new);
    ResizeLedLevels(new);
    redraw = TRUE;
  }

  /* change of outline color */
  if (new->led.outline_color != current->led.outline_color) {
    XtDestroyGC(current->led.outlineGC);
    gcValues.foreground = new->led.outline_color;
    gcMask = GCForeground;
    new->led.outlineGC = XtGetGC(current, gcMask, &gcValues);
    redraw = TRUE;
  }

  /* change of levels */
  if (new->led.levels != current->led.levels) {
    ResizeLed(new);
    ResizeLedLevels(new);
    redraw = TRUE;
  }

  /* change of colors */
  if (new->led.colors != current->led.colors) {
    DestroyColorGCs(current);
    CreateColorGCs(new);
    redraw = TRUE;
  }

  /* change of pixmaps */
  if (new->led.pixmaps != current->led.pixmaps) {
    DestroyPixmapGCs(current);
    CreatePixmapGCs(new);
    redraw = TRUE;
  }

  if (redraw && XtIsRealized(current))
    Redisplay(new);

  return(FALSE);

} /* SetValues */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw)
Widget	gw;
{
  LedWidget	w = (LedWidget) gw;

#ifdef DEBUG
printf("Resize\n");
#endif
  ResizeLed(w);
  ResizeLedLevels(w);
  Redisplay(gw);

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;		/* not used */
Region	region;		/* not used */
{
  LedWidget	w = (LedWidget) gw;

#ifdef DEBUG
printf("Redisplay\n");
#endif
  DrawOutline(w);
  w->led.oldvalue = w->led.min_value;
  DrawLeds(w); 
  DrawSticky(w);

} /* Redisplay */

/************************************************************************/
/* CreateColorGCs()							*/
/************************************************************************/
static void CreateColorGCs(w)
LedWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColoreGCs\n");
#endif
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  for (i=0; i<w->led.numcolors; i++) {
    gcValues.foreground = w->led.colors[i];
    w->led.colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

/************************************************************************/
/* CreatePixmapGCs()							*/
/************************************************************************/
static void CreatePixmapGCs(w)
LedWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreatePixmapsGCs\n");
#endif
  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->led.outline_color;
  gcValues.fill_style = FillTiled;
  gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
  for (i=0; i<w->led.numpixmaps; i++) {
    gcValues.tile = w->led.pixmaps[i];
    w->led.pixmapGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreatePixmapGCs */

/************************************************************************/
/* DestroyColorGCs()							*/
/************************************************************************/
static void DestroyColorGCs(w)
LedWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif
  if (w->led.colors != NULL)
    for (i=0; i<w->led.numcolors; i++)
      XtDestroyGC(w->led.colorGCs[i]);

} /* DestroyColorGCs */

/************************************************************************/
/* DestroyPixmapGCs()							*/
/************************************************************************/
static void DestroyPixmapGCs(w)
LedWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyPixmapGCs\n");
#endif
  if (w->led.pixmaps != NULL)
    for (i=0; i<w->led.numpixmaps; i++)
      XtDestroyGC(w->led.pixmapGCs[i]);

} /* DestroyPixmapGCs */

/************************************************************************/
/* ResizeLed()								*/
/************************************************************************/
static void ResizeLed(w)
LedWidget	w;
{
#ifdef DEBUG
printf("ResizeLed\n");
#endif
  if (w->led.orientation == XtorientVertical) {
    w->led.length = w->core.height - 2*PADDING;
    w->led.thickness = w->core.width - 2*PADDING;
  }
  else {
    w->led.length = w->core.width - 2*PADDING;
    w->led.thickness = w->core.height - 2*PADDING;
  }

} /* ResizeLed */

/************************************************************************/
/* ResizeLedLevels()							*/
/************************************************************************/
static void ResizeLedLevels(w)
LedWidget	w;
{
  register int	i;
  int		levels = w->led.levels;
  int		area, levellength;
  int		spacing, extraspace;

#ifdef DEBUG
printf("ResizeLedLevels\n");
#endif
  if (levels == 1) {
    w->led.leds[0].x = PADDING;
    w->led.leds[0].y = PADDING;
    if (w->led.orientation == XtorientVertical) {
      w->led.leds[0].width = w->led.thickness;
      w->led.leds[0].height = w->led.length;
    }
    else {
      w->led.leds[0].width = w->led.length;
      w->led.leds[0].height = w->led.thickness;
    }
  }
  else {
    area = LEDFRACTION * w->led.length;
    levellength = area / levels;
    spacing = (w->led.length-area) / (levels-1);
    extraspace = (w->led.length-area-(levels-1)*spacing) / 2;
    for (i=0; i<levels; i++) {
      if (w->led.orientation == XtorientVertical) {
	w->led.leds[levels-1-i].x = PADDING;
	w->led.leds[levels-1-i].y = PADDING + extraspace + i*levellength
				    + i*spacing;
	w->led.leds[levels-1-i].width = w->led.thickness;
	w->led.leds[levels-1-i].height = levellength;
      }
      else {
	w->led.leds[i].x = PADDING + extraspace + i*levellength
			   + i*spacing;
	w->led.leds[i].y = PADDING;
	w->led.leds[i].width = levellength;
	w->led.leds[i].height = w->led.thickness;
      }
    }
  }
  
} /* ResizeLedLevels */

/************************************************************************/
/* ClearLeds()								*/
/************************************************************************/
static void ClearLeds(w, lo, hi)
LedWidget	w;
int		lo, hi;
{
  register int	i;
  int		length;

#ifdef DEBUG
printf("ClearLeds\n");
#endif
  if (!XtIsRealized((Widget) w))
     return;

  for (i=lo; i<=hi; i++)
    if (w->led.show_outline)
      XClearArea(XtDisplay(w), XtWindow(w),
		 w->led.leds[i].x+1, w->led.leds[i].y+1,
		 w->led.leds[i].width-1, w->led.leds[i].height-1,
		 FALSE);
    else
      XClearArea(XtDisplay(w), XtWindow(w),
		 w->led.leds[i].x, w->led.leds[i].y,
		 w->led.leds[i].width, w->led.leds[i].height,
		 FALSE);
  
} /* ClearLeds */

/************************************************************************/
/* FillColorLeds()							*/
/************************************************************************/
static void FillColorLeds(w, value, lo, hi)
LedWidget	w;
int		value;
int		lo, hi;
{
  register int	i, j;
  
#ifdef DEBUG
printf("FillcolorLeds\n");
#endif
  if (!XtIsRealized((Widget) w))
     return;

  for (i=lo; i<=hi; i++) {
    if (i != hi)
      j = (float) (w->led.numcolors-1) * (float) i / (float) w->led.levels;
    else
      j = Norm(w, value) * (float) w->led.numcolors;
    if (w->led.show_outline)
      XFillRectangle(XtDisplay(w), XtWindow(w), w->led.colorGCs[j],
		     w->led.leds[i].x+1, w->led.leds[i].y+1,
		     w->led.leds[i].width-1, w->led.leds[i].height-1);
    else
      XFillRectangle(XtDisplay(w), XtWindow(w), w->led.colorGCs[j],
		     w->led.leds[i].x, w->led.leds[i].y,
		     w->led.leds[i].width, w->led.leds[i].height);
  }

} /* FillColorLeds */

/************************************************************************/
/* FillPixmapLeds()							*/
/************************************************************************/
static void FillPixmapLeds(w, value, lo, hi)
LedWidget	w;
int		value;
int		lo, hi;
{
  register int	i, j;
  
#ifdef DEBUG
printf("FillPixmapLeds\n");
#endif
  if (!XtIsRealized((Widget) w))
     return;

  for (i=lo; i<=hi; i++) {
    if (i != hi)
      j = (float) (w->led.numpixmaps-1) * (float) i / (float) w->led.levels;
    else
      j = Norm(w, value) * (float) w->led.numpixmaps;
    if (w->led.show_outline)
      XFillRectangle(XtDisplay(w), XtWindow(w), w->led.pixmapGCs[j],
		     w->led.leds[i].x+1, w->led.leds[i].y+1,
		     w->led.leds[i].width-1, w->led.leds[i].height-1);
    else
      XFillRectangle(XtDisplay(w), XtWindow(w), w->led.pixmapGCs[j],
		     w->led.leds[i].x, w->led.leds[i].y,
		     w->led.leds[i].width, w->led.leds[i].height);
  }

} /* FillPixmapLeds */

/************************************************************************/
/* EraseSticky()							*/
/************************************************************************/
static void EraseSticky(w)
LedWidget	w;
{
  int	l1, l2, levels = w->led.levels;

#ifdef DEBUG
printf("EraseSticky\n");
#endif

  if (!XtIsRealized((Widget) w))
     return;

  if (w->led.sticky_hold == 0)
    return;
  l1 = Norm(w, w->led.value) * (float) levels;
  l2 = Norm(w, w->led.sticky_value) * (float) levels;
  if (l1 >= l2)
    return;
  ClearLeds(w, l2, l2);

} /* EraseSticky */

/************************************************************************/
/* DrawSticky()								*/
/************************************************************************/
static void DrawSticky(w)
LedWidget	w;
{
  int	l, levels = w->led.levels;

#ifdef DEBUG
printf("DrawSticky\n");

#endif
  if (!XtIsRealized((Widget) w))
     return;

  if ((w->led.sticky_hold == 0) || (w->led.value == w->led.sticky_value))
    return;
  l = Norm(w, w->led.sticky_value) * (float) levels;
  if (w->led.pixmaps == None)
    FillColorLeds(w, w->led.sticky_value, l, l);
  else
    FillPixmapLeds(w, w->led.sticky_value, l, l);

} /* DrawSticky */

/************************************************************************/
/* DrawLeds()								*/
/************************************************************************/
static void DrawLeds(w)
LedWidget	w;
{
  int	lo, hi;
  int	levels = w->led.levels;

#ifdef DEBUG
printf("DrawLeds\n");
#endif
  if (XtIsRealized((Widget) w)) {
    if (w->led.value < w->led.oldvalue) {
      if (w->led.value == w->led.min_value)
	lo = 0;
      else
	lo = 1 + Norm(w, w->led.value) * (float) levels;
      hi = Norm(w, w->led.oldvalue) * (float) levels;
#ifdef DEBUG
printf("%d %d\n", lo, hi);
#endif
      ClearLeds(w, lo, hi);
    }
    else {
      lo = Norm(w, w->led.oldvalue) * (float) levels;
      hi = Norm(w, w->led.value) * (float) levels;
#ifdef DEBUG
printf("%d %d\n", lo, hi);
#endif
      if (w->led.pixmaps == None)
	FillColorLeds(w, w->led.value, lo, hi);
      else
	FillPixmapLeds(w, w->led.value, lo, hi);
    }
  }

} /* DrawLeds */

/************************************************************************/
/* DrawOutline()							*/
/************************************************************************/
static void DrawOutline(w)
LedWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DrawOutline\n");
#endif
  if (!XtIsRealized((Widget) w))
     return;

  if (w->led.show_outline)
    for (i=0; i<w->led.levels; i++)
      XDrawRectangle(XtDisplay(w), XtWindow(w), w->led.outlineGC,
		     w->led.leds[i].x, w->led.leds[i].y,
		     w->led.leds[i].width, w->led.leds[i].height);

} /* DrawOutline */

/************************************************************************/
/* LedSelect() processes button down event.				*/
/************************************************************************/
static void LedSelect(w, event)
LedWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("LedSelect\n");
#endif
  w->led.value_select = w->led.value;

} /* LedSelect */

/************************************************************************/
/* LedNotify() processes button up event.				*/
/************************************************************************/
static void LedNotify(w, event)
LedWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("LedNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->led.value_select);

} /* LedNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* LedSetValue()							*/
/************************************************************************/
extern void LedSetValue(w, value)
LedWidget	w;
int		value;
{
#ifdef DEBUG
printf("LedSetValue\n");
printf("%d %d %d\n", w->led.min_value, w->led.max_value, value);
#endif
  w->led.oldvalue = w->led.value;
  if (value < w->led.min_value) {
    printf("LedSetValue: tried to set a value (%d) ", value);
    printf("less than the min (%d)\n", w->led.min_value);
    w->led.value = w->led.min_value;
  }
  else if (value > w->led.max_value) {
    printf("LedSetValue: tried to set a value (%d) ", value);
    printf("greater than the max (%d)\n", w->led.max_value);
    w->led.value = w->led.max_value;
  }
  else
    w->led.value = value;

  DrawOutline(w);
  DrawLeds(w);
  if (w->led.sticky_hold > 0)
    if (value > w->led.sticky_value) {
      w->led.sticky_value = w->led.value;
      w->led.sticky_held = w->led.sticky_hold;
    }
    else if (w->led.sticky_held <= 0) {
      EraseSticky(w);
      w->led.sticky_value = w->led.value;
      w->led.sticky_held = w->led.sticky_hold;
    }
    else {
      w->led.sticky_held--;
      DrawSticky(w);
    }

} /* LedSetValue */

/************************************************************************/
/* LedSetColors()							*/
/************************************************************************/
extern void LedSetColors(w, numcolors, colors)
LedWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("LedSetColors\n");
#endif
  DestroyColorGCs(w);
  w->led.numcolors = numcolors;
  w->led.colors = colors;
  CreateColorGCs(w);
  DrawLeds(w);
  DrawSticky(w);

} /* LedSetColors */

/************************************************************************/
/* LedSetPixmaps()							*/
/************************************************************************/
extern void LedSetPixmaps(w, numpixmaps, pixmaps)
LedWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
#ifdef DEBUG
printf("LedSetPixmaps\n");
#endif
  DestroyPixmapGCs(w);
  w->led.numpixmaps = numpixmaps;
  w->led.pixmaps = pixmaps;
  CreatePixmapGCs(w);
  DrawLeds(w);
  DrawSticky(w);

} /* LedSetPixmaps */
