/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/interval/RCS/Interval.c,v 1.8 1994/02/25 04:46:24 aydt Exp $
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Xresource.h>
#include <X11/StringDefs.h>
#include "IntervalP.h"

#define Norm(w, v)	(((float) v - (float) w->interval.min) /	\
			 ((float) w->interval.max -			\
			  (float) w->interval.min))
#define Offset(field)	XtOffset(IntervalWidget, field)

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Resize(), Redisplay();
static void	Destroy(), DestroyColorGCs(), DestroyPixmapGCs();
static Boolean	SetValues();

static void	ClearArea(), FillArea(), DrawIntervals();
static void	IntervalSelect(), IntervalNotify();

/*********************/
/* public procedures */
/*********************/
extern void	IntervalSetIntervals(), IntervalUpdate();
extern void	IntervalSetColors(), IntervalSetPixmaps();

/*************************/
/* interval translations */
/*************************/
static char interval_translations[] = "					\
  <Btn1Down>:	IntervalSelect()\n					\
  <Btn1Up>:	IntervalNotify()					\
";

/********************/
/* interval actions */
/********************/
static XtActionsRec	interval_actions[] = {
  {"IntervalSelect",	(XtActionProc) IntervalSelect	},
  {"IntervalNotify",	(XtActionProc) IntervalNotify	},
};

/**********************/
/* interval resources */
/**********************/
static XtResource resources[] = {
  /******************/
  /* core resources */
  /******************/
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "200"},
  /**********************/
  /* interval resources */
  /**********************/
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     Offset(interval.min), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     Offset(interval.max), XtRString, "100"},
  {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
     Offset(interval.orientation), XtRString, "vertical"},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     Offset(interval.numcolors), XtRString, "10"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(interval.select), XtRCallback, (caddr_t) NULL},
};
#undef Offset

/*************************/
/* interval class record */
/*************************/
IntervalClassRec intervalClassRec = {
  (WidgetClass) &widgetClassRec,	/* superclass		  */	
  "Interval",				/* class_name		  */
  sizeof(IntervalRec),			/* size			  */
  ClassInitialize,			/* class_initialize	  */
  NULL,					/* class_part_initialize  */
  FALSE,				/* class_inited		  */
  Initialize,				/* initialize		  */
  NULL,					/* initialize_hook	  */
  XtInheritRealize,			/* realize		  */
  interval_actions,			/* actions		  */
  XtNumber(interval_actions),		/* num_actions		  */
  resources,				/* resources		  */
  XtNumber(resources),			/* resource_count	  */
  NULLQUARK,				/* xrm_class		  */
  TRUE,					/* compress_motion	  */
  TRUE,					/* compress_exposure	  */
  TRUE,					/* compress_enterleave    */
  FALSE,				/* visible_interest	  */
  Destroy,				/* destroy		  */
  Resize,				/* resize		  */
  Redisplay,				/* expose		  */
  SetValues,				/* set_values		  */
  NULL,					/* set_values_hook	  */
  XtInheritSetValuesAlmost,		/* set_values_almost	  */
  NULL,					/* get_values_hook	  */
  NULL,					/* accept_focus		  */
  XtVersion,				/* version		  */
  NULL,					/* callback_private	  */
  interval_translations,		/* tm_table		  */
  XtInheritQueryGeometry,		/* query_geometry         */
  XtInheritDisplayAccelerator,		/* display_accelerator    */
};

WidgetClass intervalWidgetClass = (WidgetClass) &intervalClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* CvtstringToOrientation() handles text orientation conversion.	*/
/************************************************************************/
static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr	*args;
Cardinal	*num_args;
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XtOrientation	orient;
  XrmQuark		q;
  char			lowerName[1000];

  strcpy( lowerName, fromVal->addr );
  q = XrmStringToQuark(lowerName);
  if (q == XtQEhorizontal) {
    orient = XtorientHorizontal;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }
  if (q == XtQEvertical) {
    orient = XtorientVertical;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }

} /* CvtStringToOrientation */

/************************************************************************/
/* ClassInitialize() performs orientation initialization.		*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
  XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
  XtQEvertical   = XrmStringToQuark(XtEvertical);
  XtAddConverter(XtRString, XtROrientation, CvtStringToOrientation,
		 NULL, (Cardinal)0);

} /* ClassInitialize */

/************************************************************************/
/* Initialize() creates initial GCs and dimensions.			*/
/************************************************************************/
static void Initialize(request, new)
Widget	request;
Widget	new;
{
  int		i;
  IntervalWidget	w = (IntervalWidget) new;
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("Initialize\n");
#endif
  /******************************/
  /* determine foreground pixel */
  /******************************/
  if (XWhitePixelOfScreen(XtScreen(w)) == w->core.background_pixel)
    w->interval.foreground = XBlackPixelOfScreen(XtScreen(w));
  else
    w->interval.foreground = XWhitePixelOfScreen(XtScreen(w));

  /***********************/
  /* initialize erase GC */
  /***********************/
  gcValues.foreground = w->core.background_pixel;
  gcMask = GCForeground;
  w->interval.eraseGC = XtGetGC(new, gcMask, &gcValues);
  /*********************/
  /* initialize colors */
  /*********************/
  gcValues.foreground = w->interval.foreground;
  gcMask = GCForeground;
  for (i=0; i<MAXCOLORS; i++) {
    w->interval.colorGCs[i] = XtGetGC(new, gcMask, &gcValues);
  }
  w->interval.numcolors = 0;

  /**********************/
  /* initialize pixmaps */
  /**********************/
  gcValues.foreground = w->interval.foreground;
  gcMask = GCForeground;
  for (i=0; i<MAXPIXMAPS; i++) {
    w->interval.pixmapGCs[i] = XtGetGC(new, gcMask, &gcValues);
  }
  w->interval.numpixmaps = 0;

  /**************************/
  /* initialize orientation */
  /**************************/
  if (w->interval.orientation == XtorientVertical) {
    if (w->core.height <= 0)
      w->interval.length = LENGTH;
    else
      w->interval.length = w->core.height;
    if (w->core.width <= 0)
      w->interval.thickness = THICKNESS;
    else
      w->interval.thickness = w->core.width;
  }
  else {
    if (w->core.width <= 0)
      w->interval.length = LENGTH;
    else
      w->interval.length = w->core.width;
    if (w->core.height <= 0)
      w->interval.thickness = THICKNESS;
    else
      w->interval.thickness = w->core.height;
  }

  w->interval.low = NULL;
  w->interval.high = NULL;
  w->interval.index = NULL;
  
} /* Initialize */

/************************************************************************/
/* SetValues() handles the setting of resources.			*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  IntervalWidget	current = (IntervalWidget) gcurrent;
  IntervalWidget	new = (IntervalWidget) gnew;
  XGCValues	gcValues;
  XtGCMask	gcMask;
  Boolean	redraw = FALSE;

#ifdef DEBUG
printf("SetValues\n");
#endif
  /*************************/
  /* change of erase color */
  /*************************/
  if (new->core.background_pixel != current->core.background_pixel) {
    XtDestroyGC(current->interval.eraseGC);
    gcValues.foreground = new->core.background_pixel;
    gcMask = GCForeground;
    new->interval.eraseGC = XtGetGC(current, gcMask, &gcValues);
    redraw = TRUE;
  }
  return(redraw);

} /* SetValues */

/************************************************************************/
/* Destroy() destroys all GCs and removes callbacks.			*/
/************************************************************************/
static void Destroy(bgw)
Widget bgw;
{
  IntervalWidget w = (IntervalWidget) bgw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->interval.eraseGC);
  DestroyColorGCs(w);
  DestroyPixmapGCs(w);
  XtRemoveAllCallbacks(w, XtNselect);

} /* Destroy */

/************************************************************************/
/* DestroyColorGCs()							*/
/************************************************************************/
static void DestroyColorGCs(w)
IntervalWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DestroyColorGCs\n");
printf("%d\n", w->interval.numcolors);
#endif
  for (i=0; i<w->interval.numcolors; i++)
    XtDestroyGC(w->interval.colorGCs[i]);

} /* DestroyColorGCs */

/************************************************************************/
/* DestroyPixmapGCs()							*/
/************************************************************************/
static void DestroyPixmapGCs(w)
IntervalWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DestroyPixmapGCs\n");
#endif
  for (i=0; i<w->interval.numpixmaps; i++)
    XtDestroyGC(w->interval.pixmapGCs[i]);

} /* DestroyPixmapGCs */

/************************************************************************/
/* CreateColorGCs() creates GCs for the color map table.		*/
/************************************************************************/
static void CreateColorGCs(w)
IntervalWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  for (i=0; i<w->interval.numcolors; i++) {
    gcValues.foreground = w->interval.colors[i];
    w->interval.colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

/************************************************************************/
/* CreatePixmapGCs() creates GCs for the pixmap table.			*/
/************************************************************************/
static void CreatePixmapGCs(w)
IntervalWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreatePixmapGCs\n");
#endif
  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->interval.foreground;
  gcValues.fill_style = FillTiled;
  gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
  for (i=0; i<w->interval.numpixmaps; i++) {
    gcValues.tile = w->interval.pixmaps[i];
    w->interval.pixmapGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreatePixmapGCs */

/************************************************************************/
/* Resize() updates dimensions then redisplays the interval widget.	*/
/************************************************************************/
static void Resize(gw)
Widget	gw;
{
  IntervalWidget	w = (IntervalWidget) gw;

#ifdef DEBUG
printf("Resize\n");
#endif
  if (w->interval.orientation == XtorientVertical) {
    w->interval.length = w->core.height;
    w->interval.thickness = w->core.width;
  }
  else {
    w->interval.length = w->core.width;
    w->interval.thickness = w->core.height;
  }
  Redisplay(gw);

} /* Resize */

/************************************************************************/
/* Redisplay() redraws the interval display.				*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;
Region	region;
{
  IntervalWidget	w = (IntervalWidget) gw;

#ifdef DEBUG
printf("Redisplay\n");
#endif
  DrawIntervals(w);

} /* Redisplay */

/************************************************************************/
/* ClearArea() clears the area in the interval low to high.		*/
/************************************************************************/
static void ClearArea(w, low, high)
IntervalWidget	w;
int		low, high;
{
  int	top, bottom;

#ifdef DEBUG
printf("ClearArea\n");
#endif
  if (w->interval.orientation == XtorientHorizontal) {
    top = (int) (Norm(w, high) * (float) w->core.width);
    bottom = (int) (Norm(w, low) * (float) w->core.width);
    XClearArea(XtDisplay(w), XtWindow(w),
	       bottom, 0, top-bottom, w->core.height, FALSE);
  }
  else {
    top = (int) (Norm(w, high) * (float) w->core.height);
    bottom = (int) (Norm(w, low) * (float) w->core.height);
    XClearArea(XtDisplay(w), XtWindow(w),
	       0, bottom, w->core.width, top-bottom, FALSE);
  }

} /* ClearArea */

/************************************************************************/
/* FillArea() fills the area in the interval low to high with a color,	*/
/* if index is > 0, or a pixmap, if index is < 0.  If index = 0, the	*/
/* area	is cleared.							*/
/************************************************************************/
static void FillArea(w, low, high, index)
IntervalWidget	w;
int		low, high, index;
{
  int	top, bottom;

#ifdef DEBUG
printf("FillArea\n");
#endif
  if (index == 0) {
    ClearArea(w, low, high);
    return;
  }
  if (w->interval.orientation == XtorientHorizontal) {
    top = (int) (Norm(w, high) * (float) w->core.width);
    bottom = (int) (Norm(w, low) * (float) w->core.width);
    if (index > 0) {
      if (index > w->interval.numcolors) {
	printf("interval.FillArea: color index %d not defined\n", index);
	return;
      }
      XFillRectangle(XtDisplay(w), XtWindow(w),
		     w->interval.colorGCs[index-1],
		     bottom, 0, top-bottom, w->core.height);
    }
    else {
      if ((-index) > w->interval.numpixmaps) {
	printf("interval.FillArea: pixmap index %d not defined\n", index);
	return;
      }
      XFillRectangle(XtDisplay(w), XtWindow(w),
		     w->interval.pixmapGCs[(-index)-1],
		     bottom, 0, top-bottom, w->core.height);
    }
  }
  else {
    top = (int) (Norm(w, high) * (float) w->core.height);
    bottom = (int) (Norm(w, low) * (float) w->core.height);
    if (index > 0) {
      if (index > w->interval.numcolors) {
	printf("interval.FillArea: color index %d not defined\n", index);
	return;
      }
      XFillRectangle(XtDisplay(w), XtWindow(w),
		     w->interval.colorGCs[index-1],
		     0, bottom, w->core.width, top-bottom);
    }
    else {
      if ((-index) > w->interval.numpixmaps) {
	printf("interval.FillArea: pixmap index %d not defined\n", index);
	return;
      }
      XFillRectangle(XtDisplay(w), XtWindow(w),
		     w->interval.pixmapGCs[(-index)-1],
		     0, bottom, w->core.width, top-bottom);
    }
  }

} /* FillArea */

/************************************************************************/
/* DrawIntervals() draws all currently defined intervals.		*/
/************************************************************************/
static void DrawIntervals(w)
IntervalWidget	w;
{
  int	i;
  int	*low = w->interval.low;
  int	*high = w->interval.high;
  int	*index = w->interval.index;

#ifdef DEBUG
printf("DrawIntervals\n");
#endif
  if (XtIsRealized(w)) {
    for (i=0; i<w->interval.numintervals; i++) {
      FillArea(w, low[i], high[i], index[i]);
    }
  }

} /* DrawIntervals */

/************************************************************************/
/* IntervalSelect() processes button down event.			*/
/************************************************************************/
static void IntervalSelect(w, event)
IntervalWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("IntervalSelect\n");
#endif
   w->interval.value_select = w->interval.min;

  if (w->interval.orientation == XtorientHorizontal) {
    w->interval.value_select +=
			(int) (((float) event->x / (float) w->core.width) *
			       (w->interval.max - w->interval.min));
  }
  else {
    w->interval.value_select += 
			(int) (((float) event->y / (float) w->core.height) *
			       (w->interval.max - w->interval.min));
  }

} /* IntervalSelect */

/************************************************************************/
/* IntervalNotify() processes button up event.				*/
/************************************************************************/
static void IntervalNotify(w, event)
IntervalWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("IntervalNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->interval.value_select);

} /* IntervalNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* IntervalSetIntervals() loads a new set of intervals and updates the	*/
/* display.								*/
/************************************************************************/
extern void IntervalSetIntervals(w, num, low, high, index)
IntervalWidget	w;
int		num, *low, *high, *index;
{
  int i;
#ifdef DEBUG
printf("IntervalSetIntervals\n");
#endif
  if ( w->interval.low == NULL ) {
     w->interval.low = (int *)XtMalloc( num * sizeof(int) );
     w->interval.high = (int *)XtMalloc( num * sizeof(int) );
     w->interval.index = (int *)XtMalloc( num * sizeof(int) );

  } else {
     free( w->interval.low );
     free( w->interval.high );
     free( w->interval.index );
     w->interval.low = (int *)XtMalloc( num * sizeof(int) );
     w->interval.high = (int *)XtMalloc( num * sizeof(int) );
     w->interval.index = (int *)XtMalloc( num * sizeof(int) );
  }
  w->interval.numintervals = num;
  for (i=0; i<w->interval.numintervals; i++) {
     w->interval.low[i] = low[i];
     w->interval.high[i] = high[i];
     w->interval.index[i] = index[i];
  }
  ClearArea(w, w->interval.min, w->interval.max);
  DrawIntervals(w);

} /* IntervalSetIntervals */

/************************************************************************/
/* IntervalSetColors() loads a new color map.				*/
/************************************************************************/
extern void IntervalSetColors(w, numcolors, colors)
IntervalWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("IntervalSetColors\n");
#endif
  DestroyColorGCs(w);
  w->interval.numcolors = numcolors;
  w->interval.colors = colors;
  CreateColorGCs(w);

} /* IntervalSetColors */

/************************************************************************/
/* IntervalSetPixmaps() loads a new pixmap table.			*/
/************************************************************************/
extern void IntervalSetPixmaps(w, numpixmaps, pixmaps)
IntervalWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
#ifdef DEBUG
printf("IntervalSetPixmaps\n");
#endif
  DestroyPixmapGCs(w);
  w->interval.numpixmaps = numpixmaps;
  w->interval.pixmaps = pixmaps;
  CreatePixmapGCs(w);

} /* IntervalSetPixmaps */

/************************************************************************/
/* IntervalUpdate() redraws the interval display.			*/
/************************************************************************/
extern void IntervalUpdate(w)
IntervalWidget	w;
{
  ClearArea(w, w->interval.min, w->interval.max);
  DrawIntervals(w);

} /* IntervalUpdate */
