/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/historydial/RCS/HistoryDial.c,v 1.7 1994/02/25 04:46:23 aydt Exp $
 */

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include "HistoryDialP.h"

#define Min(a, b)	((a) < (b) ? (a) : (b))
#define Round(x)	(x >= 0.0 ? (int)(x + .5) : (int)(x - .5))
#define Offset(field)	XtOffset(HistoryDialWidget, field)
#define Norm(w, v)	(((float)v - (float)w->dial.min_value) /	\
			 ((float)w->dial.max_value -			\
			 (float)w->dial.min_value))

extern double	cos(), sin();

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Realize(), Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	DestroyGCs(), CreateGCs();
static void	DrawDialFace(), DrawNeedle(), EraseNeedle();
static void 	ComputeTickPoints(), ComputeNeedlePoints();
static void	DestroyHistoryGCs(), CreateHistoryGCs();
static void	ClearHistory(), DrawHistory(), RotateHistory();
static void	HistDialSelect(), HistDialNotify();

/*********************/
/* public procedures */
/*********************/
extern void HistoryDialSetValue(), HistoryDialSetHistoryColors();

/****************************/
/* historydial translations */
/****************************/
static char dial_translations[] = "	\
  <Btn1Down>:	HistDialSelect()\n	\
  <Btn1Up>:	HistDialNotify()		\
";

/****************/
/* dial actions */
/****************/
static XtActionsRec	dial_actions[] = {
  {"HistDialSelect",	(XtActionProc) HistDialSelect	},
  {"HistDialNotify",	(XtActionProc) HistDialNotify	},
};

/**************************/
/* history dial resources */
/**************************/
static XtResource resources[] = {
  /******************/
  /* core resources */
  /******************/
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "100"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "100"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(core.background_pixel), XtRString, "white"},
  /******************/
  /* dial resources */
  /******************/
  {XtNhistory, XtCHistory, XtRInt, sizeof(int),
     Offset(dial.history), XtRString, "0"},
  {XtNnumHistoryColors, XtCNumHistoryColors, XtRInt, sizeof(int),
     Offset(dial.num_history_colors), XtRString, "0"},
  {XtNhistoryColors, XtCHistoryColors, XtRPointer, sizeof(Pixel *),
     Offset(dial.history_colors), XtRString, NULL},
  {XtNbeginDegree, XtCBeginDegree, XtRInt, sizeof(int),
     Offset(dial.begin_degree), XtRString,"0"},
  {XtNendDegree, XtCEndDegree, XtRInt, sizeof(int),
     Offset(dial.end_degree), XtRString,"360"},
  {XtNfaceBorderColor, XtCFaceBorderColor, XtRPixel, sizeof(Pixel),
     Offset(dial.face_border_color), XtRString, "black"},
  {XtNfaceColor, XtCFaceColor, XtRPixel, sizeof(Pixel),
     Offset(dial.face_color), XtRString, "white"},
  {XtNneedleColor, XtCNeedleColor, XtRPixel, sizeof(Pixel),
     Offset(dial.needle_color), XtRString, "black"},
  {XtNtickColor, XtCTickColor, XtRPixel, sizeof(Pixel),
     Offset(dial.tick_color), XtRString, "black"},
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     Offset(dial.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     Offset(dial.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(dial.select), XtRCallback, (caddr_t) NULL},
};
#undef	Offset

/*****************************/
/* history dial class record */
/*****************************/
HistoryDialClassRec historydialClassRec = {
  { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Dial",
    /* widget_size		*/	sizeof(HistoryDialRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	dial_actions,
    /* num_actions		*/	XtNumber(dial_actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	dial_translations,
    /* query_geometry           */	XtInheritQueryGeometry,
    /* display_accelerator      */	XtInheritDisplayAccelerator,
    /* extension                */	NULL
    }
};

WidgetClass historydialWidgetClass = (WidgetClass) &historydialClassRec;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);

} /* ClassInitialize */

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget request, new;
{
  HistoryDialWidget	w = (HistoryDialWidget) new;
  int		i;
  int		min_height, min_width;

#ifdef DEBUG
printf("Initialize\n");
#endif
  min_width = min_height = MIN_SIZE;
  if (w->core.width == 0)
    w->core.width = min_width;
  if (w->core.height == 0)
    w->core.height = min_height;
  
  CreateGCs(w);
  w->dial.dial_offsetY = 0;
  w->dial.value_offsetY = 0;
  w->dial.needle_length = 0;
  w->dial.centerX = 0;
  w->dial.centerY = 0;
  w->dial.value = w->dial.min_value;
  ClearHistory(w);
  Resize(w);
  ComputeNeedlePoints(w);

} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  HistoryDialWidget	current = (HistoryDialWidget) gcurrent;
  HistoryDialWidget	new = (HistoryDialWidget) gnew;
  Boolean	redisplay = FALSE;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("SetValues\n");
#endif
  if (new->dial.face_border_color != current->dial.face_border_color) {
    XtDestroyGC(current->dial.face_borderGC);
    GCmask = GCForeground;
    GCvalues.foreground = new->dial.face_border_color;
    new->dial.face_borderGC = XtGetGC((Widget) new, GCmask, &GCvalues);
    redisplay = TRUE;
  }
  
  if (new->dial.face_color != current->dial.face_color) {
    XtDestroyGC(current->dial.faceGC);
    GCmask = GCForeground;
    GCvalues.foreground = new->dial.face_color;
    new->dial.faceGC = XtGetGC((Widget) new, GCmask, &GCvalues);
    redisplay = TRUE;
  }
  
  if (new->core.background_pixel != current->core.background_pixel) {
    XtDestroyGC(current->dial.eraseGC);
    GCmask = GCForeground;
    GCvalues.foreground = new->core.background_pixel;
    new->dial.eraseGC = XtGetGC((Widget) new, GCmask, &GCvalues);
    redisplay = TRUE;
  }

  if (new->dial.needle_color != current->dial.needle_color) {
    XtDestroyGC(current->dial.needleGC);
    GCmask = GCForeground;
    GCvalues.foreground = new->dial.needle_color;
    new->dial.needleGC = XtGetGC((Widget) new, GCmask, &GCvalues);
    redisplay = TRUE;
  }

  if (new->dial.tick_color != current->dial.tick_color) {
    XtDestroyGC(current->dial.tickGC);
    GCmask = GCForeground;
    GCvalues.foreground = new->dial.tick_color;
    new->dial.tickGC = XtGetGC((Widget) new, GCmask, &GCvalues);
    redisplay = TRUE;
  }

  return(redisplay);
  
} /* SetValues */

/************************************************************************/
/* Realize()								*/
/************************************************************************/
static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  HistoryDialWidget	w = (HistoryDialWidget) gw;

#ifdef DEBUG
printf("Realize\n");
#endif
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch (w->dial.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |= CWBackingStore;
    attrs->backing_store = w->dial.backing_store;
    break;
  }
  XtCreateWindow(gw, InputOutput, (Visual *)CopyFromParent,
		 *GCmask, attrs);
  Resize(gw);

} /* Realize */

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(gw)
Widget gw;
{
  HistoryDialWidget w = (HistoryDialWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  DestroyGCs(w);

} /* Destroy */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw) 
Widget	gw;
{
  HistoryDialWidget	w = (HistoryDialWidget) gw;
  int		i, j, height;

#ifdef DEBUG
printf("Resize\n");
#endif
/*
  if (XtIsRealized(gw)) {
*/
    height = w->core.height;
    w->dial.radius = ((int) Min(w->core.width, height) - 
		      (int) (2 * PADDING)) / 2;
    w->dial.needle_length = NEEDLE_FRACTION * w->dial.radius;
    w->dial.needle_head_length = NEEDLE_HEAD_FRACTION * w->dial.radius;
    w->dial.centerX = w->core.width / 2;
    w->dial.centerY = w->dial.dial_offsetY + height / 2;
/*
  }
*/
  ClearHistory(w);
  ComputeNeedlePoints(w);

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;
Region	region;
{
  char		buf[512];
  HistoryDialWidget	w = (HistoryDialWidget) gw;
  
#ifdef DEBUG
printf("Redisplay\n");
#endif
  XClearArea(XtDisplay(w), XtWindow(w),
	     0, 0, w->core.width, w->core.height, False);
  DrawDialFace(w);
  DrawHistory(w);
  DrawNeedle(w);

} /* Redisplay */

/************************************************************************/
/* DestroyGCs()								*/
/************************************************************************/
static void DestroyGCs(w)
HistoryDialWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DestroyGCs\n");
#endif
  XtDestroyGC(w->dial.face_borderGC);
  XtDestroyGC(w->dial.faceGC);
  XtDestroyGC(w->dial.eraseGC);
  XtDestroyGC(w->dial.needleGC);
  XtDestroyGC(w->dial.tickGC);
  DestroyHistoryGCs(w);

} /* DestroyGCs */

/************************************************************************/
/* DestroyHistoryGCs()							*/
/************************************************************************/
static void DestroyHistoryGCs(w)
HistoryDialWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DestroyHistoryGCs\n");
#endif
  for (i=0; i<w->dial.num_history_colors; i++)
    XtDestroyGC(w->dial.historyGCs[i]);

} /* DestroyHistoryGCs */

/************************************************************************/
/* CreateGCs()								*/
/************************************************************************/
static void CreateGCs(w)
HistoryDialWidget	w;
{
  int		i;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("CreateGCs\n");
#endif
  /* create face border GC */
  GCmask = GCForeground;
  GCvalues.foreground = w->dial.face_border_color;
  w->dial.face_borderGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
  /* create face GC */
  GCmask = GCForeground;
  GCvalues.foreground = w->dial.face_color;
  w->dial.faceGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
  /* creat erase GC */
  GCmask = GCForeground;
  GCvalues.foreground = w->core.background_pixel;
  w->dial.eraseGC = XtGetGC((Widget)w, GCmask, &GCvalues);

  /* create needle GC */
  GCmask = GCForeground;
  GCvalues.foreground = w->dial.needle_color;
  w->dial.needleGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
  /* create tick GC */
  GCmask = GCForeground;
  GCvalues.foreground = w->dial.tick_color;
  w->dial.tickGC = XtGetGC((Widget)w, GCmask, &GCvalues);

  CreateHistoryGCs(w);

} /* CreateGCs */

/************************************************************************/
/* CreateHistoryGCs()							*/
/************************************************************************/
static void CreateHistoryGCs(w)
HistoryDialWidget	w;
{
  int		i;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("CreateHistoryGCs\n");
#endif
  GCmask = GCForeground;
  for (i=0; i<w->dial.num_history_colors; i++) {
    GCvalues.foreground = w->dial.history_colors[i];
    w->dial.historyGCs[i] = XtGetGC((Widget)w, GCmask, &GCvalues);
  }

} /* CreateHistoryGCs */

/************************************************************************/
/* ClearHistory()							*/
/************************************************************************/
static void ClearHistory(w)
HistoryDialWidget	w;
{
  int	i, j;
  int	centerx = w->dial.centerX, centery = w->dial.centerY;

  for (i=0; i<MAXHISTORY; i++)
    for (j=0; j<NEEDLE_POINTS-1; j++) {
      w->dial.needle_history[i][j].x = centerx;
      w->dial.needle_history[i][j].y = centery;
    }

} /* ClearHistory */

/************************************************************************/
/* EraseHistory()							*/
/************************************************************************/
static void EraseHistory(w)
HistoryDialWidget	w;
{
  int	i;
  int	centerx = w->dial.centerX, centery = w->dial.centerY;

#ifdef DEBUG
printf("EraseHistory\n");
#endif
  for (i=0; i<w->dial.history; i++)
    XDrawLines(XtDisplay(w), XtWindow(w), w->dial.eraseGC,
	       w->dial.needle_history[i],
	       NEEDLE_POINTS-1, CoordModeOrigin);

} /* EraseHistory */

/************************************************************************/
/* RotateHistory()							*/
/************************************************************************/
static void RotateHistory(w)
HistoryDialWidget	w;
{
  int	i, j;
  int	history = w->dial.history;

#ifdef DEBUG
printf("RotateHistory\n");
#endif
  /**********************************************************************/
  /* Only copy the head of the needle: points 1-5			*/
  /*									*/
  /*		 3							*/
  /*		/ \							*/
  /*	       /   \							*/
  /*	      /     \							*/
  /*	     4--1,5--2							*/
  /*             |							*/
  /*             0							*/
  /**********************************************************************/
  if (history > 0)
    XDrawLines(XtDisplay(w), XtWindow(w), w->dial.eraseGC,
	       w->dial.needle_history[history-1],
	       NEEDLE_POINTS-1, CoordModeOrigin);
  for (i=history-1; i>0; i--)
    for (j=0; j<NEEDLE_POINTS-1; j++)
      w->dial.needle_history[i][j] = w->dial.needle_history[i-1][j];
  for (j=1; j<NEEDLE_POINTS; j++)
    w->dial.needle_history[0][j-1] = w->dial.needle[j];

} /* RotateHistory */

/************************************************************************/
/* DrawHistory()							*/
/************************************************************************/
static void DrawHistory(w)
HistoryDialWidget	w;
{
  int	i;
  int	history = w->dial.history, colors = w->dial.num_history_colors;
  int	gc;

#ifdef DEBUG
printf("DrawHistory\n");
#endif
  for (i=0; i<history; i++)
    if (colors == 0)
      XDrawLines(XtDisplay(w), XtWindow(w), w->dial.needleGC,
		 w->dial.needle_history[i], NEEDLE_POINTS-1, CoordModeOrigin);
    else {
      gc = ((float) i / (float) history) * colors;
      XDrawLines(XtDisplay(w), XtWindow(w), w->dial.historyGCs[gc],
		 w->dial.needle_history[i], NEEDLE_POINTS-1, CoordModeOrigin);
    }

} /* DrawHistory */

/************************************************************************/
/* EraseNeedle()							*/
/************************************************************************/
static void EraseNeedle(w)
HistoryDialWidget	w;
{
#ifdef DEBUG
printf("EraseNeedle\n");
#endif
  XDrawLines(XtDisplay(w), XtWindow(w), w->dial.eraseGC,
	     w->dial.needle, NEEDLE_POINTS, CoordModeOrigin);

} /* EraseNeedle */

/************************************************************************/
/* DrawNeedle()								*/
/************************************************************************/
static void DrawNeedle(w)
HistoryDialWidget	w;
{
#ifdef DEBUG
printf("DrawNeedle\n");
#endif
  XDrawLines(XtDisplay(w), XtWindow(w), w->dial.needleGC,
	     w->dial.needle, NEEDLE_POINTS, CoordModeOrigin);

} /* DrawNeedle */

/************************************************************************/
/* ComputeNeedlePoints()						*/
/************************************************************************/
static void ComputeNeedlePoints(w)
HistoryDialWidget	w;
{
  Dimension		length = w->dial.needle_length;
  Dimension		head = w->dial.needle_head_length;
  register double	angle, cosangle, sinangle;
  register double	ws, wc;
  XPoint		*needle = w->dial.needle;

#ifdef DEBUG
printf("ComputeNeedlePoints\n");
#endif
  /**********************************************************************/
  /* A full circle is 2 PI radians.  Angles are measured from 12	*/
  /* o'dial, dialwise increasing.					*/
  /**********************************************************************/
  angle = PIHALF - TWOPI * (double) Norm(w, w->dial.value);
  cosangle = cos(angle);
  sinangle = sin(angle);

  /**********************************************************************/
  /* Order of points of the needle.					*/
  /*									*/
  /*		 3							*/
  /*		/ \							*/
  /*	       /   \							*/
  /*	      /     \							*/
  /*	     4--1,5--2							*/
  /*             |							*/
  /*             0							*/
  /**********************************************************************/
  wc = Round(NEEDLE_WIDTH * cosangle);
  ws = Round(NEEDLE_WIDTH * sinangle);

  /* 0 */
  needle[0].x = w->dial.centerX;
  needle[0].y = w->dial.centerY;
  /* 1 */
  needle[1].x = w->dial.centerX + Round(length * cosangle);
  needle[1].y = w->dial.centerY - Round(length * sinangle);
  /* 2 */
  needle[2].x = needle[1].x + ws;
  needle[2].y = needle[1].y + wc;
  /* 3 */
  needle[3].x = w->dial.centerX + Round((length+head) * cosangle);
  needle[3].y = w->dial.centerY - Round((length+head) * sinangle);
  /* 4 */
  needle[4].x = needle[1].x - ws;
  needle[4].y = needle[1].y - wc;
  /* 5 */
  needle[5].x = w->dial.centerX + Round(length * cosangle);
  needle[5].y = w->dial.centerY - Round(length * sinangle);

} /* ComputeNeedlePoints */

/************************************************************************/
/* ComputeTickPoints()							*/
/************************************************************************/
static void ComputeTickPoints(w, length, fraction, s)
HistoryDialWidget	w;
Dimension	length;
double		fraction;
XSegment	*s;
{
  int		cx = w->dial.centerX;
  int		cy = w->dial.centerY;
  double	angle, cosangle, sinangle;

#ifdef DEBUG
printf("ComputeTickPoints\n");
#endif
  /**********************************************************************/
  /* A full circle is 2 PI radians.  Angles are measured from 12	*/
  /* o'dial, dialwise increasing.					*/
  /**********************************************************************/
  angle = PIHALF - TWOPI * fraction;
  cosangle = cos(angle);
  sinangle = sin(angle);

  s->x1 = cx + (int)(length * sinangle);
  s->y1 = cy - (int)(length * cosangle);
  s->x2 = cx + (int)(w->dial.radius * sinangle);
  s->y2 = cy - (int)(w->dial.radius * cosangle);

} /* ComputeTickPoints */

/************************************************************************/
/* DrawDialFace()							*/
/************************************************************************/
static void DrawDialFace(w)
HistoryDialWidget	w;
{
  register int	i, s;
  Dimension	length = w->dial.needle_length+w->dial.needle_head_length+1;
  Dimension	delta = (w->dial.radius - length) / 2;

#ifdef DEBUG
printf("DrawDialFace\n");
#endif
  s = 0;
  for (i=0; i<MAXTICKS; i++) {
    if ((i % MAJOR_TICK) == 0)
      ComputeTickPoints(w, length, ((double) i)/(double) MAXTICKS,
			&w->dial.ticks[s]);
    else
      ComputeTickPoints(w, length+delta, ((double) i)/(double) MAXTICKS,
			&w->dial.ticks[s]);
    s++;
  }

  /* draw the ticks */
  XDrawSegments(XtDisplay(w), XtWindow(w), w->dial.tickGC,
		w->dial.ticks, s);
	
  /* draw dial edge */
  /***** need to add filling of dial face with face color *****/
  XDrawArc(XtDisplay(w), XtWindow(w), w->dial.face_borderGC,
	   w->dial.centerX - w->dial.radius,
	   w->dial.centerY - w->dial.radius,
	   2*w->dial.radius, 2*w->dial.radius,
	   0, 360*64);

} /* DrawDialFace */

/************************************************************************/
/* HistDialSelect() processes button down event.			*/
/************************************************************************/
static void HistDialSelect(w, event)
HistoryDialWidget	w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("HistDialSelect\n");
#endif
  w->dial.value_select = w->dial.value;

} /* HistDialSelect */

/************************************************************************/
/* HistDialNotify() processes button up event.				*/
/************************************************************************/
static void HistDialNotify(w, event)
HistoryDialWidget	w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("HistDialNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->dial.value_select);

} /* HistDialNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* HistoryDialSetValue()						*/
/************************************************************************/
extern void HistoryDialSetValue(w, v)
HistoryDialWidget	w;
int		v;
{
#ifdef DEBUG
printf("HistoryDialSetValue\n");
#endif
  EraseNeedle(w);
  RotateHistory(w);
  DrawHistory(w);
  if (v < w->dial.min_value) {
    printf("HistoryDialSetValue: tried to set a value (%d) ", v);
    printf("less than min (%d)\n", w->dial.min_value);
    w->dial.value = w->dial.min_value;
  }
  else if (v > w->dial.max_value) { 
    printf("HistoryDialSetValue: tried to set a value (%d) ", v);
    printf("greater than max (%d)\n", w->dial.max_value);
    w->dial.value = w->dial.max_value;
  }
  else
    w->dial.value = v;
  ComputeNeedlePoints(w);
  DrawNeedle(w);

} /* HistoryDialSetValue */

/************************************************************************/
/* HistoryDialSetHistoryColors()					*/
/************************************************************************/
extern void HistoryDialSetHistoryColors(w, numcolors, colors)
HistoryDialWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("HistoryDialSetHistoryColors\n");
#endif
  if (XtIsRealized(w))
    EraseHistory(w);
  DestroyHistoryGCs(w);
  w->dial.num_history_colors = numcolors;
  w->dial.history_colors = colors;
  CreateHistoryGCs(w);
  if (XtIsRealized(w)) {
    DrawHistory(w);
    DrawNeedle(w);
  }

} /* HistoryDialSetHistoryColors */
