/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/graph/RCS/Graph.c,v 1.7 1994/02/25 04:46:05 aydt Exp $
 */

#include <stdio.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "GraphP.h"

#define FontHeight(f)		(f->max_bounds.ascent +			\
				 f->max_bounds.descent+2*FONTPAD)
#define Min(x, y)		((x < y) ? x : y)
#define Offset(field)		XtOffset(GraphWidget, field)
#define Abs(a)			(((a) < 0) ? -(a) : (a))
#define NodeNorm(w, v)		(((float)v -				\
				  (float)w->graph.min_node_value) /	\
				 ((float) w->graph.max_node_value -	\
				  (float) w->graph.min_node_value + 1))
#define LinkNorm(w, v)		(((float)v -				\
				  (float)w->graph.min_link_value) /	\
				 ((float) w->graph.max_link_value -	\
				  (float) w->graph.min_link_value + 1))
#define	Done(address, type)	{(*toVal).size = sizeof(type);		\
				 (*toVal).addr = (caddr_t) address;}
#define XtEattachCenter		"center"
#define XtEattachNorthSouth	"northsouth"
#define XtEattachEastWest	"eastwest"

extern NodePtr	FindNode();

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Realize(), Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	CreateNodeBorderGC(), CreateLinkColorGCs();
static void	CreateTextGC(), CreateEraseGC();
static void	CreateNodeColorGCs(), CreateNodePixmapGCs();
static void	CreateSimpleLinkGC();
static void	DestroyNodeBorderGC(), DestroyLinkColorGCs();
static void	DestroyTextGC(), DestroyEraseGC();
static void	DestroyNodeColorGCs(), DestroyNodePixmapGCs();
static void	DestroySimpleLinkGC();
static void	DrawLink(), DrawNode(), DrawGraph();
static void	GraphSelect(), GraphNotify();
	
/*********************/
/* public procedures */
/*********************/
extern void	GraphSetNodeValue(), GraphSetLinkValue();
extern void	GraphSetNodeColors(), GraphSetNodePixmaps();
extern void	GraphSetLinkColors(), GraphSetLinkPixmaps();
extern void	GraphSetGraph();

/**********************/
/* graph translations */
/**********************/
static char graph_translations[] = "	\
  <Btn1Down>:	GraphSelect()\n		\
  <Btn1Up>:	GraphNotify()		\
";

/*****************/
/* graph actions */
/*****************/
static XtActionsRec	graph_actions[] = {
  {"GraphSelect",	(XtActionProc) GraphSelect	},
  {"GraphNotify",	(XtActionProc) GraphNotify	},
};

/*******************/
/* graph resources */
/*******************/
static XtResource resources[] = {
  /******************/
  /* core resources */
  /******************/
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "200"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(core.background_pixel), XtRString, "white"},
  /*******************/
  /* graph resources */
  /*******************/
  {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     Offset(graph.font), XtRString, "6x10"},
  {XtNtextColor, XtCTextColor, XtRPixel, sizeof(Pixel),
     Offset(graph.text_color), XtRString, "black"},
  {XtNnodeShape, XtCNodeShape, XtRInt, sizeof(int),
     Offset(graph.node_shape), XtRString, "1"},
  {XtNminNodeValue, XtCMinNodeValue, XtRInt, sizeof(int),
     Offset(graph.min_node_value), XtRString, "0"},
  {XtNmaxNodeValue, XtCMaxNodeValue, XtRInt, sizeof(int),
     Offset(graph.max_node_value), XtRString, "100"},
  {XtNminLinkValue, XtCMinLinkValue, XtRInt, sizeof(int),
     Offset(graph.min_link_value), XtRString, "0"},
  {XtNmaxLinkValue, XtCMaxLinkValue, XtRInt, sizeof(int),
     Offset(graph.max_link_value), XtRString, "100"},
  {XtNfillNodes, XtCFillNodes, XtRBoolean, sizeof (Boolean),
     Offset(graph.fill_nodes), XtRString, "FALSE"},
  {XtNattachType, XtCAttachType, XtRAttachType, sizeof (AttachType),
     Offset(graph.attach_type), XtRString, "center"},
  {XtNshowNodeName, XtCShowNodeName, XtRBoolean, sizeof (Boolean),
     Offset(graph.show_node_name), XtRString, "TRUE"},
  {XtNnodeBorderColor, XtCNodeBorderColor, XtRPixel, sizeof(Pixel),
     Offset(graph.node_border_color), XtRString, "black"},
  {XtNnumNodeColors, XtCNumNodeColors, XtRInt, sizeof(int),
     Offset(graph.num_node_colors), XtRString, "0"},
  {XtNnodeColors, XtCNodeColors, XtRPointer, sizeof(Pixel *),
     Offset(graph.node_colors), XtRString, NULL},
  {XtNnumNodePixmaps, XtCNumNodePixmaps, XtRInt, sizeof(int),
     Offset(graph.num_node_pixmaps), XtRString, "0"},
  {XtNnodePixmaps, XtCNodePixmaps, XtRPointer, sizeof(Pixel *),
     Offset(graph.node_pixmaps), XtRString, NULL},
  {XtNnumLinkColors, XtCNumLinkColors, XtRInt, sizeof(int),
     Offset(graph.num_link_colors), XtRString, "0"},
  {XtNlinkColors, XtCLinkColors, XtRPointer, sizeof(Pixel *),
     Offset(graph.link_colors), XtRString, NULL},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(graph.select), XtRCallback, (caddr_t) NULL},
};
#undef Offset

/**********************/
/* graph class record */
/**********************/
GraphClassRec graphClassRec = {
  &widgetClassRec,		/* superclass			*/
  "Graph",			/* class_name			*/
  sizeof(GraphRec),		/* widget_size			*/
  ClassInitialize,		/* class_initialize		*/
  NULL,				/* class_part_initialize	*/
  FALSE,			/* class_inited			*/
  Initialize,			/* initialize			*/
  NULL,				/* initialize_hook		*/
  Realize,			/* realize			*/
  graph_actions,		/* actions			*/
  XtNumber(graph_actions),	/* num_actions			*/
  resources,			/* resources			*/
  XtNumber(resources),		/* resource_count		*/
  NULL,				/* xrm_class			*/
  TRUE,				/* compress_motion		*/
  TRUE,				/* compress_exposure		*/
  TRUE,				/* compress_enterleave		*/
  FALSE,			/* visible_interest		*/
  Destroy,			/* destroy			*/
  Resize,			/* resize			*/
  Redisplay,			/* expose			*/
  SetValues,			/* set_values			*/
  NULL,				/* set_values_hook		*/
  XtInheritSetValuesAlmost,	/* set_values_almost		*/
  NULL,				/* get_values_hook		*/
  NULL,				/* accept_focus			*/
  XtVersion,			/* version			*/
  NULL,				/* callback_private		*/
  graph_translations,		/* tm_table			*/
  XtInheritQueryGeometry,	/* query_geometry         	*/
  XtInheritDisplayAccelerator,	/* display_accelerator		*/
  NULL				/* extension			*/
};

WidgetClass graphWidgetClass = (WidgetClass) &graphClassRec;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* CvtStringToAttachType()						*/
/************************************************************************/
void CvtStringToAttachType(args, num_args, fromVal, toVal)
XrmValuePtr	*args;
Cardinal	*num_args;
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static AttachType	attach;
  static XrmQuark	AttachCenterQ;
  static XrmQuark	AttachNorthSouthQ;
  static XrmQuark	AttachEastWestQ;
  static int		haveQuarks = 0;
  XrmQuark		q;
  char			lowerName[1000];

#ifdef DEBUG
printf("CvtStringToAttachType\n");
#endif
  if (!haveQuarks) {
    AttachCenterQ = XrmStringToQuark(XtEattachCenter);
    AttachEastWestQ = XrmStringToQuark(XtEattachEastWest);
    AttachNorthSouthQ = XrmStringToQuark(XtEattachNorthSouth);
    haveQuarks = 1;
  }
  XmuCopyISOLatin1Lowered(lowerName, (char *) fromVal->addr);
  q = XrmStringToQuark(lowerName);
  if (q == AttachCenterQ) {
    attach = AttachCenter;
    Done(&attach, AttachType);
    return;
  }
  if (q == AttachNorthSouthQ) {
    attach = AttachNorthSouth;
    Done(&attach, AttachType);
    return;
  }
  if (q == AttachEastWestQ) {
    attach = AttachEastWest;
    Done(&attach, AttachType);
    return;
  }

} /* CvtStringToAttachType */

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
}

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget request, new;
{
  int		i;
  GraphWidget	w = (GraphWidget)new;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize\n");
#endif
  if (w->core.width == 0)
    w->core.width = DEFAULT_SIZE;
  if (w->core.height == 0)
    w->core.height = DEFAULT_SIZE;

  w->graph.graph.numnodes = 0;
  w->graph.graph.nodes = NULL;
  if (w->graph.node_radius < MIN_NODE_RADIUS)
    w->graph.node_radius = MIN_NODE_RADIUS;
  else if (w->graph.node_radius > MAX_NODE_RADIUS)
    w->graph.node_radius = MAX_NODE_RADIUS;

  CreateTextGC(w);
  CreateEraseGC(w);
  CreateNodeBorderGC(w);
  CreateSimpleLinkGC(w);
  CreateNodeColorGCs(w);
  CreateNodePixmapGCs(w);
  CreateLinkColorGCs(w);

} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  int			i;
  register GraphWidget	current = (GraphWidget) gcurrent;
  register GraphWidget	new = (GraphWidget) gnew;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  Boolean               realized = XtIsRealized(gcurrent);

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    Resize(new);
    redraw = TRUE;
  }

  /* change background color */
  if (new->core.background_pixel != current->core.background_pixel) {
    DestroyEraseGC(new);
    CreateEraseGC(new);
    redraw = TRUE;
  }

  /* change of node filling */
  if (new->graph.fill_nodes != current->graph.fill_nodes)
    redraw = TRUE;

  /* change node text */
  if ((new->graph.font != current->graph.font)
      || (new->graph.text_color != current->graph.text_color)) {
    DestroyTextGC(new);
    CreateTextGC(new);
    redraw = TRUE;
  }
  
  /* change node border color */
  if (new->graph.node_border_color != current->graph.node_border_color) {
    DestroyNodeBorderGC(new);
    CreateNodeBorderGC(new);
    DestroySimpleLinkGC(new);
    CreateSimpleLinkGC(new);
    redraw = TRUE;
  }
  
  /* change node colors */
  if (new->graph.node_colors != current->graph.node_colors) {
    DestroyNodeColorGCs(new);
    CreateNodeColorGCs(new);
    redraw = TRUE;
  }
    
  /* change link colors */
  if (new->graph.link_colors != current->graph.link_colors) {
    DestroyLinkColorGCs(new);
    CreateLinkColorGCs(new);
    redraw = TRUE;
  }
    
  /* change node pixmaps */
  if (new->graph.node_pixmaps != current->graph.node_pixmaps) {
    DestroyNodePixmapGCs(new);
    CreateNodePixmapGCs(new);
    redraw = TRUE;
  }

  if (redraw && realized) {
    Redisplay(new);
  }

  return(FALSE);
  
} /* SetValues */

/************************************************************************/
/* Realize()								*/
/************************************************************************/
static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  GraphWidget	w = (GraphWidget) gw;

#ifdef DEBUG
printf("Realize\n");
#endif
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->graph.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->graph.backing_store;
    break;
  }
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);
  Resize(gw);
} /* Realize */

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(gw)
Widget gw;
{
  GraphWidget	w = (GraphWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  DestroyEraseGC(w);
  DestroyTextGC(w);
  DestroyNodeBorderGC(w);
  DestroySimpleLinkGC(w);
  DestroyNodeColorGCs(w);
  DestroyLinkColorGCs(w);
  DestroyNodePixmapGCs(w);

} /* Destroy */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw) 
Widget	gw;
{
  GraphWidget	w = (GraphWidget) gw;
  int		radius, space;

#ifdef DEBUG
printf("Resize\n");
#endif
  if (XtIsRealized(gw)) {
    radius = NODE_RADIUS_FRACTION * Min(w->core.width, w->core.height);
    w->graph.offsetX = radius + PADDING;
    w->graph.offsetY = radius + PADDING;
    w->graph.node_radius = radius;
    space = 2 * (PADDING + radius);
    ScaleGraph(&w->graph.graph,
	       w->core.width-space, w->core.height-space);
    if (w->graph.attach_type == AttachCenter)
      FindLinkAttachments(&w->graph.graph, 0, radius);
  }

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  GraphWidget	w = (GraphWidget) gw;
  
#ifdef DEBUG
printf("Redisplay\n");
#endif
if (XtIsRealized(gw))
  {
    XClearArea(XtDisplay(w), XtWindow(w), 0, 0,
	       w->core.width, w->core.height, False);
    DrawGraph(w);
  }
} /* Redisplay */

/************************************************************************/
/* CreateTextGC()							*/
/************************************************************************/
static void CreateTextGC(w)
GraphWidget	w;
{
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("CreateTextGC\n");
#endif
  gcMask = GCForeground | GCFont;
  gcValues.foreground = w->graph.text_color;
  if (w->graph.font != NULL)
    gcValues.font = w->graph.font->fid;
  else
    gcMask &= ~GCFont;
  w->graph.textGC = XtGetGC(w, gcMask, &gcValues);

} /* CreateTextGC */

/************************************************************************/
/* CreateEraseGC()							*/
/************************************************************************/
static void CreateEraseGC(w)
GraphWidget	w;
{
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("CreateEraseGC\n");
#endif
  gcMask = GCBackground | GCLineWidth;
  gcValues.background = w->core.background_pixel;
  gcValues.line_width = 2;
  w->graph.eraseGC = XtGetGC(w, gcMask, &gcValues);

} /* CreateEraseGC */

/************************************************************************/
/* CreateNodeBorderGC()							*/
/************************************************************************/
static void CreateNodeBorderGC(w)
GraphWidget	w;
{
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("CreateNodeBorderGC\n");
#endif
  gcMask = GCForeground | GCLineWidth;
  gcValues.foreground = w->graph.node_border_color;
  gcValues.line_width = 2;
  w->graph.node_borderGC = XtGetGC(w, gcMask, &gcValues);

} /* CreateNodeBorderGC */

/************************************************************************/
/* CreateSimpleLinkGC()							*/
/************************************************************************/
static void CreateSimpleLinkGC(w)
GraphWidget	w;
{
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("CreateSimpleLinkGC\n");
#endif
  gcMask = GCForeground;
  gcValues.foreground = w->graph.node_border_color;
  w->graph.linkGC = XtGetGC(w, gcMask, &gcValues);
} /* CreateSimpleLinkGC */

/************************************************************************/
/* CreateNodeColorGCs()							*/
/************************************************************************/
static void CreateNodeColorGCs(w)
GraphWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateNodeColorGCs\n");
#endif
  gcValues.fill_style = FillSolid;
  gcValues.line_width = 2;
  gcMask = GCForeground | GCFillStyle | GCLineWidth;
  for (i=0; i<w->graph.num_node_colors; i++) {
    gcValues.foreground = w->graph.node_colors[i];
    w->graph.node_colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateNodeColorGCs */

/************************************************************************/
/* CreateLinkColorGCs()							*/
/************************************************************************/
static void CreateLinkColorGCs(w)
GraphWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateLinkColorGCs\n");
#endif
  gcMask = GCForeground | GCLineWidth;
  for (i=0; i<w->graph.num_link_colors; i++) {
    gcValues.line_width = 0;
    gcValues.foreground = w->graph.link_colors[i];
    w->graph.link_colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateLinkColorGCs */

/************************************************************************/
/* CreateNodePixmapGCs()						*/
/************************************************************************/
static void CreateNodePixmapGCs(w)
GraphWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateNodePixmapGCs\n");
#endif
  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->graph.node_border_color;
  gcValues.fill_style = FillTiled;
  gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
  for (i=0; i<w->graph.num_node_pixmaps; i++) {
    gcValues.tile = w->graph.node_pixmaps[i];
    w->graph.node_pixmapGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateNodePixmapGCs */

/************************************************************************/
/* DestroyTextGC()							*/
/************************************************************************/
static void DestroyTextGC(w)
GraphWidget	w;
{
#ifdef DEBUG
printf("DestroyTextGC\n");
#endif
  XtDestroyGC(w->graph.textGC);

} /* DestroyTextGC */

/************************************************************************/
/* DestroyEraseGC()							*/
/************************************************************************/
static void DestroyEraseGC(w)
GraphWidget	w;
{
#ifdef DEBUG
printf("DestroyEraseGC\n");
#endif
  XtDestroyGC(w->graph.eraseGC);

} /* DestroyEraseGC */

/************************************************************************/
/* DestroyNodeBorderGC()						*/
/************************************************************************/
static void DestroyNodeBorderGC(w)
GraphWidget	w;
{
#ifdef DEBUG
printf("DestroyNodeBorderGC\n");
#endif
  XtDestroyGC(w->graph.node_borderGC);

} /* DestroyNodeBorderGC */

/************************************************************************/
/* DestroySimpleLinkGC()						*/
/************************************************************************/
static void DestroySimpleLinkGC(w)
GraphWidget	w;
{
#ifdef DEBUG
printf("DestroySimpleLinkGC\n");
#endif
  XtDestroyGC(w->graph.linkGC);

} /* DestroyNodeBorderGC */

/************************************************************************/
/* DestroyNodeColorGCs()						*/
/************************************************************************/
static void DestroyNodeColorGCs(w)
GraphWidget	w;
{
  register int	i;
  
#ifdef DEBUG
printf("DestroyNodeColorGCs\n");
#endif
  if (w->graph.node_colors != NULL)
    for (i=0; i<w->graph.num_node_colors; i++)
      XtDestroyGC(w->graph.node_colorGCs[i]);

} /* DestroyNodeColorGCs */

/************************************************************************/
/* DestroyLinkColorGCs()						*/
/************************************************************************/
static void DestroyLinkColorGCs(w)
GraphWidget	w;
{
  register int	i;
  
#ifdef DEBUG
printf("DestroyLinkColorGCs\n");
#endif
  if (w->graph.link_colors != NULL)
    for (i=0; i<w->graph.num_link_colors; i++)
      XtDestroyGC(w->graph.link_colorGCs[i]);

} /* DestroyLinkColorGCs */

/************************************************************************/
/* DestroyNodePixmapGCs()						*/
/************************************************************************/
static void DestroyNodePixmapGCs(w)
GraphWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyNodePixmapGCs\n");
#endif
  if (w->graph.node_pixmaps != NULL)
    for (i=0; i<w->graph.num_node_pixmaps; i++)
      XtDestroyGC(w->graph.node_pixmapGCs[i]);

} /* DestroyNodePixmapGCs */

/************************************************************************/
/* DrawGraph()								*/
/************************************************************************/
static void DrawGraph(w)
GraphWidget	w;
{
  int		i, degree;
  NodePtr	node = w->graph.graph.nodes;
  LinkPtr	link;

  if (XtIsRealized(w)) {
#ifdef DEBUG
printf("DrawGraph\n");
#endif
  while (node != NULL) {
    DrawNode(w, node);
    degree = node->todegree;
    for (i=0; i<degree; i++) {
      link = node->link[i];
      DrawLink(w, link);
    }
    node = node->next;
  }
}

} /* DrawGraph */

/************************************************************************/
/* DrawNode()								*/
/************************************************************************/
static void DrawNode(w, node)
GraphWidget	w;
NodePtr		node;
{
  int centerx,centery,colorgc,pixmapgc,radius,name_length,name_height;

  if (XtIsRealized(w)) {
    centerx = node->point.x + w->graph.offsetX;
    centery = node->point.y + w->graph.offsetY;
    colorgc = NodeNorm(w, node->value) * (float) w->graph.num_node_colors;
    pixmapgc = NodeNorm(w, node->value) * (float) w->graph.num_node_pixmaps;
    radius = w->graph.node_radius;
    name_length;
    name_height = w->graph.font->max_bounds.ascent +
    			w->graph.font->max_bounds.descent;

#ifdef DEBUG
printf("DrawNode\n");
#endif
  if (w->graph.fill_nodes) {
    if (w->graph.num_node_colors == 0) {
      if (w->graph.num_node_pixmaps == 0)
	XDrawArc(XtDisplay(w), XtWindow(w), w->graph.node_borderGC,
		 centerx - radius, centery - radius,
		 2*radius, 2*radius, 0, 360*64);
      else {
	XFillArc(XtDisplay(w), XtWindow(w), w->graph.node_pixmapGCs[pixmapgc],
		 centerx - radius, centery - radius,
		 2*radius, 2*radius, 0, 360*64);
	XDrawArc(XtDisplay(w), XtWindow(w), w->graph.node_borderGC,
		 centerx - radius, centery - radius,
		 2*radius, 2*radius, 0, 360*64);
      }
    }
    else {
      XFillArc(XtDisplay(w), XtWindow(w), w->graph.node_colorGCs[colorgc],
	       centerx - radius, centery - radius,
	       2*radius, 2*radius, 0, 360*64);
      XDrawArc(XtDisplay(w), XtWindow(w), w->graph.node_borderGC,
	       centerx - radius, centery - radius,
	       2*radius, 2*radius, 0, 360*64);
    }
  }
  else {
    if (w->graph.num_node_colors == 0)
      XDrawArc(XtDisplay(w), XtWindow(w), w->graph.node_borderGC,
	       centerx - radius, centery - radius,
	       2*radius, 2*radius, 0, 360*64);
    else
      XDrawArc(XtDisplay(w), XtWindow(w), w->graph.node_colorGCs[colorgc],
	       centerx - radius, centery - radius,
	       2*radius, 2*radius, 0, 360*64);
  }
  if (w->graph.show_node_name)
    if (node->name != NULL) {
      name_length = XTextWidth(w->graph.font,
			       node->name, strlen(node->name));
      XFillRectangle(XtDisplay(w), XtWindow(w), w->graph.eraseGC,
		     centerx - (name_length)/2 - 2,
		     centery - (name_height)/2 - 2,
		     name_length + 4, name_height + 4);
      XDrawString(XtDisplay(w), XtWindow(w), w->graph.textGC,
		  centerx - (name_length)/2,
		  centery + w->graph.font->max_bounds.descent,
		  node->name, strlen(node->name));
    }
  }
} /* DrawNode */

/************************************************************************/
/* DrawLink()								*/
/************************************************************************/
static void DrawLink(w, link)
GraphWidget	w;
LinkPtr		link;
{
  int offsetx,gc,x1,x2,radius,offsety,y1,y2;

  if (XtIsRealized(w)) {
    offsetx = w->graph.offsetX;
    offsety = w->graph.offsetY;
    gc = LinkNorm(w, link->value) * (float) w->graph.num_link_colors;
    x1 = link->segment.x1;
    y1 = link->segment.y1;
    x2 = link->segment.x2;
    y2 = link->segment.y2;
    radius = w->graph.node_radius;

#ifdef DEBUG
printf("DrawLink\n");
#endif
  switch (w->graph.attach_type) {
  case AttachEastWest:
    if (x1 < x2) {
      x1 += radius;
      x2 -= radius;
    }
    else {
      x1 -= radius;
      x2 += radius;
    }
    break;
  case AttachNorthSouth:
    if (y1 < y2) {
      y1 += radius;
      y2 -= radius;
    }
    else {
      y1 -= radius;
      y2 += radius;
    }
    break;
  }
  
  if (w->graph.num_link_colors == 0)
    XDrawLine(XtDisplay(w), XtWindow(w), w->graph.linkGC,
	      x1+offsetx, y1+offsety, x2+offsetx, y2+offsety);
  else
    XDrawLine(XtDisplay(w), XtWindow(w), w->graph.link_colorGCs[gc],
	      x1+offsetx, y1+offsety, x2+offsetx, y2+offsety);
  }
} /* DrawLink */

/************************************************************************/
/* GraphSelect() processes button down event.				*/
/************************************************************************/
static void GraphSelect(w, event)
GraphWidget		w;
XButtonPressedEvent	*event;
{
  int		i, centerx, centery, radius = w->graph.node_radius;
  NodePtr	node = w->graph.graph.nodes;

#ifdef DEBUG
printf("GraphSelect\n");
#endif
  while (node != NULL) {
    centerx = node->point.x + w->graph.offsetX;
    centery = node->point.y + w->graph.offsetY;
    if ((event->x >= (centerx - radius))
	&& (event->x <= (centerx + radius))
	&& (event->y >= (centery - radius))
	&& (event->y <= (centery + radius))) {
      w->graph.node_select = node;
      return;
    }
    node = node->next;
  }    
  w->graph.node_select = NULL;

} /* GraphSelect */

/************************************************************************/
/* GraphNotify() processes button up event.				*/
/************************************************************************/
static void GraphNotify(w, event)
GraphWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("GraphNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->graph.node_select->id);

} /* GraphNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* GraphSetGraph()							*/
/************************************************************************/
extern void GraphSetGraph(w, igraph)
GraphWidget	w;
I_GraphPtr	igraph;
{
  int	space = 2 * (PADDING + w->graph.node_radius);

#ifdef DEBUG
printf("GraphSetGraph\n");
#endif
  ReadInputGraph(igraph, &w->graph.graph);
  Resize((Widget) w);
  Redisplay((Widget) w);

} /* GraphSetGraph */

/************************************************************************/
/* GraphSetNodeColors()							*/
/************************************************************************/
extern void GraphSetNodeColors(w, numcolors, colors)
GraphWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("GraphSetNodeColors\n");
#endif
  DestroyNodeColorGCs(w);
  w->graph.num_node_colors = numcolors;
  w->graph.node_colors = colors;
  CreateNodeColorGCs(w);
  DrawGraph(w);

} /* GraphSetNodeColors */

/************************************************************************/
/* GraphSetLinkColors()							*/
/************************************************************************/
extern void GraphSetLinkColors(w, numcolors, colors)
GraphWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("GraphSetLinkColors\n");
#endif
  DestroyLinkColorGCs(w);
  w->graph.num_link_colors = numcolors;
  w->graph.link_colors = colors;
  CreateLinkColorGCs(w);
  DrawGraph(w);

} /* GraphSetLinkColors */

/************************************************************************/
/* GraphSetNodePixmaps()						*/
/************************************************************************/
extern void GraphSetNodePixmaps(w, numpixmaps, pixmaps)
GraphWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
int i;

#ifdef DEBUG
printf("GraphSetNodePixmaps\n");
#endif

  DestroyNodePixmapGCs(w);
  w->graph.num_node_pixmaps = numpixmaps;
  w->graph.node_pixmaps = pixmaps;
  CreateNodePixmapGCs(w);
  w->graph.fill_nodes = TRUE;
  DrawGraph(w);
} /* GraphSetNodePixmaps */

/************************************************************************/
/* GraphSetLinkValue()							*/
/************************************************************************/
extern void GraphSetLinkValue(w, from, to, value)
GraphWidget	w;
NodeID		to, from;
int		value;
{
  int		i;
  NodePtr	node;
  LinkPtr	link;

#ifdef DEBUG
printf("GraphSetLinkValue\n");
#endif
  node = FindNode(&(w->graph.graph), from);
  if (node == NULL) {
    printf("GraphSetLinkValue: cannot find node %d\n", from);
    exit(1);
  }
  i = FindToNode(node, to);
  if (i == MAXDEGREE) {
    printf("GraphSetLinkValues: cannot find node %d to node %d link\n",
	   from, to);
    exit(1);
  }
  link = node->link[i];
  if (value < w->graph.min_link_value) {
    printf("GraphSetLinkValue: tried to set a link value (%d) ", value);
    printf("less than the min (%d)\n", w->graph.min_link_value);
    link->value = w->graph.min_link_value;
  }
  else if (value > w->graph.max_link_value) {
    printf("GraphSetLinkValue: tried to set a link value (%d) ", value);
    printf("greater than the max (%d)\n", w->graph.max_link_value);
    link->value = w->graph.max_link_value;
  }
  else
    link->value = value;
  DrawLink(w, link);

} /* GraphSetLinkValue */

/************************************************************************/
/* GraphSetNodeValue()							*/
/************************************************************************/
extern void GraphSetNodeValue(w, id, value)
GraphWidget	w;
NodeID		id;
int		value;
{
  int		i;
  NodePtr	node;

#ifdef DEBUG
printf("GraphSetNodeValue\n");
#endif
  node = FindNode(&(w->graph.graph), id);
  if (node == NULL) {
    printf("GraphSetNodeValue: cannot find node %d\n", id);
    exit(1);
  }
  if (value < w->graph.min_node_value) {
    printf("GraphSetNodeValue: tried to set a node value (%d) ", value);
    printf("less than the min (%d)\n", w->graph.min_node_value);
    node->value = w->graph.min_node_value;
  }
  else if (value > w->graph.max_node_value) {
    printf("GraphSetNodeValue: tried to set a node value (%d) ", value);
    printf("greater than the max (%d)\n", w->graph.max_node_value);
    node->value = w->graph.max_node_value;
  }
  else
    node->value = value;
  DrawNode(w, node);

} /* GraphSetNodeValue */
