/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author:  Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * DialFormWrapper.cc:  Class wrapper for the Dial performance widget  
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/Wrappers/RCS/DialFormWrapper.C,v 1.12 1994/02/25 04:45:05 aydt Exp $
 */

#include "DialFormWrapper.h"

extern "C" {
#include "Dial.h"
}

#include <Xm/Form.h>     // Form widget class name definition
#include "XArgs.h"       // Defines the X widget argument structure type

#define ADJ_DIAL_WIDTH_ARY 0.5
#define ADJ_DIAL_HEIGHT_ARY 0.5

#define max(a,b)	 ( (a) > (b) ? (a):(b) )


DialFormWrapper::DialFormWrapper( WidgetBase *par, 
				  DialFU *dialFUPtr,
				  const XArgs& args,
				  const CString& name ) 
	        : PerfWidget( par )
{
	_setClassName( MY_CLASS );
    	_setType( "DialFormWrapper" );
	dialFUthisPtr = dialFUPtr;

        // Create the Motif Form widget
    	widget = XtCreateManagedWidget( name.getValue(), xmFormWidgetClass, 
				        par->getWidget(), args.getArgs(), 
				        args.getNArgs() );
	Arg arg;
	XtSetArg( arg, XmNfractionBase, (XtArgVal)10000 );
	XtSetValues( widget, &arg, 1 );

	// Set initial number of columns and rows
	formCols = 1;
	formRows = 1;

        // Create a single Dial to be the initial child
        childList = new Widget[1];
        childList[0] = XtCreateManagedWidget( "Dial:0,0", dialWidgetClass, 
				              widget, NULL, 0 );

	// Create Min/Max Label & other labels
	labels.createMinMaxLabel( widget, "DialMinMaxLabel" );
	setDialLabel( L_LEFT_VERT, NULLCHARSTRING );
	setDialLabel( L_BOTTOM_HORIZ, NULLCHARSTRING );
	
	addCallback( childList[0], XtNselect, &Callback::callback1, this );
}



DialFormWrapper::~DialFormWrapper()
{
	XtDestroyWidget( widget );
	delete [] childList;
}


void
DialFormWrapper::_adjustLayout( int nCols, int nRows )
{
        // Unmanage all the child widgets before resize
        XtUnmanageChildren( childList, (formRows * formCols) );

        // The plan here is to determinse the current height of "any" dial
        // (we use the first dial as our canonical dial), then set the new
        // width and height (again for a single dial).

        Dimension dialWidth, dialHeight;
        Dimension oldSingleDialWidth, oldSingleDialHeight;
        XtVaGetValues( childList[0],
                       XmNwidth, &oldSingleDialWidth,
                       XmNheight, &oldSingleDialHeight,
                          NULL );
        dialWidth = (oldSingleDialWidth*formCols) / nCols;
        dialHeight = (oldSingleDialHeight*formRows) / nRows;

        _genChildLayout( nCols, nRows );

	setDialAttr( XmNwidth, dialWidth );
	setDialAttr( XmNheight, dialHeight );

        // Make visible all the new repositioned stuff
        XtManageChildren( childList, (nRows * nCols) );
        XFlush( XtDisplay( widget ) );
}

Widget 
DialFormWrapper::_createChild( char *newWidgetName, int iCol, int iRow,
                            int dialLayoutXOrg, int dialLayoutYOrg,
			    int dialWidthProportion, int dialHeightProportion )
{
   	int i = 0;
   	Arg arglist[4];
	
   	// Set the widget attribute position and dimensions
   	XtArgVal leftPos = dialLayoutXOrg + iCol*dialWidthProportion;
	XtArgVal rightPos = leftPos + dialWidthProportion;
	XtArgVal topPos = dialLayoutYOrg + iRow*dialHeightProportion;
	XtArgVal bottomPos = topPos + dialHeightProportion;
   	XtSetArg( arglist[i], XmNleftPosition, leftPos ); i++; 
   	XtSetArg( arglist[i], XmNrightPosition, rightPos ); i++;  
   	XtSetArg( arglist[i], XmNtopPosition, topPos ); i++;
   	XtSetArg( arglist[i], XmNbottomPosition, bottomPos ); i++;
	
   	Widget newDial = XtCreateWidget( newWidgetName, dialWidgetClass, 
					 widget, arglist, i );
	addCallback( newDial, XtNselect, &Callback::callback1, this );
   	return( newDial );
}

void 
DialFormWrapper::_findDialsFormPosition( int nColElements, int nRowElements,
			  int *dialLayoutXOrg, int *dialLayoutYOrg,
			  int *dialWidthProportion, int *dialHeightProportion )
{
        int iFractionBase, lLeft, widthP, lBottom, lTop, heightP;
	getDialFormAttr( XmNfractionBase, (XtArgVal) &iFractionBase );

	if ( labels.isALabel( L_LEFT_VERT) ) {
           lLeft = labels.getLabelHeight( L_LEFT_VERT );
	   widthP = (int)( (iFractionBase - (2 * lLeft) ) / nColElements );
	} else if ( ( nRowElements > 1 ) && ( formRows = 1 ) ) {
           // If resizing to an array, the labelling layout is fixed
	   lLeft = labels.getLabelHeight( L_LEFT_VERT );
	   widthP = (int)( ( iFractionBase - (2 * lLeft ) ) / nColElements );
	} else {
           lLeft = 0;
           widthP = (int)( iFractionBase/nColElements );
	}
	*dialLayoutXOrg = lLeft;
	*dialWidthProportion = widthP;

	if ( labels.isALabel( L_BOTTOM_HORIZ ) ) {
           lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
           lTop = labels.getLabelHeight( L_BOTTOM_HORIZ );
           heightP = (int)( ( iFractionBase - lBottom - lTop) / nRowElements );
	} else if ( ( nRowElements > 1 ) && ( formRows == 1 ) ) {
           lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
           lTop = labels.getLabelHeight( L_TOP_HORIZ );
	   heightP = (int)( (iFractionBase - lBottom - lTop) / nRowElements );
	} else {
           lTop = labels.getLabelHeight( L_TOP_HORIZ );
           heightP = (int)( (iFractionBase - lTop) / nRowElements );
	}
	*dialLayoutYOrg = labels.getLabelHeight( L_TOP_HORIZ );
	*dialHeightProportion = heightP;
}



void
DialFormWrapper::_genChildLayout( int nNewCols, int nNewRows )
{
        // Calculate the new size and create a new WidgetList to hold children
        int newSize = nNewRows * nNewCols;
        WidgetList newChildList = new Widget[ newSize ];

	// Figure out where dials should be in new display
	int dialLayoutXOrg, dialLayoutYOrg;
	int dialWidthProportion, dialHeightProportion;
	_findDialsFormPosition( nNewCols, nNewRows,
			       &dialLayoutXOrg, &dialLayoutYOrg,
			       &dialWidthProportion, &dialHeightProportion );

	int maxRow = max( formRows, nNewRows );
	int maxCol = max( formCols, nNewCols );

	int iRow, iCol;
	int iEleNew, iEleOld;
	char newWidgetName[MAX_WIDGET_NAME_LEN];

	for ( iRow = 0; iRow < maxRow; iRow++ ) {
            for ( iCol = 0; iCol<maxCol; iCol++ ) {
                
             // Find new and old positions (Note: both are not always valid)
	     iEleNew = iRow * nNewCols + iCol;
	     iEleOld = iRow * formCols + iCol;
	     
	     if ( (iRow < nNewRows) && (iCol < nNewCols) ) {
                // This (iRow,iCol) is in new display
	        if ( (iRow < formRows) && (iCol < formCols)) {
                   // Also in Old display == reuse widget
		   _setADialFormPosition( iCol, iRow,
			     dialLayoutXOrg, dialLayoutYOrg,
			     dialWidthProportion, dialHeightProportion );
		   newChildList[ iEleNew ] = childList[ iEleOld ];
		} else {
                   // Not in old display -- create new dial widget
                   sprintf( newWidgetName, "Dial:%d,%d", iRow, iCol );
		   newChildList[ iEleNew ] = _createChild( newWidgetName,
				 iCol, iRow, 
				 dialLayoutXOrg, dialLayoutYOrg,
				 dialWidthProportion, dialHeightProportion );
	        }
	     } else {
                // This (iRow, iCol) is not in new dislay
                if ( (iRow < formRows) && (iCol < formCols) ) {
                   // This (iRow, iCol) in old display; destroy widget
		   XtDestroyWidget( childList[iEleOld] );
		}
	     }
	  }
       }
       // The new list is now complete.  Clear out old space/ adjust ptrs
       delete [] childList;
       childList = newChildList;
       formRows = nNewRows;
       formCols = nNewCols;
}


void
DialFormWrapper::_setADialFormPosition( int iCol, int iRow,
			   int dialLayoutXOrg, int dialLayoutYOrg,
			   int dialWidthProportion, int dialHeightProportion )
{
        Arg arglist[4];
	XtSetArg( arglist[0], XmNleftPosition,
	          (XtArgVal)( dialLayoutXOrg + iCol*dialWidthProportion ) ); 
     	XtSetArg( arglist[1], XmNrightPosition,
		 (XtArgVal)( dialLayoutXOrg + (iCol+1)*dialWidthProportion ) );
     	XtSetArg( arglist[2], XmNtopPosition, 
		  (XtArgVal)( dialLayoutYOrg + iRow*dialHeightProportion ) );
     	XtSetArg( arglist[3], XmNbottomPosition, 
	        (XtArgVal)( dialLayoutYOrg + (iRow+1)*dialHeightProportion ) );

     	int iEle = iRow * formCols + iCol;
     	XtSetValues( childList[iEle], arglist, 4 );
}


void 
DialFormWrapper::callback1( Widget callbackWidget, XtPointer /* ptr1 */,
			    XtPointer /* ptr2 */ )
{
        int iRow, iCol;
	sscanf( XtName(callbackWidget), "Dial:%d,%d", &iRow, &iCol );
	if ( (iRow >= 0 ) && ( iCol >= 0 ) ) {
           dialFUthisPtr->fuCallback( iRow, iCol );
	}
}

void
DialFormWrapper::getDialAttr( String name, XtArgVal value ) const
{
        Arg arg;
        XtSetArg( arg, name, value );
        XtGetValues( childList[0], &arg, 1 );
}

void
DialFormWrapper::getDialFormAttr( String name, XtArgVal value ) const
{
        Arg arg;
        XtSetArg( arg, name, value );
        XtGetValues( widget, &arg, 1 );
}

void 
DialFormWrapper::getDialLabel( int whichLabel,
			       const char **aLabel ) const
{
        labels.getLabel( whichLabel, aLabel );
}


void
DialFormWrapper::setDialAttr( String name, XtArgVal value )
{
        Arg arg;
        XtSetArg( arg, name, value );

        for ( int i = 0; i < (formCols * formRows); i++ ) {
            XtSetValues( childList[i], &arg, 1 );
        }
}

void
DialFormWrapper::setDialFormAttr( String name, XtArgVal value )
{
        Arg arg;
        XtSetArg( arg, name, value );
        XtSetValues( widget, &arg, 1 );
}


void 
DialFormWrapper::setDialLabel( double fuMin, double fuMax )
{
        labels.setLabel( fuMin, fuMax );
}


void 
DialFormWrapper::setDialLabel( int whichLabel, const char *stringLabel )
{
     	Boolean_ layoutChange; 
     	if ( whichLabel == L_LEFT_VERT ) {
            layoutChange = labels.setLabel( widget, "DialLabelVert",
					    whichLabel, stringLabel );
     	} else if ( whichLabel == L_BOTTOM_HORIZ ) {
            layoutChange = labels.setLabel( widget, "DialLabelHoriz",
					    whichLabel, stringLabel );
     	}

     	if ( layoutChange && ( formRows == 1 ) ) {
            int dialLayoutXOrg, dialLayoutYOrg;
            int dialWidthProportion, dialHeightProportion;
            _findDialsFormPosition( formCols, formRows,
     		                 &dialLayoutXOrg, &dialLayoutYOrg,
			         &dialWidthProportion, &dialHeightProportion );
            for ( int iCol=0; iCol < formCols; iCol++ ) {
                for ( int iRow=0; iRow < formRows; iRow++ ) {
                    _setADialFormPosition( iCol, iRow, dialLayoutXOrg, 
					  dialLayoutYOrg, dialWidthProportion, 
					  dialHeightProportion );
	        }
            }
      	}
}

        
void
DialFormWrapper::setDisplayValues( int value )
{
        Assert( (formRows == 1) && (formCols == 1) );

        DialSetValue( (DialWidget)childList[0], value );
}

void
DialFormWrapper::setDisplayValues( int nCols, int *values )
{
        if ( ( nCols != formCols ) || ( formRows != 1 ) ) {
            _adjustLayout( nCols, 1 );
        }

        int i;
        for ( i = 0; i < nCols; i++ ) {
            DialSetValue( (DialWidget)childList[i], values[i] );
        }
}

void
DialFormWrapper::setDisplayValues( int nCols, int nRows, int *values )
{
        if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
            _adjustLayout( nCols, nRows );
        }

        int nDials = nCols * nRows;
        int i;
        for ( i = 0; i < nDials; i++ ) {
            DialSetValue( (DialWidget)childList[i], values[i] );
        }
}

void
DialFormWrapper::setSingleValue( int iCol, int iRow, int value )
{
        if ( ( iCol >= formCols ) || ( iRow >= formRows ) ) {
            warning( "setSingleValue: Index (%d, %d) out of range: (%d, %d)",
                      iCol, iRow, formCols-1, formRows-1 );
            return;
        }

        int iEle = ( iRow * formCols ) + iCol;
        DialSetValue( (DialWidget)childList[iEle], value );
}

void
DialFormWrapper::updateDisplayLayout( int nCols, int nRows )
{
        if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
            _adjustLayout( nCols, nRows );
        }
}

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const DialFormWrapper::MY_CLASS = "DialFormWrapper";
