/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *               University of Illinois at Urbana-Champaign
 *               Department of Computer Science
 *               1304 W. Springfield Avenue
 *               Urbana, IL     61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *      All Rights Reserved.
 *
 * Author: Keith A. Shields (shields@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * 
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Help.C: Implements Help.
 *
 */
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/PanedW.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/List.h>
#include <Xm/PushB.h>

#include "Assert.h"
#include "Help.h"
#include "SystemErrors.h"

Help::Help( Widget parent, char *startdirectory, char *filename )
{  
	_setClassName( MY_CLASS );
   	doesDialogExist = FALSE_;

	baseHelpDir = new char[ strlen(startdirectory) + 1 ];
	if ( baseHelpDir == NULL ) {
	    abort( "Can't allocate space for baseHelpDir" );
	}
   	strcpy( baseHelpDir, startdirectory );

	currentDir = new char[ strlen(startdirectory) + 1 ];
	if ( currentDir == NULL ) {
	    abort( "Can't allocate space for currentDir" );
	}
   	strcpy( currentDir, startdirectory );

	helpFile = new char[ strlen(filename) + 1 ];
	if ( helpFile == NULL ) {
	    abort( "Can't allocate space for helpFile" );
	}
   	strcpy( helpFile, filename );

   	HelpDialog = XtCreatePopupShell( "Help", 
					topLevelShellWidgetClass, 
				 	parent, 
					NULL, 0 );
   
   	addCallback( HelpDialog, 
		     XmNdestroyCallback, &Callback::callback2,
                     this, NULL );

   	/* 
	 * Manager for help dialog - divided into action and control areas
      	 * as required for dialogs
   	 */
   	HelpPane = XtVaCreateWidget( "pane", 
					xmPanedWindowWidgetClass, 
               			        HelpDialog, 
				        XmNsashWidth, 1, 
				        XmNsashHeight, 1, 
				        NULL );

   	ControlArea = XtCreateWidget( "controlArea", 
					xmFormWidgetClass,
                		        HelpPane, 
				        NULL, 0 );

   	ActionArea = XtCreateWidget( "actionArea", 
					xmFormWidgetClass,
                		        HelpPane, 
					NULL, 0 );

   	_initializeControlWidgets();
   	_initializeActionWidgets();

   	if ( _getHelpText( baseHelpDir ) ) {
     	    _getDetailList( baseHelpDir );

     	    XtManageChild( ActionArea );
     	    XtManageChild( ControlArea );
     	    _setInitialWidth();
     	    XtManageChild( HelpPane );

	    doesDialogExist = TRUE_;
     	    XtPopup( HelpDialog, XtGrabNone );
   	}
}    

Help::~Help()
{
	/* 
	 * The work of freeing space and widgets is mostly done in callback2
	 * when the OK button is pressed.
	 */
}

void 
Help::_initializeActionWidgets()
{
   	/* 
	 * These are the dialog action buttons.  The top and previous
         * buttons are initially de-sensitized since it doesn't make sense to
         * do a top or previous operation from the top level of the hierarchy.
         */
   	okButton = XtCreateManagedWidget( "okButton", 
					xmPushButtonWidgetClass,
               				ActionArea,
					NULL, 0 );
   
   	topButton = XtVaCreateManagedWidget( "topButton", 
					xmPushButtonWidgetClass,
               				ActionArea,
					XmNsensitive, False, 
					NULL );

   	previousButton = XtVaCreateManagedWidget( "previousButton", 
                     			xmPushButtonWidgetClass, 
					ActionArea, 
					XmNsensitive, False, 
					NULL );

      	addCallback( okButton,
		     XmNactivateCallback, &Callback::callback2,
                     this, NULL);

   	addCallback( topButton,
		     XmNactivateCallback, &Callback::callback3,
                     this, NULL );

   	addCallback( previousButton,
		     XmNactivateCallback, &Callback::callback4,
               	     this, NULL );

   	/* 
	 * Since Motif action areas are never supposed to change size, we fix
         * the size of the pane the buttons are in.
         */
   	Dimension h; 
   	XtVaGetValues( okButton, XmNheight, &h, NULL );
   	XtVaSetValues( ActionArea, XmNpaneMaximum, h, XmNpaneMinimum, h, NULL );
}

void 
Help::_initializeControlWidgets()
{
   	CurrentTopicLabel = XtCreateManagedWidget( "currentTopicLabel", 
					xmLabelWidgetClass, 
					ControlArea, 
					NULL, 0 );

   	HelpText = XmCreateScrolledText( ControlArea, 
					"helpText", 
					NULL, 0 );

   	DetailLabel = XtCreateManagedWidget( "detailLabel", 
		 			xmLabelWidgetClass, 
					ControlArea, 
					NULL, 0 );

   	DetailList = XmCreateScrolledList( ControlArea, 
				       	"detail", 
					NULL, 0 );

   	XtVaSetValues( XtParent( HelpText ), 
		  	XmNtopWidget, CurrentTopicLabel,
		  	XmNbottomWidget, DetailLabel,
		  	NULL );
   
   	XtVaSetValues( DetailLabel,
		  	XmNbottomWidget, XtParent( DetailList ),
		  	NULL );

   	addCallback( DetailList, 
		     XmNbrowseSelectionCallback, &Callback::callback1,
                     this, NULL );

   	XtManageChild( DetailList );
   	XtManageChild( HelpText );
}

void
Help::_clearDetailList()
{
  	/* 
	 * Clear out current detail list.
	 */
	int numItems;
        XtVaGetValues( DetailList, XmNitemCount, &numItems, NULL );

	int i;
  	for ( i = 0; i < numItems; i++ ) {
	    Assert( detailDirList[i] != NULL );
	    delete detailDirList[i];
	    detailDirList[i] = NULL;
   	}

  	XmListDeleteAllItems( DetailList );
}

void 
Help::_getDetailList( const char *directory )
{ 
  	DIR *helpdir;
	if ( ( helpdir = opendir( directory ) ) == NULL ) {
      	    warning( "Cannot open help directory %s (%s)", directory, 
							   errorString() );
            return;
        }

	_clearDetailList();

  	/* 
	 * Go through all entries in the directory and get detail text for
         * any of those that are themselves directories except for . and ..
	 * _getDetailText() updates detailDirList.
         * The help is arranged in a hierarchical manner, so detail information
         * for the help in a given directory will be found in subdirectories.
         * This makes it easy to add/remove help topics and detail information
         */
	dirent *direntry;
	char buffer[1024];
	struct stat fileStatus;

        while ( ( direntry = readdir( helpdir ) ) != NULL ) {
	    sprintf( buffer, "%s/%s", directory, direntry->d_name );

    	    if ( stat( buffer, &fileStatus ) == -1) {
      		warning( "Cannot Access %s", buffer );
                return;
    	    }

	    if ( ( ( fileStatus.st_mode & S_IFMT ) == S_IFDIR ) 
		    && ( strcmp( direntry->d_name, "." ) != 0 ) 
	            && ( strcmp( direntry->d_name, ".." ) != 0 ) ) {
                _getDetailText( buffer );
	    }
       }

       closedir( helpdir );
}

void 
Help::_getDetailText( const char *directory )
{  
	FILE *fp;
	char buffer[1024];

	sprintf( buffer, "%s/%s", directory, helpFile );
        if ( ( fp = fopen( buffer, "r") ) == NULL) {
    	    warning( "Can't open help file %s (%s)", buffer, errorString() );
            return;
        }

  	/* 
	 * If can get into the detail directory and can open the help file,
         * use the first line of the file as the detail information that will
         * be shown on the detail list
         */
  	fscanf( fp,"%[^\n]\n", buffer );
  	fclose( fp );

  	int numItems;
  	XtVaGetValues( DetailList, XmNitemCount, &numItems, NULL );
	if ( numItems >= MAX_DETAIL_DIRS ) {
	    warning( "Too many entries in Detail List - Max of %d", 
							MAX_DETAIL_DIRS );
	    return;
	}

  	detailDirList[ numItems ] = new char[ strlen(directory) + 1 ];
	if ( detailDirList[ numItems] == NULL ) {
	    abort( "Can't allocate space for detailDirList entry" );
	}
        strcpy( detailDirList[numItems], directory );

	XmString xmstr = XmStringCreateSimple( buffer );
        XmListAddItemUnselected( DetailList, xmstr, 0 );
	XmStringFree( xmstr );
}

Boolean_ 
Help::_getHelpText( const char *directory )
{     
	char buffer[1024];
	FILE *fp;

	sprintf( buffer, "%s/%s", directory, helpFile );
	if ( ( fp = fopen( buffer, "r") ) == NULL) {
	    warning( "Cannot open help file %s (%s)", buffer, errorString() ); 
      	    return (FALSE_);
   	}

	struct stat fileStatus;
        if ( stat( buffer, &fileStatus ) == -1) {
     	    warning( "Cannot access %s", buffer );
            return ( FALSE_ );
	}

   	/* First line of the file is used as the topic. */
   	fscanf( fp, "%[^\n]\n", buffer );
	XmString xmstr = XmStringCreateSimple( buffer );
        XtVaSetValues( CurrentTopicLabel, XmNlabelString, xmstr, NULL );
	XmStringFree( xmstr );

   	/* 
	 * Figure out how much of the file is still left to read (have to
         * deduct the first line which was read as the help topic) and
         * allocate the proper amount of space.  Read in the help text,
         * null terminate the string, and set up the text widget.
         */
	int blen = strlen( buffer ) + 1;
        int amountToRead = (int)fileStatus.st_size - 1 - blen;
   	char *text = new char[ amountToRead + 1 ];
	if ( text == NULL ) {
	    abort( "Can't allocate space for text" );
	}

   	fread( text, sizeof(char), amountToRead, fp );
        text[ amountToRead - 1 ] = '\0';
        XmTextSetString( HelpText, text );

	delete[] text;
   	fclose( fp );
   	return ( TRUE_ );
}

void 
Help::_setInitialWidth() 
{
	char *theText;
   	if ( theText = XmTextGetString( HelpText ) ) {
   	    XmFontList fontList;
	    Dimension marginWidth;

     	    XtVaGetValues( HelpText, XmNfontList, &fontList, NULL );
     	    XtVaGetValues( HelpText, XmNmarginWidth, &marginWidth, NULL);

     	    XmString tempXStr = XmStringCreateLtoR( theText, 
						    XmSTRING_DEFAULT_CHARSET );
     	    Dimension maxWidth = XmStringWidth( fontList, tempXStr );

     	    XtVaSetValues( HelpText, 
			   XmNwidth, maxWidth +( 2*marginWidth ) + 10, 
			   NULL );

	    XmStringFree( tempXStr );
	    XtFree( theText );
   	}
}

void 
Help::callback1( Widget, XtPointer, XtPointer callData )
{
   	/*
	 * An item on the detail list has been selected, so we now treat 
	 * that item's directory as the current help directory 
	 */

   	XmListCallbackStruct *listdata = (XmListCallbackStruct*)callData;
   	int pos = listdata->item_position;

	delete currentDir;
	currentDir = new char[ strlen(detailDirList[pos-1]) + 1 ];
	if ( currentDir == NULL ) {
	    abort( "Can't allocate space for new currentDir" );
	}
   	strcpy( currentDir, detailDirList[pos-1] );

   	_getHelpText( currentDir );
   	_getDetailList( currentDir );

   	/* 
	 * We've gone down at least one level in the hierarchy, so it makes
      	 * sense to allow top and previous operations now
  	 */
   	XtVaSetValues( topButton, XmNsensitive, True, NULL );
   	XtVaSetValues( previousButton, XmNsensitive, True, NULL );
}

void 
Help::callback2( Widget, XtPointer, XtPointer )
{ 
	/* 
	 * This callback is activated when OK is selected or shell is closed.
	 * 
   	 * Widget destruction will automatically clean up callbacks for X, but
    	 * the callback class has internal data structures that need to be 
	 * updated as well, or the destructor for the callback class will fail. 
	 * The easiest way to fix this is to remove all the callbacks before 
	 * destroying the widget, thus keeping those internal lists accurate.
	 */
	_clearDetailList();

   	removeCallback( HelpDialog, XmNdestroyCallback, 
			&Callback::callback2, this, NULL );

   	removeCallback( DetailList, XmNbrowseSelectionCallback, 
			&Callback::callback1, this, NULL );

   	removeCallback( okButton, XmNactivateCallback, 
			&Callback::callback2, this, NULL );

   	removeCallback( topButton, XmNactivateCallback,
			&Callback::callback3, this, NULL );

   	removeCallback( previousButton, XmNactivateCallback,
			&Callback::callback4, this, NULL );

   	XtDestroyWidget( HelpDialog );

	delete helpFile;
	delete currentDir;
	delete baseHelpDir;

   	doesDialogExist = FALSE_;
}

void 
Help::callback3( Widget, XtPointer, XtPointer )
{
	/* 
	 * Top button was pressed, so go back to entry point in hierarchy.
	 */
	delete currentDir;
	currentDir = new char[ strlen(baseHelpDir) + 1 ];
	if ( currentDir == NULL ) {
	    abort( "Can't allocate space for top currentDir" );
	}
   	strcpy( currentDir, baseHelpDir);

   	_getHelpText( currentDir );
   	_getDetailList( currentDir );

   	/* 
	 * Can't do a top or previous from the top level.
	 */
   	XtVaSetValues( topButton, XmNsensitive, False, NULL );
   	XtVaSetValues( previousButton, XmNsensitive, False, NULL );
}

void 
Help::callback4( Widget, XtPointer, XtPointer )
{  
	/*
	 * Previous button was pressed so go back a level in the hierarchy.
         * This is done by getting rid of everything behind the last backslash
         * in the current help directory, effectively going up a level.  We
	 * just leave our currentDir string "where it is" and put in a null
	 * in place of the /. Checks are done to ensure we don't go past 
	 * the top level.
   	 */
	char *lastSlash = strrchr( currentDir, '/' );

   	if ( (lastSlash != NULL) && (strcmp( currentDir, baseHelpDir ) != 0) ) {
	    *lastSlash = '\0';
            _getHelpText( currentDir );
            _getDetailList( currentDir );
	}

   	/*
	 * Can't do a top or previous if we're now at the top level 
	 */
   	if ( strcmp( currentDir, baseHelpDir ) == 0) {
       	    XtVaSetValues( topButton, XmNsensitive, False, NULL );
       	    XtVaSetValues( previousButton, XmNsensitive, False, NULL );
   	}
}

Boolean_ 
Help::dialogExists() const
{
  	return( doesDialogExist );
}

/*
 *      Initialize the static data.  
 */
const char *const Help::MY_CLASS = "Help";


