/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InputPipeSocket.cc: Implements InputPipeSocket class.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/InputPipeSocket.C,v 1.22 1994/03/15 19:38:55 aydt Exp $
 *	
 */
#include <string.h>

#include "InputPipeSocket.h"

#include "BinaryPipeReader.h"
#include "CString.h"
#include "FUWrapper.h"
#include "InputPort.h"
#include "ModuleId.h"
#include "RecordDictionary.h"
#include "SystemErrors.h"

InputPipeSocket::InputPipeSocket( StreamPipe *inPipe, FUWrapper *w, 
						      ModuleId *src )
{
	_setClassName( MY_CLASS );

	pipe = inPipe;
	wrapper = w;
	source = src;

	_objectName = NULL;
	pipeIndex = -1;
	reader = NULL;
	dictionary = NULL;

	inputPortsCnt = 0;
	inputPortsBound = NULL;

	outputFieldsCnt = 0;
	outputFieldsBound = NULL;
}

InputPipeSocket::~InputPipeSocket()
{
	if ( _objectName != NULL ) {
	    int len = strlen( _objectName ) + 1;
	    delete[] _objectName;
	}

	if ( reader != NULL ) {
	    delete reader;
	}

	if ( dictionary != NULL ) {
	    delete dictionary;
	}

	if ( inputPortsCnt != 0 ) {
	    delete[] inputPortsBound;
	}

	if ( outputFieldsCnt != 0 ) {
	    delete[] outputFieldsBound;
	}
}

void
InputPipeSocket::_doFieldExtraction()
{
	activeTag = currentHeader.tag;
	RecordDossier& inDossier = dictionary->fetch( activeTag );
	reader->getData( inDossier );
	ActionRecord& actionRecord = actionTable[ activeTag ];
	Assert( ActionDoesExtraction( actionRecord.action ) );

	int i, fld;
	Assert( actionRecord.inputCnt == inputPortsCnt );

	for ( i = 0; i < inputPortsCnt; i++ ) {
	    fld = actionRecord.fieldForInput[i];
	    Assert( fld != -1 );
	    inputPortsBound[i]->setValueP( inDossier.getValueP( fld ), fld );
	}

	if ( outputFieldsCnt != 0 ) {
	    RecordDossier *outDossier = wrapper->getDossier();
	    Assert( actionRecord.outputCnt == outputFieldsCnt );

	    for ( i = 0; i < outputFieldsCnt; i++ ) {
		int outfld = outputFieldsBound[i];
		fld = actionRecord.fieldForOutput[i];
	        Assert( fld != -1 );
		outDossier->setValue( outfld, inDossier.getValue( fld ) );
	    }
	}
	    
}

void
InputPipeSocket::_readDescriptor()
{
	static StructureDescriptor descr;	// Reuse descr throughout

	reader->getDescriptor( descr );
	if ( dictionary->insert( currentHeader.tag, descr ) ) {
	    actionTable[currentHeader.tag]._reset( ActionDiscard );
	    newRecordType = TRUE_;
	} else {
	    Assert( ActionIsSet( getAction( currentHeader.tag ) ) );
	}
}

void
InputPipeSocket::bindFieldToInputPort( int tag, int fieldId, 
						const CString& inputPortName )
{
	Assert( tag < actionTable.numEntries() );

	for ( int i = 0; i < inputPortsCnt; i++ ) {
	    if ( inputPortName == inputPortsBound[i]->getName() ) {
		actionTable[tag]._bindToInputPort( fieldId, i );
		return;
	    }
	}

	abort( "Input Port %s not bound to socket %d\n", 
			inputPortName.getValue(), pipeIndex );
}
	
void
InputPipeSocket::bindFieldToOutputField( int tag, int fieldId, 
				 	 const CString& outputFieldName )
{
	Assert( tag < actionTable.numEntries() );

	FieldDescriptor *FD;
	RecordDossier   *RD = wrapper->getDossier();
	Assert( RD != 0 );

	for ( int i = 0; i < outputFieldsCnt; i++ ) {
	    FD = RD->getField( outputFieldsBound[i] );
	    Assert( FD != NULL );
	    if ( outputFieldName == FD->getName() ) {
		actionTable[tag]._bindToOutputField( fieldId, i );
		return;
	    }
	}

	abort( "Output Field %s not bound to socket %d\n", 
			outputFieldName.getValue(), pipeIndex );
}

Boolean_
InputPipeSocket::bindingToInputActive( InputPort *inPort, int fieldNum ) 
{
	/*
	 * If no active tag yet, then binding is not active
	 */
	if ( activeTag == -1 ) {
	    return FALSE_;
	}

	/*
	 * If the record we have marked as active is not marked for 
	 * fieldExtraction then binding is not active.
	 */
	if ( ! ActionDoesExtraction( actionTable[activeTag].action )  ) {
	    return FALSE_;
	}

	/*
	 * Find input port in the list of ones we fill - if it's not
	 * there binding is not active (this shouldn't really happen
	 * at this point.  We expect to find it.   See if the field
	 * satisfying the input port is the one we expect.  If it isn't
	 * then binding isn't active.  Else it is.
	 */
	for ( int i = 0; i < inputPortsCnt; i++ ) {
	    if ( inputPortsBound[i] == inPort ) {
		if ( actionTable[activeTag].fieldForInput[i] != fieldNum ) {
		    return FALSE_;
		} else {
		    return TRUE_;
		}
	    }
	}
	return FALSE_;		// don"t expect this here!!
}

void 
InputPipeSocket::configure( Boolean_ singleModule )
{
	/* 
	 * If we have an unprocessed data packet in our currentHeader then
	 * return... This should only happen when reconfiguring a single
	 * module.
	 */
	if ( dataPending ) {
	    Assert( singleModule == TRUE_ );
	    return;
	}

	/*
	 * Read packets from the input pipe until the pipe is empty or 
	 * a data packet is seen.  Attributes and Commands are passed
	 * through.  New descriptors are added to the dictionary.  
	 * A data packet marks the end of the configuration phase.  
	 * If we have an empty pipe and singleModule is true, or if
	 * we have an empty dictionary, we flush our upstream module's 
	 * dictionary if we haven't seen any descriptors during the run
	 * - this module may have been added late.
	 */
	Boolean_ continueDoLoop = TRUE_;
	Boolean_ flushUpstream = 
		CnvToBoolean_( singleModule || (dictionary->entryCount()==0) ); 

	do {
	    currentHeader = reader->getPacketHeader();

	    switch ( currentHeader.type ) {
	      case PIPE_EMPTY:
		  if ( flushUpstream ) {		
		      source->getWrapper()->flushOutputDictionary();
		      flushUpstream = FALSE_;	// switch off else infinite loop
		  } else {
		      continueDoLoop = FALSE_;
		  }
		  break;

	      case PKT_ATTRIBUTE:
	      case PKT_COMMAND:
	          wrapper->writePacket( currentHeader, 
					 reader->getPacketDataPtr() );
	   	  break;

	      case PKT_DESCRIPTOR:
		  flushUpstream = FALSE_;
		  _readDescriptor();
	   	  break;

	      case PKT_DATA:
	   	  dataPending = TRUE_;
	   	  continueDoLoop = FALSE_;
	   	  break;

	      case PKT_UNKNOWN:
	      default:
	          warning( "configure: Unknown packet type with tag=%d\n",
		           currentHeader.tag);
	          break;
	    }

	} while ( continueDoLoop );

	newRecordType = FALSE_;
}

int
InputPipeSocket::entriesInDictionary( ) const
{
	return dictionary->entryCount();
}

ActionType
InputPipeSocket::getAction( int recordTag ) 
{
	return actionTable[recordTag].action;
}

const RecordDossier&
InputPipeSocket::getActiveRecord() const
{
	return dictionary->fetch( activeTag );
}

int 
InputPipeSocket::getFieldToInputPort( int recordTag, int inputPortIdx ) 
{
	Assert( inputPortIdx < inputPortsCnt );
	Assert( ActionDoesExtraction( actionTable[recordTag].action ) );

	return actionTable[recordTag].fieldForInput[inputPortIdx];
}

int 
InputPipeSocket::getFieldToOutputField( int recordTag, 
					int outputFieldIdx ) 
{
	Assert( outputFieldIdx < outputFieldsCnt );
	Assert( ActionDoesExtraction( actionTable[recordTag].action ) );

	return actionTable[recordTag].fieldForOutput[outputFieldIdx];
}

int
InputPipeSocket::getInputPort( int inputPortBound ) const
{
	Assert( inputPortBound < inputPortsCnt );
	InputPort *ip = inputPortsBound[ inputPortBound ] ;
	int numInputPorts = wrapper->inputPortCount();

	for ( int i = 0; i < numInputPorts; i++ ) {
	    if ( ip == wrapper->getInputPort( i ) ) {
		return i;
	    }
	}

	abort( "Input port %d on pipe not in FUWrapper", inputPortBound,
							 pipeIndex );
	return -1;	// Should not get here
}

const char*					/* virtual */
InputPipeSocket::getObjectName() const
{
	if ( _objectName != NULL ) {
	    return _objectName;
	} else {
	    return getClassName();
	}
}

void 
InputPipeSocket::init( int thePipeIndex ) 
{
	if ( pipeIndex == -1 ) {
	    pipeIndex = thePipeIndex;

	    Assert( _objectName == NULL );
	    char buf[256];
	    sprintf( buf, "%s(%s)[%d]", getClassName(), 
					wrapper->getName().getValue(),
					pipeIndex );
	    _objectName = new char[ strlen( buf ) + 1 ];
	    strcpy( _objectName, buf );

	    Assert( reader == NULL );
	    reader = new BinaryPipeReader( pipe );

	    Assert( dictionary == NULL );
	    Assert( inputPortsBound == NULL && inputPortsCnt == 0 );
	    Assert( outputFieldsBound == NULL && outputFieldsCnt == 0 );
	} else {
	    if ( thePipeIndex != pipeIndex ) {
		abort( "Pipe index mismatch in init( %d ).  Expected %d.",
			thePipeIndex, pipeIndex );
	    }

	    delete dictionary;

	    if ( inputPortsCnt != 0 ) {
		delete[] inputPortsBound;
		inputPortsCnt = 0;
		inputPortsBound = NULL;
	    }

	    if ( outputFieldsCnt != 0 ) {
		delete[] outputFieldsBound;
		outputFieldsCnt = 0;
		outputFieldsBound = NULL;
	    }

	    actionTable.resetAllInvalid();
	}

	dictionary = new RecordDictionary;
	activeTag = -1;
	currentHeader = noHeader;
	dataPending = FALSE_;
	newRecordType = FALSE_;
}

Boolean_ 
InputPipeSocket::loadConfigurationFromDir( const CString& dir, int moduleIdx )
{
	/*
	 * init() has already been called from the FUWrapper so 
	 * we start with a 'clean slate'.
	 * 1) Call configure() to reload dictionary entries from
	 *    desCRIPTors on pipe.
	 * 2) Open the configuration file.  
	 * 3) Reload Input Ports and Output Fields bound.
	 * 4) Reload Action Table.
	 */

	configure( FALSE_ );		// Reload dictionary from pipe

	int i;
	static char buf[1024];
	sprintf( buf, "%s/module-%d.pipe-%d", dir.getValue(), moduleIdx,
					                      pipeIndex );
        FILE *fp = fopen( buf, "r" );
        if ( fp == NULL ) {
            error( "loadConfiguration::\nUnable to open for reading %s: %s", 
		          buf, errorString() );
            return( FAILURE_ );
        } else {
	    fscanf( fp, "%[^\n]\n", buf );          // Source module comment
	    fscanf( fp, "%[^\n]\n", buf );          // Input Ports comment
	    fscanf( fp, "%d\n", &inputPortsCnt );
	    if ( inputPortsCnt > 0 ) {
        	inputPortsBound = new InputPort* [ inputPortsCnt ];
	        for ( i = 0; i < inputPortsCnt; i++ ) {
		    fscanf( fp, "%[^\n]\n", buf );
		    inputPortsBound[i] = wrapper->getInputPort( buf );
		    Assert( inputPortsBound[i] != NULL );
		}
	    }

	    fscanf( fp, "%[^\n]\n", buf );          // Output Fields comment
	    fscanf( fp, "%d\n", &outputFieldsCnt );
	    if ( outputFieldsCnt > 0 ) {
        	outputFieldsBound = new int [ outputFieldsCnt ];
	        for ( i = 0; i < outputFieldsCnt; i++ ) {
		    fscanf( fp, "%d\n", &outputFieldsBound[i] );
		}
	    }

	    actionTable.loadConfiguration( fp );
	    fclose( fp );

	    /* set socket binding and constrain traits on input ports bound */
	    int a, f;
	    Boolean_ rc;

	    for ( i = 0; i < inputPortsCnt; i++ ) {
		inputPortsBound[i]->setSocketBinding( pipeIndex, this );
	    }

	    for ( a = 0; a < actionTable.numEntries(); a++ ) {
		ActionRecord& actionRecord = actionTable[a];
		if ( ActionDoesExtraction( actionRecord.action ) ) {
		    if ( ! dictionary->contains(a) ) {
			// Designed to catch input file w/ different record
			// types.
			abort( "%s expected a record type not found in input.",
						wrapper->getName().getValue() );
		    }
		    for ( i = 0; i < inputPortsCnt; i++ ) {
			f = actionRecord.fieldForInput[i];
			FieldDescriptor *fieldD = 
					      dictionary->fetch(a).getField(f);
			rc = inputPortsBound[i]->constrainTraits( 
							fieldD->getTraits() );
			Assert( rc );
		    }
		}
	    }

        }
	return SUCCESS_;
}

Boolean_ 
InputPipeSocket::processData()
{
	/*
	 * We may get to this point and not have any input ports
	 * that we are satisfying (only Copy Thru or Discard types),
	 * so we need to make sure we have input Ports before we
	 * check their status.  If we so, we see if our previous 
	 * value has been consumed.  If it hasn't, return without 
	 * reading any more packets.
	 */
	if ( inputPortsCnt > 0 ) {
	    if ( inputPortsBound[0]->valueStatus() == NEW_ )  {
	        return TRUE_;
	    }
	}

	/* 
	 * Process packets from the pipe until the pipe is empty or
	 * a Data Record with Action Extract is found.  Attributes and
	 * Commands are passed through.  Descriptors are added to the
	 * dictionary, and new entries set the newRecordType flag.
	 * If we see a Discard or CopyThru DATA type, and no Extract 
	 * following it, then mark our current input ports as invalid
	 * before returning.  (For asynch stuff)
	 */
	Boolean_ continueLooping = TRUE_;
	Boolean_ clearInputPorts = FALSE_;

        while ( continueLooping ) {

	    if ( dataPending ) {
	    	dataPending = FALSE_;
	    } else {
	    	currentHeader = reader->getPacketHeader();
	    }

	    switch ( currentHeader.type ) {
	      case PIPE_EMPTY:
		  continueLooping = FALSE_;
		  break;

	      case PKT_ATTRIBUTE:
	      case PKT_COMMAND:
	          wrapper->writePacket( currentHeader, 
					reader->getPacketDataPtr() );
	   	  break;

	      case PKT_DESCRIPTOR:
		  _readDescriptor();
	   	  break;

	      case PKT_DATA:
		  switch( getAction( currentHeader.tag ) ) {
		    case ActionDiscard:
		        clearInputPorts = TRUE_;
			break;
		   
		    case ActionCopyThru:
		        clearInputPorts = TRUE_;
			wrapper->writePacket( currentHeader,
					      reader->getPacketDataPtr() );
			break;
		
		    case ActionCopyThruAndExtract:
			wrapper->writePacket( currentHeader,
					      reader->getPacketDataPtr() );
			// and fall through to...

		    case ActionExtract:
	                Assert( inputPortsCnt > 0 );
		        clearInputPorts = FALSE_;
			_doFieldExtraction();
			continueLooping = FALSE_;
			break;
		    
		    default:
			error( "processData::invalid Action for Record Tag %d",
			       currentHeader.tag );
			break;
		  }
		  break;

	      case PKT_UNKNOWN:
	      default:
		  error( "processData: unknown packet type, tag = %d",
			 currentHeader.tag );
		  break;
	    }

	}

	if ( clearInputPorts ) {	// Last DATA was Discard or CopyThru
	    for ( int i = 0; i < inputPortsCnt; i++ ) {
		inputPortsBound[i]->clearValueP();
	    }
	}

	Assert( ( currentHeader.type == PIPE_EMPTY )  || 
		( currentHeader.type == PKT_DATA )  );

	return ( CnvToBoolean_(currentHeader.type == PKT_DATA) );

}

void
InputPipeSocket::resetConfiguration() {

	    if ( inputPortsCnt != 0 ) {
		delete[] inputPortsBound;
		inputPortsCnt = 0;
		inputPortsBound = NULL;
	    }

	    if ( outputFieldsCnt != 0 ) {
		delete[] outputFieldsBound;
		outputFieldsCnt = 0;
		outputFieldsBound = NULL;
	    }

	    actionTable.resetAllInvalid();
}

Boolean_ 
InputPipeSocket::saveConfigurationToDir( const CString& dir, 
					 int moduleIdx ) const
{
	int i;
	static char buf[1024];

	sprintf( buf, "%s/module-%d.pipe-%d", dir.getValue() , moduleIdx, 
						               pipeIndex );
        FILE *fp  = fopen( buf, "w" );
        if ( fp == NULL ) {
            error( "saveConfiguration::\nUnable to open for writing %s: %s", 
		          buf, errorString() );
	    return FAILURE_;
	} else {
	    fprintf( fp, "# Pipe from module %s\n", 
					    source->getName().getValue() );
	    fprintf( fp, "# Input Ports bound to Socket (count and names)\n" );
	    fprintf( fp, "%d\n", inputPortsCnt );
	    for ( i = 0; i < inputPortsCnt; i++ ) {
		fprintf( fp, "%s\n", inputPortsBound[i]->getName().getValue() );
	    }

	    fprintf( fp, 
		       "# Output Fields bound to Socket (count and indices)\n");
	    fprintf( fp, "%d\n", outputFieldsCnt );
	    for ( i = 0; i < outputFieldsCnt; i++ ) {
		fprintf( fp, "%d\n", outputFieldsBound[i] );
	    }

	    actionTable.saveConfiguration( fp );
            fclose( fp );
        }
	return SUCCESS_;

}

void
InputPipeSocket::setActionType( int recordTag, ActionType action )
{
	Assert( recordTag < actionTable.numEntries() );
	actionTable[recordTag]._reset( action, inputPortsCnt, outputFieldsCnt );
}

void
InputPipeSocket::setInputPorts( int count, int *inputPortIdxs )
{
	Assert( inputPortsCnt == 0 );
	Assert( inputPortsBound == NULL );

	if ( count > 0 ) {
	    inputPortsCnt = count;
	    inputPortsBound = new InputPort* [ inputPortsCnt ];
	    for ( int i = 0; i < count; i++ ) {
		inputPortsBound[i] = wrapper->getInputPort( inputPortIdxs[i] );
		Assert( inputPortsBound[i] != NULL );
		inputPortsBound[i]->setSocketBinding( pipeIndex, this );
	    }
	}
}

void
InputPipeSocket::setOutputFields( int count, int *outputFieldIdxs )
{
	Assert( outputFieldsCnt == 0 );
	Assert( outputFieldsBound == NULL );

	if ( count > 0 ) {
	    outputFieldsCnt = count;
	    outputFieldsBound = new int [ outputFieldsCnt ];
	    for ( int i = 0; i < count; i++ ) {
		outputFieldsBound[i] = outputFieldIdxs[i]; 
	    }
	}
}

void 			/* virtual */
InputPipeSocket::printOn( ostream& os ) const 
{
	os << form( "InputPipeSocket 0x%x: pipe=0x%x wrapper=0x%x\n",
		   	this, pipe, wrapper );
	os << form( "dataPending=%d  newRecordType=%d  activeTag = %d\n",
		   	dataPending, newRecordType, activeTag );
	os << form( "currentHeader: length=%d type=%d tag=%d\n",
		    	currentHeader.length, 
		    	currentHeader.type, 
		    	currentHeader.tag );

	os << "Dictionary: ";
	if ( dictionary ) {
		os << NL << *dictionary << NL;
	} else {
		os << "NONE\n";
	}

	int i;
        os << form( "%d Input Ports: ", inputPortsCnt );
	for ( i = 0; i < inputPortsCnt; i++ ) {
	    os << form( "%s ", inputPortsBound[i]->getName().getValue() );
	}
        os << form( "\n%d Output Fields: ", outputFieldsCnt );
	for ( i = 0; i < outputFieldsCnt; i++ ) {
	    os << form( "%d ", outputFieldsBound[i] );
	}
	os << NL;

	actionTable.printOn( os );
}

/*
 * Initialize the static data structure
 */
const char *const InputPipeSocket::MY_CLASS = "InputPipeSocket";

PacketHeader InputPipeSocket::noHeader = { 0, PIPE_EMPTY, 0 };
