/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FileOutputWrapper.cc: Writes packets seen on input pipe to output SDDF file.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/FileOutputWrapper.C,v 1.19 1994/03/15 00:16:40 aydt Exp $
 */

#include "FileOutputWrapper.h"

#include "AsciiPipeWriter.h"
#include "BinaryPipeReader.h"
#include "BinaryPipeWriter.h"
#include "ConversionPipeWriter.h"
#include "CStringObjList.h"
#include "FUParams.h"
#include "FileSelectionDialog.h"
#include "GeneralDialogBox.h"
#include "OutputFileStreamPipe.h"
#include "Pablo.h"
#include "PabloResources.h"
#include "ParamConfig.h"
#include "RecordDictionary.h"
#include "RecordDictionaryPIterator.h"
#include "StructureDictionary.h"


FileOutputWrapper::FileOutputWrapper( const CString& fileName ) 
		 : Wrapper( )
{
        _setClassName( MY_CLASS );
	filename = fileName;

	inputPipe = NULL;
	pipeReader = NULL;		
	outputPipeToFile = NULL;
	pipeWriter = NULL;
}

FileOutputWrapper::~FileOutputWrapper()
{
	if ( pipeWriter != NULL ) {
	    delete pipeWriter;
	}

	if ( outputPipeToFile != NULL ) {
	    delete outputPipeToFile;
	}

	if ( pipeReader != NULL ) {
	    delete pipeReader;
	}
}

Boolean_		/* virtual */
FileOutputWrapper::addInputPipe( StreamPipe *pipe, ModuleId *source )
{
	if ( inputPipe != NULL ) {		// pipe already connected
	    error( "Only one input is allowed per FileOutput module." );
	    return FAILURE_;
	}

	sourceModule = source;
	inputPipe = pipe;
	pipeReader = new BinaryPipeReader( inputPipe );
	return SUCCESS_;
}

Boolean_		/* virtual */
FileOutputWrapper::addOutputPipe( StreamPipe * /* pipe */ )
{
	error( "Cannot add an output pipe to a FileOutput module." );
	return FAILURE_;
}

void		  	/* virtual */
FileOutputWrapper::configure( Boolean_ singleModule )
{
	if ( inputPipe == NULL ) {		// wrong sequence
	    warning( "\nConnect input pipe to %s before configuring.",
			 getName().getValue() );
	    return;
	}

	if ( needsConfiguration ) {

	    /* 
	     * Create a FileSelectionDialog, set the title, match any files
	     * in the Data directory, restrict display to files only and 
	     * deactivate the Cancel button. Finally, if filename is not 
	     * NOMATCH, use it as default selection.
	     */

	    FileSelectionDialog *FSD = new FileSelectionDialog;
	    CString configPanelName = "Select Output File for " + getName();
            FSD->setXmStringResource( XmNdialogTitle, 
				      (char *)configPanelName.getValue() );
            CString dirMask = pabloData.defaultDataFileDir;
            dirMask = dirMask + "/*";
            FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
            FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_REGULAR );
	    FSD->setButtonSensitive( XmDIALOG_CANCEL_BUTTON, FALSE_ );
	    if ( filename != CString::NOMATCH ) {
                FSD->setXmStringResource( XmNdirSpec, 
					  (char *)filename.getValue() );
	    }

	    /*
	     * Ask user for file name via selection dialog until something valid
	     * is entered.
	     */
            Boolean done = FALSE_;
            while ( !done ) {
                FSD->run(); 			// Only okay button is active
                filename = FSD->getFileName();
	        outputPipeToFile = new OutputFileStreamPipe( 
							 filename.getValue() );
	        if ( outputPipeToFile->successfulOpen() ) {
		    done = TRUE_;
	        } else {
	    	    delete outputPipeToFile;
	    	    outputPipeToFile = NULL;	
	        }
            }
	    delete FSD;

	    /* 
	     * Determine type of file and attach the appropriate type 
	     * of PipeWriter.
	     */
	    outputType = BINARY_SDDF;		// Most common
	    Boolean noTypeGiven = TRUE_;

	    const char *file = filename.getValue();
	    char *suffix = strrchr( filename, '.' );
	    if ( suffix != NULL ) { 
		if ( strcmp( suffix+1, "ascii" ) == 0 ) {
		    outputType = ASCII_SDDF;
		    noTypeGiven = FALSE_;
		} else if ( strcmp( suffix+1, "binary" ) == 0 ) {
		    noTypeGiven = FALSE_;
		}
	    }

	    /* Ask what type they want */
	    if ( noTypeGiven ) { 
		CStringObjList	FileTypes;
		FileTypes.addElement( "Binary" );
		FileTypes.addElement( "Ascii" );
		FileTypes.addElement( "Binary; reverse byte order" );

		int typeId = -1;
		FUParams params;
		params.addRadioButtonsParam( "Output File Type",
					     BaseFUParamEntry::Integer, 0,
					     FileTypes );
		
		ParamConfig pc ( Pablo::TopLevel(), params, getName() );
		pc.run();
		BaseFUParamEntry& typeEntry = params.getEntry( 
							   "Output File Type" );
	 	if ( typeEntry.valueIsValid() ) {
		    typeId = typeEntry.getValue().getInteger();
		    if ( typeId == 1 ) {
		        outputType = ASCII_SDDF;
		    } else if ( typeId == 2 ) {
		        outputType = CONVERT_BINARY_SDDF;
		    }	
		    noTypeGiven = FALSE_;
		}
	    }

	    /* Choice was invalid for some reason - default to binary */
	    if ( noTypeGiven ) {
	       Pablo::GeneralDialog()->run(
		           "Defaulting to type binary for output file %s.",
			    file );
	    }

            if ( outputType == ASCII_SDDF ) { 
                pipeWriter = new AsciiPipeWriter( outputPipeToFile );
            } else if ( outputType == CONVERT_BINARY_SDDF ) { 
                pipeWriter = new ConversionPipeWriter( outputPipeToFile );
            } else {
                pipeWriter = new BinaryPipeWriter( outputPipeToFile );
            }

	    /* 
	     * Use filename in our object name for more detailed messages 
	     */
            char *s = strrchr( filename.getValue(), '/' );
            CString name;
            if ( s == NULL ) {
                    name = name + filename;
            } else {
                    s++;
                    name = name + s;
            }
            _setObjectName( name.getValue() );

	    if ( singleModule ) {
		sourceModule->getWrapper()->flushOutputDictionary();
	    }
	    needsConfiguration = FALSE_;

	} else {		
	    Assert( filename != CString::NOMATCH );
	    Pablo::GeneralDialog()->run( "Module %s output file is: %s", 
			            getName().getValue(), filename.getValue() );
	}
	
}

Boolean_		/* virtual */
FileOutputWrapper::loadConfigurationFromDir( const CString& /* dir */,
					     int /* moduleIndex */ )
{
        /* 
         * On second thought, I don't think we need an init() here. The order
         * in which this gets calls insures that the init() was done before
         * we got to this point (the wrapper was created & then init() called.
         * I'll just comment it out for now in case I'm wrong.... Ruth.
         */
        /* init(); */

	configure( FALSE_ );

	return SUCCESS_;
}

void			/* virtual */
FileOutputWrapper::init()
{
	Wrapper::init();
	outputDictionary = new RecordDictionary;

	if ( pipeWriter != NULL ) {
	    delete pipeWriter;
	    pipeWriter = NULL;
	}

	if ( outputPipeToFile != NULL ) {
	    delete outputPipeToFile;
	    outputPipeToFile = NULL;
	}

}

Boolean_ 		/* virtual */
FileOutputWrapper::ready()
{
	if ( inputPipe == NULL ) {
	    return FALSE_;
	}

	currentHeader = pipeReader->getPacketHeader();
	if ( currentHeader.type == PIPE_EMPTY ) {
	    return FALSE_;
	} else {
	    return TRUE_;
	}
}

void			/* virtual */
FileOutputWrapper::restart()
{
	/* 
	 * In the future we probably want to rework this so that you
	 * can just rewind the output file & not delete/new as is done
	 * here.		Ruth 6/93
	 */
	Assert( needsConfiguration == FALSE_ );
	Assert( pipeWriter != NULL );
	Assert( outputPipeToFile != NULL );

/*
	delete pipeWriter;
	delete outputPipeToFile;

	outputPipeToFile = new OutputFileStreamPipe( filename.getValue() );
        if ( outputType == ASCII_SDDF ) { 
            pipeWriter = new AsciiPipeWriter( outputPipeToFile );
        } else if ( outputType == CONVERT_BINARY_SDDF ) { 
            pipeWriter = new ConversionPipeWriter( outputPipeToFile );
        } else {
            pipeWriter = new BinaryPipeWriter( outputPipeToFile );
        }
*/
	pipeWriter->rewind();
}

void			/* virtual */
FileOutputWrapper::run( Boolean_& /* errorFlag */ )
{
	Attributes *Ap;
	StructureDescriptor *SDp;

	while ( currentHeader.type != PIPE_EMPTY ) {	// clear out pipe

	    if ( pipeWriter == NULL ) {			// no output file
	        ;
	    } else if ( outputType == BINARY_SDDF ) {	// both binary pipes
		switch( currentHeader.type ) {
                  case PKT_DESCRIPTOR:
                      SDp = new StructureDescriptor();
                      pipeReader->getDescriptor( *SDp );
                      if ( outputDictionary->insert(currentHeader.tag, *SDp) ) {
                          pipeWriter->putDescriptor( *SDp, currentHeader.tag );
		      }
                      delete SDp;
                      break;
		  case PKT_ATTRIBUTE:
                  case PKT_DATA:
                  case PKT_COMMAND:
	              pipeWriter->putPacket( currentHeader, 
				             pipeReader->getPacketDataPtr() );
		      break;
		}
	    } else {					// output is ascii file
                switch( currentHeader.type ) {
                  case PKT_ATTRIBUTE:
                      Ap = new Attributes();
                      pipeReader->getAttributes( *Ap );
                      pipeWriter->putAttributes( *Ap );
                      delete Ap;
                      break;
                  case PKT_DESCRIPTOR:
                      SDp = new StructureDescriptor();
                      pipeReader->getDescriptor( *SDp );
                      if ( outputDictionary->insert(currentHeader.tag, *SDp) ) {
                          pipeWriter->putDescriptor( *SDp, currentHeader.tag );
		      }
                      delete SDp;
                      break;
                  case PKT_DATA:
                      pipeReader->getData( outputDictionary->fetch( 
							  currentHeader.tag ) );
                      pipeWriter->putData( outputDictionary->fetch(
							  currentHeader.tag ) );
                      break;
                  case PKT_COMMAND:
                      pipeWriter->putCommand( currentHeader.tag );
                      break;
                }
            }
            currentHeader = pipeReader->getPacketHeader();
        }
}

void			/* virtual */
FileOutputWrapper::writeLayoutToFP( FILE *fp ) const
{
 	fprintf( fp, "ow %s %s\n", getName().getValue(), filename.getValue() );
}

void 
FileOutputWrapper::printOn( ostream& os ) const 
{
	os << "FileOutputWrapper\n";
}

/*
 * Initialize the static data
 */
const char *const FileOutputWrapper::MY_CLASS = "FileOutputWrapper";
