/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FileInputWrapper.C: Reads packets from input SDDF file and writes them to
 *			output pipe(s).
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/FileInputWrapper.C,v 1.27 1994/03/15 19:38:52 aydt Exp $
 */

#include "FileInputWrapper.h"

#include "AsciiPipeReader.h"
#include "BinaryPipeReader.h"
#include "CStringObjList.h"
#include "FUParams.h"
#include "FileSelectionDialog.h"
#include "GeneralDialogBox.h"
#include "InputFileStreamPipe.h"
#include "Pablo.h"
#include "PabloResources.h"
#include "RecordDictionary.h"
#include "RecordDossier.h"
#include "SystemErrors.h"

#include "BargraphFormWrapper.h"
#include "FormDialogWidget.h"
#include "LabelWidget.h"
#include "util.h"

FileInputWrapper::FileInputWrapper( const CString& fileName ) 
		: Wrapper( )
{
	_setClassName( MY_CLASS );

	inputPipeFromFile = NULL;
	pipeReader = NULL;
	filename = fileName;
	dossierCache = NULL;
	dialog = NULL;
	titleBar = NULL;
	barGraphForm = NULL;
	reportBytesRead = CnvToBoolean_(pabloData.displayBytesProcessed);
}

FileInputWrapper::~FileInputWrapper()
{
	if ( pipeReader != NULL ) {
	    delete pipeReader;
	}

	if ( inputPipeFromFile != NULL ) {
	    delete inputPipeFromFile;
	}

	if ( dossierCache != NULL ) {
	    delete[ ] dossierCache;
	}

	if ( reportBytesRead ) {
	    delete barGraphForm;
	    delete titleBar;
	    delete dialog;
	}
}

void 
FileInputWrapper::_processData()
{
	int sourceTag = currentHeader.tag;
	RecordDossier *dossierPtr = NULL;

	/*
	 * We look in our cache of RecordDossier pointers if the source tag 
	 * is within the range we are saving.  Update cache entry if it's 
	 * the first access.  Always look to master dictionaries if outside 
	 * cache range.
	 */
	if ( sourceTag < TAG_CACHE_SIZE ) {
	    dossierPtr = dossierCache[ sourceTag ];

	    if ( dossierPtr == NULL ) {
		int sysTag = Pablo::TagMappingDict()->fetch( moduleIdentifier, 
							     sourceTag );
		if ( sysTag == FAILURE_ ) {
	    	    abort( "Couldn't translate tag %d", sourceTag );
		} else {
		    RecordDossier& rd = outputDictionary->fetch( sysTag );
		    if ( rd.getName() == CString::NOMATCH ) {
		        abort( "No outputDictionary entry for tag %d", sysTag );
		    } else {
		        dossierPtr = &rd;
		        dossierCache[ sourceTag ] = dossierPtr;
		    }
	       }
	    }

	} else {
	    int sysTag = Pablo::TagMappingDict()->fetch( moduleIdentifier, 
							 sourceTag );
	    if ( sysTag == FAILURE_ ) {
	        abort( "Couldn't translate tag %d", sourceTag );
	    } else {
	        RecordDossier& rd = outputDictionary->fetch( sysTag );
	        if ( rd.getName() == CString::NOMATCH ) {
	            abort( "No outputDictionary entry for tag %d", sysTag );
		} else {
		    dossierPtr = &rd;
		}
	   }
	}

	pipeReader->getData( *dossierPtr );
	writeData( *dossierPtr );
}

void 
FileInputWrapper::_processDescriptor( StructureDescriptor &descr )
{
	/* 
	 * We use the index of our module and the tag from the packet header
	 * to resolve this (srcTag, srcId, descr). The Resolver does this
	 * for us - if this is a new combination, it will be entered in
	 * the Pablo TagMappingDictionary and in the Pablo StructureDictionary
	 * and a new systemTag will be assigned.  If it isn't new, we get
	 * back the already-assigned systemTag.  A FAILURE_ occurs if it's
	 * a duplicate structure name, but the structure doesn't match.
	 * We insert it in our outputDictionary and send it down the pipe
	 * unless it's a duplicate outputDictionary entry.
	 */
	int sourceTag = currentHeader.tag;
	int sysTag = resolver.determineTag( moduleIdentifier, sourceTag, descr);

	if ( sysTag == FAILURE_ ) {
	    abort( "Resolver returns FAILURE_ for Descriptor with name %s\n",
		   descr.getName().getValue() );
	}

	if ( outputDictionary->insert( sysTag, descr ) ) {
	    writeDescriptor( descr, sysTag );
	}
}


Boolean_                /* virtual */
FileInputWrapper::addInputPipe( StreamPipe * /* pipe */, 
				ModuleId * /* source */ ) 
{
	error( "Cannot add an input pipe to a FileInput module." );
	return FAILURE_;
}

void 			/* virtual */
FileInputWrapper::configure( Boolean_ /* singleModule */ )	
{
	if ( binaryOutput == NULL ) {		// wrong sequence of events
	    warning( "\nConnect output pipe to %s before configuring.",
		     getName().getValue() );
	    return;
	}

	if ( needsConfiguration ) {

	    /* 
	     * Create a FileSelectionDialog, set the title, match any files
	     * in the Data directory, restrict display to files only and 
	     * deactivate the Cancel button. Finally, if filename is not 
	     * NOMATCH, use it as default selection.
	     */

	    FileSelectionDialog *FSD = new FileSelectionDialog;
	    CString configPanelName = "Select Input File for " + getName();
            FSD->setXmStringResource( XmNdialogTitle, 
				      (char *)configPanelName.getValue() );
            CString dirMask = pabloData.defaultDataFileDir;
            dirMask = dirMask + "/*";
            FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
            FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_REGULAR );
	    FSD->setButtonSensitive( XmDIALOG_CANCEL_BUTTON, FALSE_ );
	    if ( filename != CString::NOMATCH ) {
                FSD->setXmStringResource( XmNdirSpec, 
		    			  (char *)filename.getValue() );
	    }

	    /*
	     * Ask user for file name via selection dialog until something valid
	     * is entered.
	     */
            Boolean done = FALSE_;
            while ( !done ) {
                FSD->run(); 			// Only okay button is active
                filename = FSD->getFileName();
	        inputPipeFromFile = new InputFileStreamPipe( 
							  filename.getValue() );
		PabloSddfType fileType = inputPipeFromFile->getInputType();
	        if ( ( fileType == ASCII_SDDF ) || 
		     ( fileType == BINARY_SDDF ) ||
		     ( fileType == CONVERT_BINARY_SDDF ) ) {
		    pipeReader = inputPipeFromFile->createPipeReader(); 
		    done = TRUE_;
	        } else {
		    if ( fileType == NOT_SDDF ) {
	       		info( "File %s is not an SDDF file\n", 
							filename.getValue() );
		    }
	    	    delete inputPipeFromFile;
	    	    inputPipeFromFile = NULL;	
	        }
            }
	    delete FSD;

	    /* 
	     * Use filename in our object name for more detailed messages 
	     */
            char *s = strrchr( filename.getValue(), '/' );
            CString name;
            if ( s == NULL ) {
                name = name + filename;
            } else {
                s++;
                name = name + s;
            }
            _setObjectName( name.getValue() );

	    /*
	     * Set label and maximum value on bargraph display if in use
	     */
	    if ( reportBytesRead ) {
	        char buf[256];
                off_t bytesInFile = inputPipeFromFile->getFileSize();
	        sprintf( buf, "%s (%d bytes)", name.getValue(), 
					       (int)bytesInFile );
	        titleBar->setStringValue( buf );
                barGraphForm->setBargraphAttr( "maxValue", 
					       (XtArgVal)bytesInFile );
	    }

	    /*
	     * Now we read the packets in the input file until it is 
	     * empty or until we see the first data packet.
	     */
	    currentHeader = pipeReader->getPacketHeader();

	    while( ( currentHeader.type != PIPE_EMPTY ) && 
		   ( currentHeader.type != PKT_DATA ) ) {

	        switch ( currentHeader.type ) {
	          case PKT_ATTRIBUTE:
		    {
		    Attributes *ap = new Attributes;
	            pipeReader->getAttributes( *ap );
		    writeAttributes( *ap );
		    delete ap;
		    }
	            break;
    
	          case PKT_DESCRIPTOR:
		    {
		    StructureDescriptor *sdp = new StructureDescriptor;   
	            pipeReader->getDescriptor( *sdp );
	            _processDescriptor( *sdp );
		    delete sdp;
		    }
	            break;

	          case PKT_COMMAND:
		    writeCommand( currentHeader.tag );
	            break;

	          case PKT_UNKNOWN:
	          default:
	            warning("\nUnknown packet type %d (length=%d tag=%d) in %s",
		         currentHeader.type, currentHeader.length, 
		         currentHeader.tag, getName().getValue() );
		    break;
	        }

		if ( reportBytesRead ) {
  	            barGraphForm->setDisplayValues( 
				     (int)inputPipeFromFile->getByteOffset() );
		}

	        currentHeader = pipeReader->getPacketHeader();
	    }

	    if ( currentHeader.type == PKT_DATA ) {
		dataPending = TRUE_;
	    }
	    needsConfiguration = FALSE_;

	} else {		

	    Assert( filename != CString::NOMATCH );
	    Pablo::GeneralDialog()->run( "Module %s input file is: %s", 
		              getName().getValue(), filename.getValue() );
	}

}

void 			/* virtual */
FileInputWrapper::init()	
{
	Wrapper::init();			

	outputDictionary = new RecordDictionary;

	if ( pipeReader != NULL ) {
	    delete pipeReader;
	    pipeReader = NULL;
	}

	if ( inputPipeFromFile != NULL ) {
	    delete inputPipeFromFile;
	    inputPipeFromFile = NULL;
	}

	currentHeader = noHeader;
	dataPending = FALSE_;

	if ( dossierCache == NULL ) {
	    dossierCache = new RecordDossier* [ TAG_CACHE_SIZE ];
	}

	for ( int i = 0; i < TAG_CACHE_SIZE; i++ ) {
	    dossierCache[i] = NULL;
	}

	if ( reportBytesRead && ( dialog == NULL ) ) {
       	    dialog = new FormDialogWidget( Pablo::TopLevel(), NullArgs, 
					   "FileInputDialog" );

	    Widget dialogWidget = dialog->getWidget();
	    XtVaSetValues( XtParent( dialogWidget ), 
			   XmNtitle, getName().getValue(),
			   NULL );

	    titleBar = new LabelWidget( dialog, NullArgs, "Title", 
					     getName().getValue() );
            titleBar->manage();

	    barGraphForm = new BargraphFormWrapper( dialog, NULL, NullArgs, 
							    "Contents" );
            barGraphForm->setBargraphLabel( L_BOTTOM_HORIZ, "bytes processed" );
	    dialog->manage();

	    /* 
	     * From here on, we don't want this dialog placed in the center 
	     * of the top level Pablo shell.
	     */
	    XtVaSetValues( dialogWidget, 
			   XmNdefaultPosition, False, 
			   NULL );
	}
}

Boolean_		/* virtual */
FileInputWrapper::loadConfigurationFromDir( const CString& dir,
					    int /* moduleIndex */ )
{ 
        /* 
         * On second thought, I don't think we need an init() here. The order
         * in which this gets calls insures that the init() was done before
         * we got to this point (the wrapper was created & then init() called.
         * I'll just comment it out for now in case I'm wrong.... Ruth.
         */
        /* init(); */

	/*
	 * If we have a bargraph display showing the bytes processed
	 * in the file, then we try to load the position and size of that 
	 * display. Otherwise, we don't even try to load the configuration.  
	 * We use the name that would typically be used for a Functional 
	 * Unit rather than a Wrapper, because the information is more "FUish".
	 */

	if ( reportBytesRead ) {
            static char buf[1024];
            sprintf( buf, "%s/%s", dir.getValue() , getName().getValue() );
            FILE *fp = fopen( buf, "r" );
            if ( fp != NULL ) {
	        int x, y, width, height;
            	fscanf( fp,"%*[^\n]\n" );		// comment line
            	if ( fscanf( fp, "%d %d %d %d\n", &x, &y, &width, 
							  &height ) == 4 ) {
  	            barGraphForm->setPerfWidgetPosition( x, y, width, height );
		}
 		fclose( fp );
	    }
        }

	configure( FALSE_ );
	return SUCCESS_;
}

Boolean_ 		/* virtual */
FileInputWrapper::ready()
{
	// No output pipes; can't run. (This could cause backed-up pipe)
	if ( binaryOutput == NULL ) {
	    return FALSE_;
	}

	// Configuration not done; can't run
	if ( needsConfiguration ) {
	    return FALSE_;
	}
	
	// Data packet waiting from configure()
	if ( dataPending ) {	
	    dataPending = FALSE_;
	    return TRUE_;
	}

	// Get next packet header; If none there, empty.  Else, ready to run.
	currentHeader = pipeReader->getPacketHeader();
	if ( currentHeader.type == PIPE_EMPTY ) {
	    if ( reportBytesRead ) {
  	        barGraphForm->setDisplayValues( 
			 	(int) inputPipeFromFile->getByteOffset() );
	    }
	    return FALSE_;
	} else {
	    return TRUE_;
	}
}

void			/* virtual */
FileInputWrapper::restart()
{
	Assert ( needsConfiguration == FALSE_ );
	Assert ( pipeReader != NULL );
	Assert ( inputPipeFromFile != NULL );

	currentHeader = noHeader;
	dataPending = FALSE_;

	if ( pabloData.restartReuseInfile ) {
	    pipeReader->rewind();
	} else {
	    needsConfiguration = TRUE_;
	    configure( TRUE_ );
	}

}

void			/* virtual */
FileInputWrapper::run( Boolean_& errorFlag )
{
	switch ( currentHeader.type ) {

	  case PIPE_EMPTY:		
	    break;

	  case PKT_ATTRIBUTE:
	    {
	    Attributes *ap = new Attributes;
	    pipeReader->getAttributes( *ap );
	    writeAttributes( *ap );
	    delete ap;
	    }
	    break;

	  case PKT_DESCRIPTOR:
	    {
            StructureDescriptor *sdp = new StructureDescriptor;   
	    pipeReader->getDescriptor( *sdp );
	    _processDescriptor( *sdp );
	    delete sdp;
	    }
	    break;

	  case PKT_DATA:
	    _processData();
	    break;

	  case PKT_COMMAND:
	    writeCommand( currentHeader.tag );
	    break;

	  case PKT_UNKNOWN:
	  default:
	    /* 
	     * We never expect to get here because the PipeReader
	     * skips bogus packets while reading the file in getPacketHeader()
	     * which is called from the ::ready method.  But... leave it
	     * in just in case.
	     */
	    warning( "\nUnknown packet type %d (length=%d tag=%d) in %s",
		     currentHeader.type, currentHeader.length, 
		     currentHeader.tag, getName().getValue() );
	    errorFlag = TRUE_;
	    break;
	}

	if ( reportBytesRead ) {
  	    barGraphForm->setDisplayValues( 
				(int) inputPipeFromFile->getByteOffset() );
	}
}

Boolean_			/* virtual */
FileInputWrapper::saveConfigurationToDir( const CString &dir, 
					  int /* moduleIndex */ ) const
{
	/*
	 * If we have the bargraph display showing the bytes processed
	 * in the file, then we save the position and size of that display.
	 * Otherwise, we save nothing.  We use the name that would
	 * typically be used for a Functional Unit rather than a Wrapper,
	 * because the information is more "FUish".
	 */

	Boolean_ result = SUCCESS_;

	if ( reportBytesRead ) {
            static char buf[1024];

            sprintf( buf, "%s/%s", dir.getValue() , getName().getValue() );
            FILE *fp = fopen( buf, "w" );
            if ( fp == NULL ) {
                warning( "saveConfigToDir::Unable to open %s: %s", buf,
                                                          errorString() );
		result = FAILURE_;
	    } else {
		/*
		 * For some reason I can't figure out, with mwm the coordinates
		 * are off by 8 in the X direction and 25 in the Y.  This
		 * doesn't happen with twm.  The mouse positions are "correct"
		 * but xwininfo reports them wrong too.  That means that
		 * the display "walks" across the screen as the configuration
		 * is saved/restored.
		 */
		int x, y, width, height;
  	        barGraphForm->getPerfWidgetPosition( &x, &y, &width, &height );

                fprintf( fp, "# Bargraph: X Y Width Height\n" );
                fprintf( fp, "%d %d %d %d\n", x, y, width, height );
 		fclose( fp );
	    }
        }
	return result;
}


void 			
FileInputWrapper::writeLayoutToFP( FILE *fp ) const	/* virtual */
{
	fprintf( fp, "iw %s %s\n", getName().getValue(), filename.getValue() );
}

void 		
FileInputWrapper::printOn( ostream& os ) const 		/* virtual */
{
	os << "------------------------------\n";
	os << form("FileInputWrapper(file=%s)\n", filename.getValue());
	os << "outputDictionary:\n";
	if ( outputDictionary != NULL ) {
	      outputDictionary->printOn( os );
	}
	os << "------------------------------\n";
}

/*
 * Initialize the static data 
 */
const char *const FileInputWrapper::MY_CLASS = "FileInputWrapper";
PacketHeader FileInputWrapper::noHeader = { 0, PIPE_EMPTY, 0 };
