/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ActionRecord.cc: Methods for the ActionRecord class.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/ActionRecord.C,v 1.8 1994/02/25 04:41:26 aydt Exp $
 */

#include "Assert.h" 
#include "ActionRecord.h"

ActionRecord::ActionRecord( )
{
	action = ActionInvalid;
	inputCnt = 0;	
	fieldForInput = NULL;
	outputCnt = 0;
	fieldForOutput = NULL;
}

ActionRecord::~ActionRecord()
{
	if ( inputCnt > 0 ) {
	    delete [] fieldForInput;
	}

	if ( outputCnt > 0 ) {
	    delete [] fieldForOutput;
	}
}

void
ActionRecord::_bindToInputPort( int fieldId, int inputPortIdx )
{
	Assert( ActionDoesExtraction( action ) );
	Assert( inputPortIdx < inputCnt );
	fieldForInput[inputPortIdx] = fieldId;
}

void
ActionRecord::_bindToOutputField( int fieldId, int outputFieldIdx )
{
	Assert( ActionDoesExtraction( action ) );
	Assert( outputFieldIdx < outputCnt );
	fieldForOutput[outputFieldIdx] = fieldId;
}

void
ActionRecord::_loadConfiguration( FILE *fp )
{
	ActionType act;
	int in, out, fld, i;

	fscanf( fp, "%*[^\n]\n" );          // Comment line
	fscanf( fp, "%d %d %d\n", &act, &in, &out );
	_reset( act, in, out );

	if ( in > 0 ) {
	    fscanf( fp, "%*[^\n]\n" );     // Comment line for Inputs
	    for ( i = 0; i < in; i++ ) {
	        fscanf( fp, "%d\n", &fld );
	        if ( i < inputCnt ) {	       // Expect True only if Extraction
		    fieldForInput[i] = fld;
	        }
	    }
	}
	    
	if ( out > 0 ) {				
	    fscanf( fp, "%*[^\n]\n" );   // Comment line for Outputs
	    for ( i = 0; i < out; i++ ) {
	        fscanf( fp, "%d\n", &fld );
	        if ( i < outputCnt ) {	       // Expect True only if Extraction
		    fieldForOutput[i] = fld;
	        }
	    }
	}
}

void
ActionRecord::_reset( ActionType type, int inputs, int outputs )
{
	/*
	 * Reset the action, input count, and output count.  The
	 * fieldForInput and fieldForOutput array cells are reset to -1's.
	 * If the action does not does not do Extraction, then the input and
	 * output counts will be set to 0 regardless of what the values
	 * are that it was called with.
	 */
	action = type;
	_setInputCount( inputs );		
	_setOutputCount( outputs );
}

void
ActionRecord::_saveConfiguration( FILE *fp ) const
{
	int i;

	fprintf( fp, "# Action ID (%s); Input Ports & Output Fields bound\n", 
	             _actionName() );				// comment
	fprintf( fp, "%d %d %d\n", action, inputCnt, outputCnt );

	if ( inputCnt > 0 ) {
	    fprintf( fp, "# Fields bound to Input Ports\n" );
	    for ( i = 0; i < inputCnt; i++ ) {
	        fprintf( fp, "%d\n", fieldForInput[i] );
	    }
	}
	if ( outputCnt > 0 ) {
	    fprintf( fp, "# Fields bound to Output Dossier Fields\n" );
	    for ( i = 0; i < outputCnt; i++ ) {
	        fprintf( fp, "%d\n", fieldForOutput[i] );
	    }
	}
}

void
ActionRecord::_setInputCount( int count )
{
	/*
	 * For Extraction types, set the count of input fields to the
	 * new value, initializing the fieldForInput array to all -1's.
	 * For Non-Extraction types, set inputCnt to zero and free space.
	 */
	
	if ( ActionDoesExtraction( action ) ) {	
	    if ( inputCnt != count ) {		   // space requirements changed

		if ( inputCnt != 0 ) {
		    delete [] fieldForInput;
		}

		if ( count > 0 ) {
		    fieldForInput = new int [ count ];
		    inputCnt = count;
		} else {
		    inputCnt = 0;
		    fieldForInput = NULL;
		}
	    }
		
	    for ( int i = 0; i < inputCnt; i++ ) {  // initialize elements to -1
		fieldForInput[i] = -1;
	    }
	} else {
	    if ( inputCnt != 0 ) {
		delete [] fieldForInput;
		inputCnt = 0;
		fieldForInput = NULL;
	    }
	}

	return;

}

void
ActionRecord::_setOutputCount( int count )
{
	/*
	 * For Extraction types, set the count of output fields to the
	 * new value, initializing the fieldForOutput array to all -1's.
	 * For Non-Extraction types, set outputCnt to zero and free space.
	 */
	
	if ( ActionDoesExtraction( action ) ) {	
	    if ( outputCnt != count ) {		   // space requirements changed

		if ( outputCnt != 0 ) {
		    delete [] fieldForOutput;
		}

		if ( count > 0 ) {
		    fieldForOutput = new int [ count ];
		    outputCnt = count;
		} else {
		    outputCnt = 0;
		    fieldForOutput = NULL;
		}
	    }
		
	    for ( int i = 0; i < outputCnt; i++ ) { // initialize elements to -1
		fieldForOutput[i] = -1;
	    }
	} else {
	    if ( outputCnt != 0 ) {
		delete [] fieldForOutput;
		outputCnt = 0;
		fieldForOutput = NULL;
	    }
	}

	return;
}

void 			/* virtual */
ActionRecord::printOn( ostream& strm ) const
{
	int i;

	strm << form( "Action=%s", _actionName() );
	strm << form( "\n %d fields for inputs:", inputCnt );
	for ( i = 0; i < inputCnt; i++ ) {
	    strm << form( " %d", fieldForInput[i] );
	}
	strm << form( "\n %d fields for outputs:", outputCnt );
	for ( i = 0; i < outputCnt; i++ ) {
	    strm << form( " %d", fieldForOutput[i] );
	}
}
