/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * VoidPtrList.cc: methods for the VoidPtrList class that maintains a list
 * of void* types in an array structure.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/UsefulClasses/RCS/VoidPtrList.C,v 1.13 1994/02/25 04:34:36 aydt Exp $
 */
#include <string.h>

#include "Assert.h" 
#include "VoidPtrList.h"

static void * NullObj = 0; 	// Return value of list ele in case of ERROR

VoidPtrList::VoidPtrList( int initialSize )
{
	list = new void* [ initialSize ];
	nItems = 0;
	nItemsAllocated = initialSize;

	for ( int i = 0; i < nItemsAllocated; i++ ) {
	    list[i] = NullObj;
	}
}

VoidPtrList::VoidPtrList( const VoidPtrList& vpl )
{
	nItemsAllocated = vpl.allocatedSize();
	list = new void* [ nItemsAllocated ];
	for ( int i = 0; i < nItemsAllocated; i++ ) {
	    list[i] = vpl[i];
	}
	nItems = vpl.count();
}

VoidPtrList::~VoidPtrList()
{
	delete[] list;
}

void 
VoidPtrList::_growList( int n )
{
        // We should not be here if element n exists in current list.
	Assert( n >= nItemsAllocated );

	void* *oldList = list;
	int newSize = ( n * 2 ) + 1;
	list = new void* [ newSize ];

	// Move original entries and initialize new entries to NullObj
	for ( int i = 0; i < nItemsAllocated; i++ ) {	
	    list[i] = oldList[i];
	}
	while ( i < newSize ) {			
	    list[i] = NullObj;
	    i++;
	}

	delete[] oldList;
	nItemsAllocated = newSize;
}

int 
VoidPtrList::addElement( void *element )
{
	if ( nItems == nItemsAllocated ) {
		_growList( nItems );
	}

	int index = nItems++;
	list[ index ] = element;
	return index;
}

void 
VoidPtrList::deleteElementAt( int n )
{
	if ( n >= nItemsAllocated ) {
           _growList( n );
	}
	if ( n >= nItems ) {
           nItems = n + 1;
	}
        for ( int i=n; i < nItems-1; i++ )  {
            list[i] = list[i+1];
	}
        nItems--;
        list[nItems] = NullObj;
}


void* const
VoidPtrList::getElement( int n ) const
{
	if ( ( n < 0 ) || ( n >= nItems ) ) {
	    warning( "Index %d out of bounds in VoidPtrList::getElement", n );
	    return NullObj;
	}
	return list[ n ];
}

int 
VoidPtrList::lookup( void *elt ) const
{
        int i;
        for ( i = 0; i < nItems; i++ ) {
            if ( strcmp( (char *)list[i], (char *)elt ) == 0 ) {
               return( i );
            }
        }
        return( -1 );
}

void 
VoidPtrList::setElement( int n, void *element )
{
	if ( n < 0 ) {
	    warning( " Index less than 0 in VoidPtrList::setElement( %d )", n );
	}

	if ( n >= nItemsAllocated ) {
		_growList( n );
	}
	if ( n >= nItems ) {
		nItems = n + 1;
	}
	list[ n ] = element;
}

VoidPtrList&
VoidPtrList::operator=( const VoidPtrList& vpl )
{
	delete[] list;		// Free old space
	
	nItemsAllocated = vpl.allocatedSize();
	list = new void* [ nItemsAllocated ];
	for ( int i = 0; i < nItemsAllocated; i++ ) {
	    list[i] = vpl[i];
	}
	nItems = vpl.count();
	return *this;
}

void *const 
VoidPtrList::operator[]( int index ) const
{
	if ( ( index < 0 ) || ( index >= nItems ) ) {
	    return NullObj;
	} else {
	    return list[ index ];
	}
}

void 			/* virtual */
VoidPtrList::printOn( ostream& strm ) const
{
	strm << form( "VoidPtrList(%x): %d in use; %d allocated.\n",
			this, nItems, nItemsAllocated);
}

