/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BooleanArray.cc -  Methods for the BooleanArray class.  Note that 
 * many of these methods contain duplicate code... it was added to help 
 * performance by reducing the number of methods which must be called to 
 * accomplish a task that occurs frequently.  Also note that most of 
 * the bounds checking is disables if ASSERT is not defined.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Primitives/RCS/BooleanArray.C,v 1.10 1994/03/15 19:43:31 aydt Exp $
 */

#include <memory.h>
#include <stdarg.h>
#include <stdlib.h>

#include "BooleanArray.h"
#include "Assert.h"

BooleanArray::BooleanArray( int _dimension )
{
	_setClassName( MY_CLASS );
	dimension = _dimension;

	cellsInDim = new int[dimension];	
	strides = new int[dimension];
	int i;
	for ( i = 0; i < dimension; i++ ) { 
	    cellsInDim[i] = 0;
	    strides[i] = 0;
	}
	cellCount = 0;				// no data cells yet
	data = NULL;
}

BooleanArray::BooleanArray( const BooleanArray& array )
{
	_setClassName( MY_CLASS );
	dimension = array.dimension;

	cellsInDim = new int[dimension];
	strides = new int[dimension];
	int bytesToCopy = dimension * sizeof( int );
	memcpy( cellsInDim, array.cellsInDim, bytesToCopy );
	memcpy( strides, array.strides, bytesToCopy );

	cellCount = array.cellCount;
	if ( cellCount > 0 ) {
	    bytesToCopy = cellCount * sizeof( unsigned char );
	    data = (unsigned char *)malloc( bytesToCopy );
	    Assert( data != NULL );		// Need Exception Handler
	    memcpy( data, array.data, bytesToCopy );
	} else {
	    data = NULL;
	}
}

BooleanArray::BooleanArray( const BooleanArray *arrayP )
{
	_setClassName( MY_CLASS );
	dimension = arrayP->dimension;

	cellsInDim = new int[dimension];
	strides = new int [dimension];
	int bytesToCopy = dimension * sizeof( int );
	memcpy( cellsInDim, arrayP->cellsInDim, bytesToCopy );
 	memcpy( strides, arrayP->strides, bytesToCopy );

	cellCount = arrayP->cellCount;
	if ( cellCount > 0 ) {
	    bytesToCopy = cellCount * sizeof( unsigned char );
	    data = (unsigned char *)malloc( bytesToCopy );
	    Assert( data != NULL );		// Need Exception Handler
	    memcpy( data, arrayP->data, bytesToCopy );
	} else {
	    data = NULL;
	}
}

BooleanArray::~BooleanArray()
{
	delete[] cellsInDim;
	delete[] strides;
	if ( cellCount > 0 ) {
	    Assert( data != NULL );
	    free( data );
	}
} 

Boolean_
BooleanArray::getCellValue( int dim0Index ) const
{

#	ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    error( "getCellValue(): BooleanArray is not populated" );
	    return( FALSE_ );
	}
	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
							// an invalid dimIndex
	    error( "getCellValue(): Index 0 = %d is out of range", dim0Index );
	    return( FALSE_ );	
	}
#	endif ASSERT

	int index = strides[0] * dim0Index;

#	ifdef ASSERT
	if ( index >= cellCount ) {
	    error( "getCellValue(): Index = %d is out of range", index );
	    return( FALSE_ );
	}
#	endif ASSERT

	return CnvToBoolean_( data[index] );
} 

Boolean_
BooleanArray::getCellValue( int dim0Index, int dim1Index ) const
{

#	ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    error( "getCellValue(): BooleanArray is not populated" );
	    return( FALSE_ );
	}
	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    error( "getCellValue(): Index 0 = %d is out of range", dim0Index );
	    return( FALSE_ );	
	}
	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
	    error( "getCellValue(): Index 1 = %d is out of range", dim1Index );
	    return( FALSE_ );	
	}
#	endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index;

#	ifdef ASSERT
	if ( index >= cellCount ) {
	    error( "getCellValue(): Index = %d is out of range", index );
	    return( FALSE_ );
	}
#	endif ASSERT

	return CnvToBoolean_( data[index] );
} 

Boolean_
BooleanArray::getCellValue( int dim0Index, int dim1Index, 
			    int dim2Index ... ) const
{
	va_list args;
	va_start( args, dim2Index );			// varargs setup

#	ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    error( "getCellValue(): BooleanArray is not populated" );
	    return( FALSE_ );
	}
	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    error( "getCellValue(): Index 0 = %d is out of range", dim0Index );
	    return( FALSE_ );	
	}
	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
	    error( "getCellValue(): Index 1 = %d is out of range", dim1Index );
	    return( FALSE_ );	
	}
	if ( ( dim2Index >= cellsInDim[2] ) || ( dim2Index < 0 ) ) {
	    error( "getCellValue(): Index 2 = %d is out of range", dim2Index );
	    return( FALSE_ );	
	}
#	endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index
			+ strides[2] * dim2Index;

	int i, dimIndex;
	for ( i = 3; i < dimension; i++ ) {
	    dimIndex = va_arg( args, int );

#	    ifdef ASSERT
	    if ( ( dimIndex >= cellsInDim[i] ) || ( dimIndex < 0 ) ) {
	        error( "getCellValue(): Index %d = %d is out of range",
	               i, dimIndex );
		return( FALSE_ );
	    }
#	    endif ASSERT

	    index += strides[i] * dimIndex;
	}
	va_end( args );
	
#	ifdef ASSERT
	if ( index >= cellCount ) {
	    error( "getCellValue(): Index = %d is out of range", index );
	    return( FALSE_ );
	}
#	endif ASSERT

	return CnvToBoolean_( data[index] );
} 

Boolean_
BooleanArray::getTheCellValue( int index ) const
{

#	ifdef ASSERT
	if ( index >= cellCount ) {			// invalid index
	    error( "getTheCellValue(): Index = %d is out of range", index );
	    return( FALSE_ );
	}
#	endif ASSERT
	
	return CnvToBoolean_( data[index] );
} 


Boolean_
BooleanArray::setCellValue( Boolean_ value, int dim0Index ... ) 
{
	va_list args;
	va_start( args, dim0Index );	// varargs setup

#	ifdef ASSERT
	if ( cellCount == 0 ) {
	    error( "setCellValue(): called before dimension size(s) set");
   	    return( FAILURE_ );
	}
	if ( ( dim0Index < 0 ) || ( cellsInDim[0] <= dim0Index ) ) {
	    error( "setCellValue(): Index 0 = %d is out of range", dim0Index );
	    return( FAILURE_ );
	}
#       endif ASSERT

	int index = strides[0] * dim0Index;

	int dimIndex, i;
	for ( i = 1; i < dimension; i++ ) {
  	    dimIndex = va_arg( args, int );

#	    ifdef ASSERT
	    if ( ( dimIndex < 0 ) || ( cellsInDim[i] <= dimIndex ) ) {
	        error( "setCellValue(): Index %d = %d is out of range",
	    	       i, dimIndex );
	        return( FAILURE_ );
	    }
#	    endif ASSERT

	    index += strides[i] * dimIndex;
	}

	va_end( args );

#       ifdef ASSERT
	if ( index >= cellCount ) {
	    error( "setCellValue() - Invalid index = %d", index );
	    return( FAILURE_ );
	}
#       endif ASSERT

	data[index] = (unsigned char)value;
	return( SUCCESS_ );
}

void 
BooleanArray::setDimSizes( const int *intArray )
{
	/* 
	 * We try to be a bit clever and only free/realloc space if really
	 * necessary.  It is "really necessary" if the number of cells in
	 * the array has changed. Else, we can just reuse the old space.
	 * In either case, we guarantee that the data space is initialized
	 * to zeros.
	 */

	int newCellCount = 1;			// set new cell counts
	for ( int i = 0; i < dimension; i++ ) {
	    cellsInDim[i] = intArray[i];
	    newCellCount *= cellsInDim[i];
	}

	if ( newCellCount != cellCount ) {	// must resize, free and alloc
	    if ( cellCount > 0 ) {
	        free( data );
	    }
	    if ( newCellCount > 0 ) {
	        data = (unsigned char *)calloc( newCellCount, 
						sizeof( unsigned char ) );
	        Assert( data != NULL );    	// need exception handler
	    } else {
	        data = NULL;
	    }
	    cellCount = newCellCount;
	} else {				// reuse original data space
	    if ( data != NULL ) {
	        memset( data, 0, cellCount * sizeof( unsigned char ) );
	    }
	}

	if ( cellCount > 0 ) { 			// recalculate strides
	    strides[0] = cellCount / cellsInDim[0];
	    for ( i = 1; i < dimension; i++ ) {
            	strides[i] = strides[i-1] / cellsInDim[i];
	    }
	} else {
	    for ( i = 0; i < dimension; i++ ) {
            	strides[i] = 0;
	    }
	}

}

Boolean_
BooleanArray::setTheCellValue( int index, Boolean_ value ) 
{
#	ifdef ASSERT
	if ( cellCount == 0 ) {
	    error( "setTheCellValue(): called before dimension sizes set");
	    return( FAILURE_ );
	}
	if ( index >= cellCount) {
	    error( "setTheCellValue() - Invalid index = %d", index );
	    return( FAILURE_ );
	}
#	endif ASSERT

	data[index] = (unsigned char) value;
	return( SUCCESS_ );
} 

BooleanArray&
BooleanArray::operator=( const BooleanArray &array )
{
	/* 
	 * Note that this is the only way we can change the 
	 * dimension of an array after it has been constructed!
	 */

	int 	  i;
	Boolean_  reSize = FALSE_;

	if ( dimension != array.dimension ) {
	    reSize = TRUE_;
	} else {
	    for ( i = 0; i < dimension; i++ ) {
	        if ( cellsInDim[i] != array.cellsInDim[i] ) {
	    	    reSize = TRUE_;
	    	    break;
	        }
	    }
	}

	if ( reSize ) {
	    delete[] cellsInDim;	      // Free original space
	    delete[] strides;
	    if ( cellCount > 0 ) {
	        free( data );
	    }
	    data = NULL;

	    dimension = array.dimension;
	    cellCount = array.cellCount;

	    cellsInDim = new int[dimension];
	    strides = new int[dimension];

	    memcpy( cellsInDim, array.cellsInDim, dimension * sizeof(int) );
	    memcpy( strides, array.strides, dimension * sizeof(int) );

	    if ( cellCount > 0 ) {
	        data = (unsigned char *)malloc( 
					   cellCount * sizeof(unsigned char) );
	        Assert( data != NULL );	    // Need Exception Handler
	    }
	}

	if ( cellCount > 0 ) {
	    memcpy( data, array.data, cellCount * sizeof( unsigned char ) );
	}

	return *this;
}

void
BooleanArray::printOn( ostream& os ) const
{
	int           i;
	unsigned char *cp = data;

	os << "<BooleanArray begin>\n";
	os << "Dimensions: " << dimension << NL;
	os << "Cells in each dimension: " ;

	int *index = new int[ dimension ];	// To keep indicies in each dim
	for ( i = 0; i < dimension; i++ ) {	// zero index array & print 
	    index[i] = 0;
	    os << cellsInDim[i] << " " ;
	}

	int cell;
	for ( cell = 0; cell < cellCount; cell++ ) {
	    cout << "\nCell";
	    for ( i = 0; i < dimension; i++ ) {
	    	cout << "[" << index[i] << "]";
	    }
	    cout << "= ";
	    if ( *cp++ ) {
	    	cout << "T";
	    } else {
	    	cout << "F";
	    }
	    
	    for ( i = dimension - 1; i >= 0; i-- ) {
	    	if ( ++index[i] < cellsInDim[i] ) {
	    		break;
	    	} else {
	    		index[i] = 0;
	    	}
	    }
	}
	delete index;

	os << "\n<BooleanArray end>\n";
}

/*
 * Initialize the static data 
 */
const char *const BooleanArray::MY_CLASS = "BooleanArray";
