/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Bobby A.A. Nazief (nazief@cs.uiuc.edu)
 * Author: Taed Nelson (nelson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BinaryPipeWriter.C -  write packets to the binary output pipes.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/BinaryPipeWriter.C,v 1.17 1994/02/25 04:34:19 aydt Exp $
 */

#include <stdlib.h>
#include <memory.h>

#include "Assert.h"
#include "Attributes.h"
#include "BinaryPipeWriter.h"
#include "DataCharacteristics.h"
#include "GetPut.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"
#include "StreamPipe.h"
#include "SystemErrors.h"

BinaryPipeWriter::BinaryPipeWriter( StreamPipe *thePipe ) 
		 :PipeWriter( thePipe )
{
	_setClassName( MY_CLASS );
	currentHeader.type = PIPE_EMPTY;
	currentHeader.length = 0;

        bufferSize = PACKET_HEADER_SIZE;
        packetBuffer = (char *) malloc( bufferSize );
	if ( packetBuffer == NULL ) {
	    abort( "Can't malloc space for bufferSize %d: %s",
		    bufferSize,  errorString() );
	}

        /* 
         * We only allow binary pipe writers to be connected to pipes
         * that are open for output.
         */
        if ( thePipe->isOutputPipe() ) { 
            _writeSddfHeader();   
	    if (  thePipe->isFilePipe() ) {
   	        startOffset = thePipe->getByteOffset();
	    } else {
		startOffset = -1;
	    }
        } else {
            error( "\n Can't connect to data stream %s",
                    thePipe->getObjectName() );
            deletePipe( thePipe );
        }
}

BinaryPipeWriter::~BinaryPipeWriter()
{
	free( packetBuffer );
}

void
BinaryPipeWriter::_writePacket()
{
	char *bufPtr = packetBuffer;
	int nWritten;
	
	functionPut( &bufPtr, &currentHeader.length );
	int type = (int) currentHeader.type;
	functionPut( &bufPtr, &type );		// Note we write this as an int!
	functionPut( &bufPtr, &currentHeader.tag );

	for ( int p = 0; p < activePipes; p++ ) {
	    nWritten = pipe[p]->put( packetBuffer, currentHeader.length );

	    if ( nWritten != currentHeader.length ) {
		    error( "packet write failed! pipe=%d length=%d nWritten=%d",
		      		p, currentHeader.length, nWritten );
	    }
	}
}

void		/* virtual */
BinaryPipeWriter::_writeSddfHeader() const
{
	/*
	 * Only write the Sddf header information to StreamPipes attached
	 * to Files!  Internal pipes always carry data in the native mode
	 * and don't need to worry about the Data Characteristics.
	 *
	 * Allocation of buffer isn't that neat but oh well...
	 */
	StreamPipe *thePipe = pipe[activePipes-1];

	if ( thePipe->isFilePipe() ) {
	    char buffer[128];
	    int hdrSize = DataCharacteristics::writeNativeHeader( buffer,
								   128 );
	    if ( hdrSize == -1) {
		abort( "Buffer not large enough for header - RECOMPILE" );
	    } else {
	        int nWritten = thePipe->put( buffer, hdrSize );
	        if ( nWritten != hdrSize ) {
		abort( "Header write failed. %s; expected %d; wrote %d",
		        thePipe->getObjectName(), hdrSize, nWritten );
	        }
	    }
	}
}

/*
 * Some notes on memory allocation:
 * 1) Rather than first check to see how many bytes
 * we need to hold an object instance (Attributes, StructureDescriptor or 
 * RecordDossier), I decided to first call objToBits() and only allocate more
 * space if that fails - then call it again.  I'm guessing that this will
 * cause "problems" when the system first starts but that in a short time
 * packetBuffer will be large enough to hold most objects.  Since the
 * bytesNeeded() methods are themselves quite expensive, I elected not to 
 * call them every time before calling objToBits to ensure that packetBuffer
 * is big enough.  This may be a bad decision in the long run -- we'll see.
 * 2) I have also decided never to reduce the size of packetBuffer. This will
 * be unnecessarily wasteful of memory if we have one HUGE packet but then
 * may other small ones.  Again, this is somewhat trial and error.  We may
 * want to add some "high water mark" in the future.		-RAA
 */

void		/* virtual */
BinaryPipeWriter::putAttributes()
{
	int dataBytes = pipeAttributes.objToBits( 
						packetBuffer+PACKET_HEADER_SIZE,
					      	bufferSize-PACKET_HEADER_SIZE );
	if ( dataBytes == -1 ) {
		dataBytes = pipeAttributes.bytesNeeded();
		bufferSize = dataBytes + PACKET_HEADER_SIZE;

		packetBuffer = (char *) realloc( packetBuffer, bufferSize );
	        if ( packetBuffer == NULL ) {
	            abort( "Can't realloc space for bufferSize %d: %s",
		            bufferSize,  errorString() );
	        }

	 	dataBytes = pipeAttributes.objToBits( 
						packetBuffer+PACKET_HEADER_SIZE,
					      	bufferSize-PACKET_HEADER_SIZE );
		Assert( dataBytes != -1 )
	}

	currentHeader.length = dataBytes + PACKET_HEADER_SIZE;
	currentHeader.type = PKT_ATTRIBUTE;
	currentHeader.tag = 0;		// tag not used for Attr pkt
	_writePacket();
}

void		/* virtual */
BinaryPipeWriter::putAttributes( const Attributes& attributes )
{
	pipeAttributes = attributes;
	putAttributes();
}

void		/* virtual */
BinaryPipeWriter::putCommand( int pktTag )
{
	currentHeader.length = PACKET_HEADER_SIZE;
	currentHeader.type = PKT_COMMAND;
	currentHeader.tag = pktTag;
	_writePacket();
}
	
void		/* virtual */
BinaryPipeWriter::putData( const RecordDossier& recordDossier )
{
	int dataBytes = recordDossier.objToBits( 
						packetBuffer+PACKET_HEADER_SIZE,
					     	bufferSize-PACKET_HEADER_SIZE );
	if ( dataBytes == -1 ) {
		dataBytes = recordDossier.bytesNeeded();
		bufferSize = dataBytes + PACKET_HEADER_SIZE;

		packetBuffer = (char *) realloc( packetBuffer, bufferSize );
		if ( packetBuffer == NULL ) {
	    	    abort( "Can't realloc space for bufferSize %d: %s",
			    bufferSize,  errorString() );
		}

	 	dataBytes = recordDossier.objToBits( 
						packetBuffer+PACKET_HEADER_SIZE,
					      	bufferSize-PACKET_HEADER_SIZE );
		Assert( dataBytes != -1 )
	}

	currentHeader.length = dataBytes + PACKET_HEADER_SIZE;
	currentHeader.type = PKT_DATA;
	currentHeader.tag = recordDossier.getTag();
	_writePacket();
}

void		/* virtual */
BinaryPipeWriter::putDescriptor( const StructureDescriptor& structDescr,
				 int pktTag )
{
	int dataBytes = structDescr.objToBits( packetBuffer+PACKET_HEADER_SIZE,
					       bufferSize-PACKET_HEADER_SIZE );
	if ( dataBytes == -1 ) {
		dataBytes = structDescr.bytesNeeded();
		bufferSize = dataBytes + PACKET_HEADER_SIZE;

		packetBuffer = (char *) realloc( packetBuffer, bufferSize );
		if ( packetBuffer == NULL ) {
	    	    abort( "Can't realloc space for bufferSize %d: %s",
			    bufferSize,  errorString() );
		}

	 	dataBytes = structDescr.objToBits( 
					       packetBuffer+PACKET_HEADER_SIZE,
					       bufferSize-PACKET_HEADER_SIZE );
		Assert( dataBytes != -1 )
	}

	currentHeader.length = dataBytes + PACKET_HEADER_SIZE;
	currentHeader.type = PKT_DESCRIPTOR;
	currentHeader.tag = pktTag;
	_writePacket();
}

void	
BinaryPipeWriter::putPacket( const PacketHeader& pktHeader, 
			     const char* pktData )
{
	/* 
	 * Update currentHeader with the new header.
	 * Allocate a bigger packet buffer if necessary to hold packet.
	 * Copy packet data into packet buffer at appropriate offset.
	 * Call _writePacket() to send it to the pipe.
	 */
	currentHeader = pktHeader;

	if ( currentHeader.length > bufferSize ) {
	    bufferSize = currentHeader.length;
	    packetBuffer = (char *) realloc( packetBuffer,  bufferSize );
	    if ( packetBuffer == NULL ) {
	          abort( "Can't realloc space for packet size %d: %s",
			  bufferSize,  errorString() );
	    }
	}

	memcpy( packetBuffer+PACKET_HEADER_SIZE, pktData,
		currentHeader.length-PACKET_HEADER_SIZE );

	_writePacket();
}
		
/*
 * Initialize the static data 
 */
const char *const BinaryPipeWriter::MY_CLASS = "BinaryPipeWriter";
