/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InfrastructureManager.cc: Implements InfrastructureManager.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Managers/RCS/InfrastructureManager.C,v 1.28 1994/04/07 01:36:46 aydt Exp $
 *	
 */

#include <ctype.h>

#include "ConnectionId.h"
#include "ConnectionIdPtrList.h"
#include "ModuleId.h"
#include "ModuleIdPtrList.h"

#include "InfrastructureManager.h"
#include "FunctionalUnit.h"
#include "FunctionalUnitManager.h"
#include "Wrapper.h"
#include "FUWrapper.h"
#include "FileInputWrapper.h"
#include "FileOutputWrapper.h"
#include "StreamPipe.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"
#include "FIFOStreamPipe.h"
#include "IntList.h"
#include "Pablo.h"
#include "GraphDisplay.h"
#include "PabloMainInterface.h"
#include "PabloResources.h"
#include "PacketHeader.h"


InfrastructureManager::InfrastructureManager() 
{
	moduleIds = new ModuleIdPtrList;
	connectionIds = new ConnectionIdPtrList;

	sortedModuleIds = NULL;
	modulesConfigured = FALSE_;
}

InfrastructureManager::~InfrastructureManager()
{
	_deleteMyObjects();
}

void
InfrastructureManager::_deleteMyObjects()
{
	int 		i;
	ModuleId 	*mid;
	Wrapper 	*wrapper;
	FunctionalUnit 	*fu;
	ConnectionId 	*cid;
	StreamPipe 	*pipe;

	/* 
	 * For each module in the module list 
	 *   - delete the ModuleId, the Wrapper, the FU
	 * For each connection in the connection list
	 *   - delete the ConnectionId, the Pipe
	 * Delete the ModuleIdList, the ConnectionIdList, the SortedModuleList
	 * Set the configured flag to FALSE
	 */

	for ( i = 0; i < moduleIds->count(); i++ ) {
	    mid = moduleIds->getElement( i );
	    wrapper = mid->getWrapper();
	    fu = wrapper->getFunctionalUnit();

	    delete mid;
	    delete wrapper;
	    if ( fu != NULL ) {
		delete fu;
	    }
	}

	for ( i = 0; i < connectionIds->count(); i++ ) {
	    cid = connectionIds->getElement( i );
	    pipe = cid->getPipe();

	    delete cid;
	    delete pipe;
	}

	delete moduleIds;
	delete connectionIds;
	if ( sortedModuleIds != NULL ) {
	    delete sortedModuleIds;
	}
	modulesConfigured = FALSE_;
}

void 
InfrastructureManager::_reSort()
{
	if ( sortedModuleIds ) {
	    delete sortedModuleIds;
	}

	int i;
	int n = moduleIds->count();
	int sortIdx = n - 1;			// where to start the sort
	IntList marked( n + 1 );		// track items still to sort

	for ( i = 0; i < n; i++ ) {
	    marked[i] = 0;
	}

	sortedModuleIds = new ModuleIdPtrList( n + 1 );  
	ModuleId *id;
	IntList stack( n + 1 );

	for ( i = 0; i < n; i++ ) {
	    id = moduleIds->getElement(i);
	    if ( !marked[i] ) {
		_topSortAux( id, moduleIds, marked, sortedModuleIds, 
			     sortIdx, stack );
	    }
	}
}

void 
InfrastructureManager::_topSortAux( ModuleId *id, ModuleIdPtrList *ids,
				   IntList& marked, ModuleIdPtrList *sorted, 
			           int& sortIdx, IntList& stack )
{
	stack.addElement( id->getIndex() );
	marked[ id->getIndex() ] = 1;

	ConnectionIdPtrList *outgoing = new ConnectionIdPtrList;

	int i;
	for ( i = 0; i < connectionIds->count(); i++ ) {
	    ConnectionId *conn = connectionIds->getElement( i );
	    if ( conn->getInput() == id ) {
		outgoing->addElement( conn );
	    }
	}

	for ( i = 0; i < outgoing->count(); i++ ) {
	    ModuleId *outId = outgoing->getElement(i)->getOutput();
	    if ( !marked[outId->getIndex()] ) {
		_topSortAux( outId, ids, marked, sorted, sortIdx, stack );
	    }
	}

	int n = stack.count();
	int idx = stack.getElement( n - 1 );
	stack.deleteElementAt( n - 1 );
	sorted->setElement( sortIdx, ids->getElement(idx) );
	sortIdx--;
	delete outgoing;	// free list of outgoing connections
}

void 
InfrastructureManager::configure()
{
	// Run the module Wrappers from "top to bottom" of the graph,
	// causing Descriptors at the head of the input file(s) to filter
	// down through the modules allowing them to be configured.
	// configure() is called with FALSE_ because we are configuring 
	// the entire system, not just individual modules.

	if ( sortedModuleIds == NULL ) {
		info( "No modules to configure!" );
		return ;	// all Done!
	}

	Boolean_ singleModuleFlag = FALSE_;
	ModuleId *id;
	int i;
	
	for ( i = 0; i < sortedModuleIds->count(); i++ ) {
	    id = sortedModuleIds->getElement( i );

	    if ( Pablo::MainInterface() ) {
		Pablo::MainInterface()->beginConfigureOf( id );
	    }
			
	    id->getWrapper()->configure( singleModuleFlag );	

	    if ( Pablo::MainInterface() ) {
		Pablo::MainInterface()->endConfigureOf( id );
	    }
	}
	modulesConfigured = TRUE_;
}

Boolean_ 
InfrastructureManager::loadConfigurationFromDir( const CString& dir )
{
	int i;
	ModuleId *mid;

	for ( i = 0; i < sortedModuleIds->count(); i++ ) {
	    mid = sortedModuleIds->getElement( i );
	    RETURN_ON_FAILURE( mid->loadConfigurationFromDir( dir ) );
	}
	modulesConfigured = TRUE_;
	return SUCCESS_;
}

ModuleId *
InfrastructureManager::lookupModule( int moduleIndex ) const
{
	return moduleIds->getElement( moduleIndex ) ;
}

void
InfrastructureManager::reset()
{
	/*
	 * To reset, first delete all the objects currently associated
	 * with the IM.  Then, reinitialize lists and flags.
	 */

	_deleteMyObjects();

	moduleIds = new ModuleIdPtrList;
	connectionIds = new ConnectionIdPtrList;
	sortedModuleIds = NULL;
	modulesConfigured = FALSE_;
	ModuleId::_resetUniqueId();
}

void
InfrastructureManager::restart()
{
	/*
	 * To restart, call restart methods for all module wrappers.
	 */
	for ( int i = 0; i < moduleIds->count(); i++ ) {
	    moduleIds->getElement(i)->getWrapper()->restart();
	}
}

Boolean_
InfrastructureManager::run( int nPasses, Boolean_ singleStepping )
{
	/* 
	 * NOTE: if nPasses is large, the expose events aren't processed
	 * in the 'middle'... You may want to check them periodically
	 * or at least do a XmUpdateDisplay... Ruth 8/92
	 */
	 
	if ( sortedModuleIds == NULL ) {
	    info( "No modules to run!" );
	    Pablo::MainInterface()->runStop();
	    return TRUE_;       // so we don't mark as execution compelete
	}

	if ( modulesConfigured == FALSE_ ) {
	    info( "Modules must be configured before being run!" );
	    Pablo::MainInterface()->runStop();
	    return TRUE_;      // so we don't mark as execution complete
	}

	ModuleId *modId;
	Wrapper  *wrapper;
	Boolean_ checkAgain;
	Boolean_ aModuleRan;
	Boolean_ errorFlag = FALSE_;

	/*
	 * cycle through the entire graph nPasses times -
	 * or until no module ran on the last pass - or until
	 * an error was seen in one of the modules.
	 */
	do {
	    nPasses--;
	    aModuleRan = FALSE_;

	    /* 
	     * Cycle through the modules in sorted order.  First
	     * check to see if a module is ready to run, then run
	     * it if it's ready.  If the module is a FU, then after 
	     * it has run, check to see if it can run again -- this "drains"
	     * all it's input pipes.  (For FileInput modules we only want
	     * one packet per cycle.  For FileOutput modules, the run method
	     * empties the incoming pipe so we don't need to check again.)
	     */

	    for ( int i = 0; i < sortedModuleIds->count(); i++ ) {

		modId = sortedModuleIds->getElement(i);
		wrapper = modId->getWrapper();

		if ( wrapper->getWrapperClass() == FUWrapperClass ) {
	            checkAgain = TRUE_;
		} else {
		    checkAgain = FALSE_;
		}

		do {
		    if ( singleStepping ) {
		        Pablo::MainInterface()->waitForStepContinue( modId );
		    }

		    if ( wrapper->ready() ) {
		        wrapper->run( errorFlag );
			aModuleRan = TRUE_;
		    } else {
			checkAgain = FALSE_;  // always stop if not ready!
		    }
		} while ( checkAgain );
	    }	

	} while ( ( nPasses > 0 ) && aModuleRan && !errorFlag );

	/*
	 * If an error occured in one of the modules we want to stop
	 * the run so that the user can maybe fix it through reconfiguration.
	 */
	if ( errorFlag ) {
	    Pablo::MainInterface()->runStop();
	}

	return ( aModuleRan );
}

Boolean_ 
InfrastructureManager::saveConfigurationToDir( const CString& dir ) const
{
	int i;
	ModuleId *mid;

	for ( i = 0; i < moduleIds->count(); i++ ) {
	    mid = moduleIds->getElement( i );
	    RETURN_ON_FAILURE( mid->saveConfigurationToDir( dir ) );
	}
	return SUCCESS_;
}

Boolean_ 
InfrastructureManager::writeLayoutToFP( FILE *fp )
{
	int 	     i;
	ModuleId     *mid;
	ConnectionId *cid;

	for ( i = 0; i < moduleIds->count(); i++ ) {
	    mid = moduleIds->getElement( i );
	    mid->writeLayoutToFP( fp );
	}

	for ( i = 0; i < connectionIds->count(); i++ ) {
	    cid = connectionIds->getElement( i );
	    cid->writeLayoutToFP( fp );
	}

	return SUCCESS_;
}


/***********************************************************
 *							   *
 * Adding and Deleting modules				   *
 *							   *
 ***********************************************************/


int 
InfrastructureManager::addFileInputModule( const CString& moduleName,
					   const CString& fileName )
{
	Wrapper *wrapper = new FileInputWrapper( fileName );
	ModuleId *mid = new ModuleId( wrapper, moduleName, "sys:FileInput" );

	int midx = moduleIds->addElement( mid );
	mid->setIndex( midx );
	_reSort();

	wrapper->setIdentifier( mid->getIdentifier() );
	wrapper->setName( moduleName );
	wrapper->init();
	
	return midx;
}

int 
InfrastructureManager::addFileOutputModule( const CString& moduleName,
					    const CString& fileName )
{
	Wrapper *wrapper = new FileOutputWrapper( fileName );
	ModuleId *mid = new ModuleId( wrapper, moduleName, "sys:FileOutput" );

	int midx = moduleIds->addElement( mid );
	mid->setIndex( midx );
	_reSort();

	wrapper->setIdentifier( mid->getIdentifier() );
	wrapper->setName( moduleName );
	wrapper->init();

	return midx;
}

int 
InfrastructureManager::addModule( const CString& fuName, 
				  const CString& moduleName )
{
	FunctionalUnit *fu = Pablo::FUM()->getFU( fuName );
	if ( fu == NULL ) {
	    error( "Tried to add FU %s but it doesn't exist", 
			(const char *)fuName );
	    return -1;
	}

	Wrapper *wrapper = new FUWrapper( fu, fuName );
	ModuleId *mid = new ModuleId( wrapper, moduleName, fuName );

	int midx = moduleIds->addElement( mid );
	mid->setIndex( midx );
	_reSort();

	wrapper->setIdentifier( mid->getIdentifier() );
	wrapper->setName( moduleName );
	wrapper->init();

	return midx;
}

Boolean_
InfrastructureManager::deleteModule( int moduleIndex )
{
	/*
	 * Look up the module to be deleted and make sure it exists.
	 */
	ModuleId *mid = moduleIds->getElement( moduleIndex );
	if ( mid == NULL ) {
	   warning( "No ModuleId for index %d.", moduleIndex );
	   return FAILURE_;
	}

	/*
	 * If this module is the input for any connections (has output
	 * pipes associated with it), we can't delete the module.
	 */
	ConnectionId *cid;
	int i;
	for (  i = 0; i < connectionIds->count(); i++ ) {
	    cid = connectionIds->getElement( i );
	    if ( cid->getInput() == mid ) {
	        info( "Sorry, can't delete a module that has output pipe(s)." );
	        return FAILURE_;
	    }
	}

	/*
	 * For any connections where this module is the destination, we
	 * need to do several things for each connection...
	 * 1) Look up the source of the connection and remove the 
	 *    associated pipe from that source since it will no longer 
	 *    be connected to anything.
	 * 2) Delete the pipe
	 * 3) Delete the connection.
	 * 4) Remove the connection from the list of connections.
	 */
	int countOfConnections = connectionIds->count();
	
	for ( i = 0; i < countOfConnections; i++ ) {
	    cid = connectionIds->getElement( i );

	    if ( cid->getOutput() == mid ) {
		StreamPipe *thePipe = cid->getPipe();
		cid->getInput()->getWrapper()->deleteOutputPipe( thePipe );
		delete thePipe;
		delete cid;
		connectionIds->deleteElementAt( i );
		i--;			// so we don"t skip "moved entry"
		countOfConnections--;	// one less to process
	    }
	}
	
	/*
	 * Get the associated Wrapper and FU.
	 * Delete the ModuleId, the Wrapper, and the FU.
	 * Delete the entry in the list of modules & adjust the indicies
	 *  in the ModuleId instances that come later in the list.
	 * Resort our list of ModuleIds
	 */

	Wrapper *wrapper = mid->getWrapper();
	FunctionalUnit *fu = wrapper->getFunctionalUnit();

	delete mid;
	delete wrapper;
	if ( fu != NULL ) {
	    delete fu;
	}

	moduleIds->deleteElementAt( moduleIndex );
	for ( i = moduleIndex; i < moduleIds->count(); i++ ) {
	    mid = moduleIds->getElement( i );
	    mid->setIndex( i );
	}

	_reSort();
	return SUCCESS_;
}

Boolean_ 
InfrastructureManager::addConnection( int fromIdx, int toIdx )
{
	if ( fromIdx == toIdx ) {
	    warning( "Can't connect a module to itself!\n" );
	    return FAILURE_;
	}

	ModuleId *from = moduleIds->getElement( fromIdx ) ;
	ModuleId *to = moduleIds->getElement( toIdx ) ;

	/* 
	 * Look up this connection in our list to make sure the two 
	 * modules aren't already connected.  If they are, then we 
	 * don't allow a second connection.
	 */
	ConnectionId *cid;
	for ( int i = 0; i < connectionIds->count(); i++ ) {
	    cid = connectionIds->getElement( i );
	    if ( (cid->getInput() == from && cid->getOutput() == to) ||
	         (cid->getInput() == to && cid->getOutput() == from) ) 
	    {
		warning( "Modules %s and %s already connected\n",
			  cid->getInput()->getName().getValue(),
			  cid->getOutput()->getName().getValue());
		return FAILURE_; 
	    }
	}
	     
	/* 
	 * Create pipe to connect the modules and notify the wrappers
	 * about the new pipe.  If for some reason, the wrapper can't
	 * accept a pipe of the form requested (input or output), delete
	 * the pipe and return FAILURE_; 
	 */
	CString pipeName = from->getName() + "_to_" + to->getName();
	StreamPipe *pipe = new FIFOStreamPipe( 100000, pipeName );

	if ( from->getWrapper()->addOutputPipe( pipe ) == FAILURE_ ) {
	    delete pipe;
	    return FAILURE_;
	}

	if ( to->getWrapper()->addInputPipe( pipe, from ) == FAILURE_ ) {
	     from->getWrapper()->deleteOutputPipe( pipe );
	     delete pipe;
	     return FAILURE_;
	}

	/* 
	 * Our pipe is created and the module wrappers know about it.
	 * Add the connection to the connection list.
	 * Resort our modules based on the new connectivity information.
	 */
	cid = new ConnectionId( from, to, pipe );
	connectionIds->addElement( cid );
	_reSort();
	
	return SUCCESS_;
}


/*
 * Initialize the static data 
 */
const char *const InfrastructureManager::MY_CLASS = "InfrastructureManager";

