/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *               University of Illinois at Urbana-Champaign
 *               Department of Computer Science
 *               1304 W. Springfield Avenue
 *               Urbana, IL     61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *      All Rights Reserved.
 *
 * Author: Keith A. Shields (shields@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ConstantsDialog.cc : This module implements a custom dialog for
 *                      specifying constant data values for input ports and
 *                       output record fields
 * 
 *
 */
#include <stdlib.h>

#include "ConstantsDialog.h"
#include "PabloHelpSystem.h"

ConstantsDialog::ConstantsDialog(Widget parentApp, 
				 InterfaceClass *pabloInterface,
				 ConfigBoard *statusBoard)
{
  _setClassName(MY_CLASS);
  PabloInterface = pabloInterface;
  StatusBoard = statusBoard;	

  DialogShell = XtCreatePopupShell("ConstantsDialog", 
				   topLevelShellWidgetClass, parentApp, NULL, 
				   0);
				   
  Widget panedWindow  = XtVaCreateWidget("dialogPane",
					 xmPanedWindowWidgetClass, 
					 DialogShell, 
					 XmNsashWidth, 1, 
					 XmNsashHeight, 1,
					 NULL );

  _buildControlArea(panedWindow);
  _buildActionArea(panedWindow);
  _setDialogTitle();

  XtManageChild(panedWindow);
}



ConstantsDialog::~ConstantsDialog()
{
  free(PortTextFields);
  free(OutputTextFields);
  free(IsInputPortBoundToConstant);
  free(IsOutputFieldBoundToConstant);
  XtDestroyWidget(DialogShell);
}



void ConstantsDialog::_buildActionArea(Widget panedWindow)
{
  Widget actionArea = XtCreateWidget("actionArea", xmFormWidgetClass, 
				     panedWindow, NULL, 0);

  Widget cont = XtCreateManagedWidget("continue", xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  Widget reset = XtCreateManagedWidget("reset", xmPushButtonWidgetClass,
				       actionArea, NULL, 0);

  Widget help = XtCreateManagedWidget("help", xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  addCallback(cont, XmNactivateCallback, &Callback::callback1, this,
	      NULL);

  addCallback(reset, XmNactivateCallback, &Callback::callback2, this,
	      NULL);

  addCallback(help, XmNactivateCallback, &Callback::helpCallback, this,
	      NULL);


  /* fix height of action area */
  Dimension h; 
  XtVaGetValues(reset, XmNheight, &h, NULL);
  XtVaSetValues(actionArea, 
		XmNpaneMaximum, h, 
		XmNpaneMinimum, h,
		NULL);  

  XtManageChild( actionArea );
}



void ConstantsDialog::_buildControlArea(Widget panedWindow)
{
  Widget controlArea = XtCreateWidget("controlArea", xmFormWidgetClass, 
				      panedWindow, NULL, 0 );

  char *moduleName = (char *) PabloInterface->getModuleName();
  XmString moduleNameX = XmStringCreateSimple(moduleName);
  Widget moduleNameLabel = XtVaCreateManagedWidget("moduleNameLabel", 
                                                   xmLabelWidgetClass, 
                                                   controlArea,
                                                   XmNlabelString, moduleNameX,
                                                   NULL);
  XmStringFree(moduleNameX);


  _buildPortFrame(controlArea, moduleNameLabel);
  _buildOutputFrame(controlArea, moduleNameLabel);

  XtManageChild(controlArea);
}



void ConstantsDialog::_buildOutputFrame(Widget controlArea, Widget nameLabel)
{
  Widget outputFrame = XtVaCreateWidget("outputFrame", xmFrameWidgetClass,
					controlArea, 
					XmNtopWidget, nameLabel,
					NULL);

  Widget outputForm = XtCreateWidget("outputForm", xmFormWidgetClass,
				     outputFrame, NULL, 0);

  Widget outputLabel = XtCreateManagedWidget("outputLabel", xmLabelWidgetClass,
					     outputForm, NULL, 0);
  
  Widget outputWindow = XtVaCreateManagedWidget("outputWindow", 
						xmScrolledWindowWidgetClass, 
						outputForm,
						XmNtopWidget, outputLabel,
						NULL, 0);

  Widget outputContainer =  XtVaCreateWidget("outputContainer",  
					     xmRowColumnWidgetClass, 
					     outputWindow, 
					     XmNnumColumns, 2,
					     NULL);

  _setupOutputConstants(outputContainer);

  XtManageChild(outputContainer);
  XtManageChild(outputWindow);
  XtManageChild(outputForm);
  XtManageChild(outputFrame);
}



void ConstantsDialog::_buildPortFrame(Widget controlArea, Widget nameLabel)
{
  Widget portFrame = XtVaCreateWidget("portFrame", xmFrameWidgetClass,
				      controlArea, 
				      XmNtopWidget, nameLabel, 
				      NULL);

  Widget portForm = XtCreateWidget("portForm", xmFormWidgetClass,
				   portFrame, NULL, 0);

  Widget portLabel = XtCreateManagedWidget("portLabel", xmLabelWidgetClass,
					   portForm, NULL, 0);
  
  Widget portWindow = XtVaCreateManagedWidget("portWindow", 
					      xmScrolledWindowWidgetClass, 
					      portForm,
					      XmNtopWidget, portLabel,
					      NULL, 0);

  Widget portContainer =  XtVaCreateWidget("portContainer",  
					   xmRowColumnWidgetClass, 
					   portWindow, 
					   XmNnumColumns, 2,
					   NULL);
  
  _setupPortConstants(portContainer);

  XtManageChild(portContainer);
  XtManageChild(portWindow);
  XtManageChild(portForm);
  XtManageChild(portFrame);
}



void ConstantsDialog::_setDialogTitle()
{
  char buf[1024];
  
  sprintf(buf, "Specify User Input Values");
  XtVaSetValues(DialogShell, XmNtitle, buf, NULL);
}



void ConstantsDialog::_setupPortConstants(Widget portContainer)
{
  Widget portLabel;
  char *portName, *inputPortValue;
  XmString portNameX;

  int numConstantPorts = 0;

  int numInPorts = PabloInterface->getNumberInputPorts();

  IsInputPortBoundToConstant = (Boolean *) malloc(sizeof(Boolean)*
						  numInPorts);



  for (int portIdx=0; portIdx<numInPorts; portIdx++) {
    if (PabloInterface->isInputPortBoundToConstant(portIdx)) {
      IsInputPortBoundToConstant[portIdx] = True;
      numConstantPorts++;
    } else {
      IsInputPortBoundToConstant[portIdx] = False;
    }
  }

  PortTextFields = (Widget *) malloc(sizeof(Widget)*numInPorts);

  for (portIdx=0; portIdx<numInPorts; portIdx++) {
    if (IsInputPortBoundToConstant[portIdx]) {
      portName = (char *) PabloInterface-> getInputPortName(portIdx);
      portNameX = XmStringCreateSimple(portName);
      portLabel = 
	XtVaCreateManagedWidget(form("constantInputPortLabel%d", portIdx),
				xmLabelWidgetClass, portContainer,
				XmNlabelString, portNameX,
				NULL);

      PortTextFields[portIdx] = 
	XtVaCreateManagedWidget("portTextField", xmTextFieldWidgetClass, 
				portContainer,
				XmNuserData, (XtPointer) portIdx,
				NULL);

      addCallback(PortTextFields[portIdx], XmNactivateCallback, 
		  &Callback::callback3, this, NULL);

      inputPortValue = (char *) PabloInterface->getInputPortValue(portIdx);

      XmTextFieldSetString(PortTextFields[portIdx], inputPortValue);
      XmStringFree(portNameX);
    }
  }
}



void ConstantsDialog::_setupOutputConstants(Widget outputContainer)
{
  Widget outputLabel;
  char *outputName, *outFieldValue;
  XmString outputNameX;

  int numConstantOutputs = 0;
  int numOutFields = PabloInterface->getNumberOutputFields();

  IsOutputFieldBoundToConstant = (Boolean *) malloc(sizeof(Boolean)*
						    numOutFields);

  for (int outIdx=0; outIdx<numOutFields; outIdx++) {
    if (PabloInterface->isOutputFieldBoundToConstant(outIdx)) {
      IsOutputFieldBoundToConstant[outIdx] = True;
      numConstantOutputs++;
    } else {
      IsOutputFieldBoundToConstant[outIdx] = False;
    }
  }

  OutputTextFields = (Widget *) malloc(sizeof(Widget)*numOutFields);

  for (outIdx=0; outIdx<numOutFields; outIdx++) {
    if (IsOutputFieldBoundToConstant[outIdx]) {
      outputName = (char *) PabloInterface->getOutputFieldName(outIdx);
      outputNameX = XmStringCreateSimple(outputName);
      outputLabel = XtVaCreateManagedWidget(form("constantOutputLabel%d", 
						 outIdx),
					    xmLabelWidgetClass,
					    outputContainer,
					    XmNlabelString, outputNameX,
					    NULL);

      OutputTextFields[outIdx] = 
	XtVaCreateManagedWidget("outputTextField", xmTextFieldWidgetClass, 
				outputContainer, 
				XmNuserData, (XtPointer) outIdx,
				NULL);

      addCallback(OutputTextFields[outIdx], XmNactivateCallback, 
		  &Callback::callback4, this, NULL);

      outFieldValue = (char *) PabloInterface->getOutputFieldValue(outIdx);

      XmTextFieldSetString(OutputTextFields[outIdx], outFieldValue);
      XmStringFree(outputNameX);
    }
  }
}



void ConstantsDialog::callback1(Widget /* button */, XtPointer /* ptr1 */,
				      		     XtPointer /* ptr2 */)
{
  /* callback called when continue is selected */

  // check to see if all user specified values are ok
  char *sendVal, *returnVal;
  Boolean nullValueExists = False, valueChanged = False;

  int numInPorts = PabloInterface->getNumberInputPorts();

  for (int portIdx=0; portIdx<numInPorts; portIdx++) {
    if (IsInputPortBoundToConstant[portIdx]) {
      sendVal = XmTextFieldGetString(PortTextFields[portIdx]);
      returnVal = (char*) PabloInterface->setInputPortValue(portIdx, sendVal);
      XmTextFieldSetString(PortTextFields[portIdx], returnVal);

      if (strcmp(returnVal, "\0") == 0) {
	nullValueExists = True;
      }

      if (strcmp(sendVal, returnVal) != 0) {
	valueChanged = True;
      }
      free(sendVal);
    }
  }

  int numOutFields = PabloInterface->getNumberOutputFields();

  for (int outIdx=0; outIdx<numOutFields; outIdx++) {
    if (IsOutputFieldBoundToConstant[outIdx]) {
      sendVal = XmTextFieldGetString(OutputTextFields[outIdx]);
      returnVal = (char*) PabloInterface->setOutputFieldValue(outIdx, sendVal);
      XmTextFieldSetString(OutputTextFields[outIdx], returnVal);

      if (strcmp(returnVal, "\0") == 0) {
	nullValueExists = True;
      }

      if (strcmp(sendVal, returnVal) != 0) {
	valueChanged = True;
      }
      free(sendVal);
    }
  }

  if (nullValueExists) {
    Pablo::GeneralDialog()->run("A null constant value has been specified.");
  } else if (valueChanged) {
    Pablo::GeneralDialog()->
      run("A value that you specified has been changed by the system.  Please Verify.");
  } else {
    DoneRunning = True;
  }
}



void ConstantsDialog::callback2(Widget /* button */, XtPointer /* ptr1 */,
				      	       	     XtPointer /* ptr2 */)
{
  /* callback called when reset is selected */
  char *inputPortValue, *outputFieldValue;

  PabloInterface->restoreOriginalConstants();

  int numInPorts = PabloInterface->getNumberInputPorts();
  for (int portIdx=0; portIdx<numInPorts; portIdx++) {
    if (IsInputPortBoundToConstant[portIdx]) {
      inputPortValue = (char *) PabloInterface->getInputPortValue(portIdx);
      XmTextFieldSetString(PortTextFields[portIdx], inputPortValue);
    }
  }

  int numOutFields = PabloInterface->getNumberOutputFields();

  for (int outIdx=0; outIdx<numOutFields; outIdx++) {
    if (IsOutputFieldBoundToConstant[outIdx]) {
      outputFieldValue = (char *) PabloInterface->getOutputFieldValue(outIdx);
      XmTextFieldSetString(OutputTextFields[outIdx], outputFieldValue);
    }
  }
}


void ConstantsDialog::callback3(Widget portField, XtPointer /* ptr1 */,
				      		  XtPointer /* ptr2 */)
{
  /* callback called when a constant value is entered for an input port */
  int portNum;
  char *sendVal, *returnVal;

  XtVaGetValues(portField,
		XmNuserData, &portNum,
		NULL);

  /* If the user input value is ambiguous, Pablo will return its best guess 
     about what it should be and that will be displayed in the text field.
     NULL will be returned if Pablo is unable to determine the correct value
     of the user input field */

  sendVal = XmTextFieldGetString(PortTextFields[portNum]);
  returnVal = (char*) PabloInterface->setInputPortValue(portNum, sendVal);
  free(sendVal);

  XmTextFieldSetString(PortTextFields[portNum], returnVal);
}



void ConstantsDialog::callback4(Widget outputField, XtPointer /* ptr1 */,
				      		    XtPointer /* ptr2 */)
{
  /* callback called when a constant value is entered for an output field */
  int outputNum;
  char *sendVal, *returnVal;

  XtVaGetValues(outputField,
		XmNuserData, &outputNum,
		NULL);

  /* If the user input value is ambiguous, Pablo will return its best guess 
     about what it should be and that will be displayed in the text field.
     NULL will be returned if Pablo is unable to determine the correct value
     of the user input field */

  sendVal = XmTextFieldGetString(OutputTextFields[outputNum]);
  returnVal = (char*) PabloInterface->setOutputFieldValue(outputNum, sendVal);
  free(sendVal);

  XmTextFieldSetString(OutputTextFields[outputNum], returnVal);
}



void ConstantsDialog::helpCallback(Widget /* button */, XtPointer /* ptr1 */,
                                      			XtPointer /* ptr2 */)
{
  /* callback called when help is selected */
  Pablo::HelpSystem()->giveHelpOn( "ConstantsDialog" );
}



void ConstantsDialog::run()
{
  DoneRunning = False;

  XtManageChild(DialogShell);

  while (!DoneRunning) {
    XEvent event;
    XtAppNextEvent(XtWidgetToApplicationContext(DialogShell), &event );
    XtDispatchEvent( &event );
  }
  
  XtUnmanageChild(DialogShell);

}



/*
 * Initialize the static data
 */
const char *const ConstantsDialog::MY_CLASS = "ConstantsDialog";







