/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ConfigBoard.cc:   This module implements the configuration binding
 *                   interface "scoreboard"
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Interface/RCS/ConfigBoard.C,v 1.13 1994/04/11 15:59:23 aydt Exp $
 *	
 */
#include <stdlib.h>
#include <Xm/Label.h>

#ifdef ultrix
extern "C" {
/* for some reason, strdup isn't in <string.h> on ULTRIX V4.2 (Rev. 96) */
char* strdup(const char*);
}
#endif

#include "ConfigBoard.h"
#include "ConfigBoardDetail.h"
#include "ConfigBoardPixmaps.h"

#include "CString.h"
#include "DisplayColorTables.h"
#include "InterfaceClass.h"
#include "IntList.h"
#include "Pablo.h"
#include "PabloHelpSystem.h"
#include "VoidPtrList.h"

ConfigBoard::ConfigBoard( Display *display,InterfaceClass *interfaceClassIn)
{
	currentColorIndex = 0;
	numberInputPipes = 0;
	numberInputPorts = 0;
	numberOutputPorts = 0;
	numberOutputFields = 0;

        boardShell = XtVaAppCreateShell( "boardShell", "boardShell",
				         topLevelShellWidgetClass,
				         display,
				            NULL );

	Widget boardMainWindow = XtVaCreateManagedWidget( "boardMainWindow",
					           xmFormWidgetClass,
					           boardShell,
						      NULL );

	configBoardDetail = new ConfigBoardDetail( boardMainWindow,
						   interfaceClassIn );

	inputPipeGraphics = new VoidPtrList(1);
	inputPortGraphics = new VoidPtrList(1);
	outputPortGraphics = new VoidPtrList(1);
	outputFieldGraphics = new VoidPtrList(1);
	outputFieldContainers = new VoidPtrList(1);
	outputFieldList = new VoidPtrList(1);

	fieldExtractedInputPort = new IntList(1);
	fieldExtractedOutputField = new IntList(1);

	interfaceClass = interfaceClassIn;

	_setupStages( boardMainWindow );
	_setupGraphics( boardMainWindow );
	_setupConfigColormap( boardMainWindow );

}


ConfigBoard::~ConfigBoard()
{
	delete inputPipeGraphics;
	delete inputPortGraphics;
	delete outputPortGraphics;
	delete outputFieldGraphics;
	delete outputFieldContainers;
	int i;
	for (i=0; i<outputFieldList->count(); i++) {
            free( outputFieldList->getElement(i) );
        }   
        delete outputFieldList;

	delete inputPipeBindingTable;
	delete inputPortBindingTable;
	delete outputPortBindingTable;
	delete outputFieldToPipeBindingTable;
	delete outputFieldToPortBindingTable;
	
	delete bindingColorTable;

	delete configBoardDetail;

	XFreePixmap( XtDisplay(boardShell), pipePixmap );
	XFreePixmap( XtDisplay(boardShell), userInputPipePixmap );
	XFreePixmap( XtDisplay(boardShell), inputPortPixmap );
	XFreePixmap( XtDisplay(boardShell), outputPortPixmap );
	XFreePixmap( XtDisplay(boardShell), fieldPixmap );

        XtUnrealizeWidget( boardShell );
	XtDestroyWidget( boardShell );
}


int  ConfigBoard::_convertFieldNameToNumber( const char *fieldName ) const
{
        int fieldNumber;
        const char *currentFieldName;
        for (fieldNumber=0; fieldNumber<numberOutputFields; fieldNumber++) {
           currentFieldName = 
			(const char *)outputFieldList->getElement(fieldNumber);
	   if ( strcmp(currentFieldName, fieldName) == 0 ) {
              break;
           }
        }       
	return(fieldNumber);
}


const char * ConfigBoard::_convertFieldNumberToName( int fieldNumber ) const
{
        return( (const char *)outputFieldList->getElement(fieldNumber) );        
}


int ConfigBoard::_convertGraphicToFieldNumber( Widget graphicWidget )
{
        int i;
	for (i=0; i<outputFieldGraphics->count(); i++) {
            if (graphicWidget == (Widget)outputFieldGraphics->getElement(i)) {
               return( i );
	    }
	}
	return (i-1);	// Should never get here - select last field if we do
}


int ConfigBoard::_INTLEN( int value )
{
        int i = 0;

	if (value == 0) return(1);
	while( value > 0 ) {
            value /= 10;
	    i++;
	}
	return(i);
}




void ConfigBoard::_addInputPipe( char *pipeName )
{
	Widget inputPipeIconContainer = XtVaCreateManagedWidget(
					      "inputPipeIconContainer",
					      xmFormWidgetClass,
					      inputPipeWorkWindow,
					         NULL );
	
	char *currentWidgetName = (char *)malloc( 
			    (_INTLEN(numberInputPipes)+1) * sizeof(char) );
        sprintf( currentWidgetName, "%d", numberInputPipes );
	Widget inputPipeGraphic = XtVaCreateManagedWidget(
					   currentWidgetName,
					   xmPushButtonWidgetClass,
					   inputPipeIconContainer,
					   XmNrightAttachment, XmATTACH_NONE,
					      NULL );
	if ( strcmp( pipeName, "User Input") == 0 ) {
           XtVaSetValues( inputPipeGraphic,
			  XmNlabelPixmap, userInputPipePixmap,
			     NULL );
	} else {
           XtVaSetValues( inputPipeGraphic,
			  XmNlabelPixmap, pipePixmap,
			     NULL );
	}
	free(currentWidgetName);
	inputPipeGraphics->addElement( (void *)inputPipeGraphic );
	addCallback( inputPipeGraphic, XmNactivateCallback,
		     &Callback::callback1, this, 
		     inputPipeWorkWindow );
	
	XmString inputPipeLabelString = XmStringCreateSimple(pipeName);
	Widget inputPipeLabel =  XtVaCreateManagedWidget(
					      "inputPipeLabel",
					      xmLabelWidgetClass,
					      inputPipeIconContainer,
			      XmNlabelString, inputPipeLabelString,
			      XmNtopAttachment, XmATTACH_WIDGET,
                              XmNtopWidget, inputPipeGraphic,
			      XmNbottomAttachment, XmATTACH_FORM,
				NULL );
	XmStringFree( inputPipeLabelString );
        _centerGraphicAboveLabel( inputPipeGraphic, inputPipeLabel );
	numberInputPipes++;
}



void ConfigBoard::_addInputPort()
{
        char currentWidgetName[80];
	Widget inputPortIconContainer = XtVaCreateManagedWidget(
					      "inputPortIconContainer",
					      xmFormWidgetClass,
					      inputPortWorkWindow,
					         NULL );
        sprintf( currentWidgetName, "%d", numberInputPorts );
	Widget inputPortGraphic = XtVaCreateManagedWidget(
					    currentWidgetName,
					    xmPushButtonWidgetClass,
					    inputPortIconContainer,
					    XmNlabelPixmap, inputPortPixmap,
					    XmNrightAttachment, XmATTACH_NONE,
					        NULL );

	inputPortGraphics->addElement( (void *)inputPortGraphic );
	addCallback( inputPortGraphic, XmNactivateCallback,
		     &Callback::callback1, this, 
		     inputPortWorkWindow );
					       
	char *portName = (char *)
                        interfaceClass->getInputPortName( numberInputPorts );

	XmString inputPortLabelString = XmStringCreateSimple( portName );
	Widget inputPortLabel =  XtVaCreateManagedWidget(
			      "inputPortLabel",
			      xmLabelWidgetClass,
			      inputPortIconContainer,
			      XmNlabelString, inputPortLabelString,
			      XmNtopAttachment, XmATTACH_WIDGET,
                              XmNtopWidget, inputPortGraphic,
			      XmNbottomAttachment, XmATTACH_FORM,
				  NULL );
	XmStringFree( inputPortLabelString );
        _centerGraphicAboveLabel( inputPortGraphic, inputPortLabel );

	fieldExtractedInputPort->addElement( 0 );
	numberInputPorts++;
}


void ConfigBoard::_addOutputPort()
{
	Widget outputPortIconContainer = XtVaCreateWidget(
					      "outputPortIconContainer",
					      xmFormWidgetClass,
					      outputPortWorkWindow,
					         NULL );

	char *currentWidgetName = (char *)malloc( 
			      (_INTLEN(numberOutputPorts)+1) * sizeof(char) );
        sprintf( currentWidgetName, "%d", numberOutputPorts );
	Widget outputPortGraphic = XtVaCreateManagedWidget(
					    currentWidgetName,
					    xmPushButtonWidgetClass,
					    outputPortIconContainer,
					    XmNlabelPixmap, outputPortPixmap,
					    XmNrightAttachment, XmATTACH_NONE,
						NULL );
	free(currentWidgetName);
	outputPortGraphics->addElement( (void *)outputPortGraphic );
	addCallback( outputPortGraphic, XmNactivateCallback,
		     &Callback::callback1, this, 
		     outputPortWorkWindow );
					       
	char *portName = (char *)
                       interfaceClass->getOutputPortName( numberOutputPorts);

	XmString outputPortLabelString = XmStringCreateSimple( portName );
	Widget outputPortLabel =  XtVaCreateManagedWidget(
			      "outputPorIcontLabel",
			      xmLabelWidgetClass,
			      outputPortIconContainer,
			      XmNlabelString, outputPortLabelString,
			      XmNtopAttachment, XmATTACH_WIDGET,
                              XmNtopWidget, outputPortGraphic,
			      XmNbottomAttachment, XmATTACH_FORM,
				  NULL );
	XmStringFree( outputPortLabelString );
	numberOutputPorts++;
        _centerGraphicAboveLabel( outputPortGraphic, outputPortLabel );
	XtManageChild( outputPortIconContainer );
}



void ConfigBoard::_centerGraphicAboveLabel( Widget graphic, Widget label )
{
	Dimension labelWidth;
	XtVaGetValues( label,
		       XmNwidth, &labelWidth,
		          NULL );

	Dimension graphicWidth;
	XtVaGetValues( graphic,
		       XmNwidth, &graphicWidth,
		          NULL );

	Position shiftedXPosition = (Position)(labelWidth-graphicWidth)/2;
	if ( shiftedXPosition > 0 ) {
	   XtVaSetValues( graphic,
		          XmNleftAttachment, XmATTACH_POSITION,
		          XmNleftPosition, shiftedXPosition,
		             NULL );
	} else {
           XtVaSetValues( label,
		          XmNleftAttachment, XmATTACH_POSITION,
		          XmNleftPosition, shiftedXPosition,
		             NULL );
        }
}



void ConfigBoard::_deleteOutputFieldBinding( int fieldNumber )
{
	int previousPortForField =
                      outputFieldToPortBindingTable->getElement(fieldNumber);

	// First check the output port
	if (previousPortForField >= 0) {
	   int bindsOnPreviousPort =
                    outputPortBindingTable->getElement(previousPortForField);
	   if (bindsOnPreviousPort == 1 ) {
              Display *display = XtDisplay(boardShell);
              XtVaSetValues(
                   (Widget)outputPortGraphics->getElement(previousPortForField),
		   XmNbackground, 
		   WhitePixel( display, DefaultScreen(display)),
		   NULL );
	   }
	   outputPortBindingTable->setElement( previousPortForField,
	      outputPortBindingTable->getElement(previousPortForField) - 1 );
	   outputFieldToPortBindingTable->setElement( fieldNumber, -1 );
        }
        
	// Must also check the input pipe
	int previousPipeForField =
                     outputFieldToPipeBindingTable->getElement(fieldNumber);
	if (previousPipeForField >= 0) {
	    int bindsOnPreviousPipe =
                    inputPipeBindingTable->getElement(previousPipeForField);
 	    if (bindsOnPreviousPipe == 1 ) {
            	Display *display = XtDisplay(boardShell);
                XtVaSetValues(
                    (Widget)inputPipeGraphics->getElement(previousPipeForField),
		    XmNbackground, 
		    WhitePixel( display, DefaultScreen(display)),
		    NULL );
	     }
	     inputPipeBindingTable->setElement( previousPipeForField,
	            inputPipeBindingTable->getElement(previousPipeForField)-1 );
	     outputFieldToPipeBindingTable->setElement( fieldNumber, -1 );
        }
}



Pixel ConfigBoard::_getNewBindingColor( int useExistingColor,
				        Widget widgetUsingExistingColor )
{				       
	Pixel newBindingColor;
	// If a binding exists, use the current pipe color
	if ( useExistingColor ) {
             XtVaGetValues( widgetUsingExistingColor,
			    XmNbackground, &newBindingColor,
			       NULL );
	// Otherwise, get a new color
	} else {
	     newBindingColor =
                   bindingColorTable->getColormapEntry( currentColorIndex++);
	     currentColorIndex = 
                    (currentColorIndex % bindingColorTable->getNumColors() );
	}
	return( newBindingColor );

}


	 
void ConfigBoard::_setupGraphics( Widget boardMainWindow )
{
	Display *display = XtDisplay( boardMainWindow );
	unsigned int defaultScreen = DefaultScreen( display );

	pipePixmap = XCreatePixmapFromBitmapData(
				display,
				RootWindow( display, defaultScreen ),
				pipePixmap_bits, 
				pipePixmap_width, pipePixmap_height,
				BlackPixel( display, defaultScreen ),
				WhitePixel( display, defaultScreen ),
			        DefaultDepth( display, defaultScreen ) );

	userInputPipePixmap =  XCreatePixmapFromBitmapData(
				display,
				RootWindow( display, defaultScreen ),
				userInputPipePixmap_bits, 
				userInputPipePixmap_width,
				userInputPipePixmap_height,
				BlackPixel( display, defaultScreen ),
				WhitePixel( display, defaultScreen ),
			        DefaultDepth( display, defaultScreen ) );

	inputPortPixmap = XCreatePixmapFromBitmapData(
				display,
				RootWindow( display, defaultScreen ),
				inputPortPixmap_bits, 
			        inputPortPixmap_width, inputPortPixmap_height,
				BlackPixel( display, defaultScreen ),
				WhitePixel( display, defaultScreen ),
			        DefaultDepth( display, defaultScreen ) );

	outputPortPixmap = XCreatePixmapFromBitmapData(
				display,
				RootWindow( display, defaultScreen ),
				outputPortPixmap_bits, 
			        outputPortPixmap_width,outputPortPixmap_height,
				BlackPixel( display, defaultScreen ),
				WhitePixel( display, defaultScreen ),
			        DefaultDepth( display, defaultScreen ) );

	fieldPixmap = XCreatePixmapFromBitmapData(
				display,
				RootWindow( display, defaultScreen ),
				fieldPixmap_bits, 
			        fieldPixmap_width, fieldPixmap_height,
				BlackPixel( display, defaultScreen ),
				WhitePixel( display, defaultScreen ),
			        DefaultDepth( display, defaultScreen ) );
}



void ConfigBoard::_setupStages( Widget boardMainWindow )
{
        Widget moduleNameLabel = XtVaCreateManagedWidget(
					    "boardModuleNameLabel",
					    xmLabelWidgetClass,
					    boardMainWindow,
					    XmNtopAttachment, XmATTACH_FORM,
					    XmNrightAttachment, XmATTACH_NONE,
	                                       NULL );
	XmString moduleNameLabelString =  XmStringCreateSimple(
                                    (char *)interfaceClass->getModuleName());
	Widget moduleName = XtVaCreateManagedWidget( "boardModuleName",
			xmLabelWidgetClass,
	  		boardMainWindow,
                        XmNlabelString, moduleNameLabelString,
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_WIDGET,
			XmNleftWidget, moduleNameLabel,
			   NULL );
	XmStringFree( moduleNameLabelString );

	Widget separator1 = XtVaCreateManagedWidget( "boardModuleName",
			xmSeparatorWidgetClass,
			boardMainWindow,
			XmNtopAttachment, XmATTACH_WIDGET,
                        XmNtopWidget, moduleNameLabel,
                           NULL );

	Widget stageForm = XtVaCreateManagedWidget( "boardStageForm",
			xmFormWidgetClass,
			boardMainWindow,
			XmNtopAttachment, XmATTACH_WIDGET,
                        XmNtopWidget, separator1,
                           NULL );

	inputPipeBindingTable = new IntList( 1 );
	_setupStage( stageForm, inputPipeWorkWindow, "inputPipe" );
	
	inputPortBindingTable = new IntList( 1 );
	_setupStage( stageForm, inputPortWorkWindow, "inputPort" );

	outputPortBindingTable = new IntList( 1 );
	_setupStage( stageForm, outputPortWorkWindow, "outputPort" );

	outputFieldToPortBindingTable = new IntList( 1 );
        outputFieldToPipeBindingTable = new IntList( 1 );
	_setupStage( stageForm, outputFieldWorkWindow, "outputField" );

	Widget configBoardControlArea = XtVaCreateManagedWidget( 
					   "configBoardControlArea",
					   xmFormWidgetClass,
					   boardMainWindow,
					   XmNtopAttachment, XmATTACH_WIDGET,
					   XmNtopWidget, stageForm,
						NULL );
	Widget controlDismissButton = XtVaCreateManagedWidget( 
					   "configBoardDismissButton",
					   xmPushButtonWidgetClass,
					   configBoardControlArea,
						NULL );
	addCallback( controlDismissButton, XmNactivateCallback,
		     &Callback::callback2, this, NULL ); 

	Widget controlHelpButton = XtVaCreateManagedWidget(
					   "configBoardHelpButton",
					   xmPushButtonWidgetClass,
					   configBoardControlArea,
						NULL );
	addCallback( controlHelpButton, XmNactivateCallback,
		     &Callback::helpCallback, this, NULL ); 
}


void ConfigBoard::_setupStage( Widget stageParent, 
			       Widget& stageChildWorkWindow,
			       CString childWidgetPrefix )
			       
{
	CString currentWidgetName;
	
        currentWidgetName = childWidgetPrefix + "Container";
        Widget stageContainer = XtVaCreateManagedWidget( currentWidgetName,
					   xmFormWidgetClass,	
	                                   stageParent,
                                              NULL );
					   
	currentWidgetName = childWidgetPrefix + "Label";
	Widget stageTitle = XtVaCreateManagedWidget( currentWidgetName,
					    xmLabelWidgetClass,
					    stageContainer,
						NULL );

	currentWidgetName = childWidgetPrefix + "ScrolledWindow";
	Widget stageChildScrolledWindow = XtVaCreateManagedWidget( 
					   currentWidgetName,
					   xmScrolledWindowWidgetClass,
					   stageContainer,
					   XmNtopAttachment, XmATTACH_WIDGET,
					   XmNtopWidget, stageTitle,
                                               NULL );
	Widget verticalScrollBarID;
        XtVaGetValues( stageChildScrolledWindow,
		       XmNverticalScrollBar, &verticalScrollBarID,
		          NULL );
	if ( verticalScrollBarID ) {
           XtUnmanageChild( verticalScrollBarID );
	}
	currentWidgetName = childWidgetPrefix + "WorkWindow";
        stageChildWorkWindow = XtVaCreateManagedWidget(
					   currentWidgetName,
					   xmRowColumnWidgetClass,
					   stageChildScrolledWindow,
					   XmNorientation, XmHORIZONTAL,
					      NULL );
}



void ConfigBoard::_setupConfigColormap( Widget boardMainWindow )
{
       static char *newColorList [] = { 
	  "#ff0000",
	  "#00ff00",
	  "#0000ff",
	  "#00ffff",
	  "#ff00ff",
	  "#ffff00",	
	  "#b03060",
	  "#458b00"
	};

	bindingColorTable = new DisplayColorTables();
	bindingColorTable->loadNewColormap( newColorList, 8 );
	bindingColorTable->gencolor( XtDisplay(boardMainWindow),
				     XtScreen(boardMainWindow) );
}



void ConfigBoard::addInputPipeOutputFieldBinding( int pipeNumber,
						  const char *fieldName )
{
        int fieldNumber = _convertFieldNameToNumber( fieldName );

        if ( ( (pipeNumber < 0 ) || (pipeNumber >= numberInputPipes) ) ||
             ( (fieldNumber < 0 ) || (fieldNumber >= numberOutputFields) ) ) {
             return;
	}

	Pixel newBindingColor = _getNewBindingColor( 
			    inputPipeBindingTable->getElement(pipeNumber),
			    (Widget)inputPipeGraphics->getElement(pipeNumber) );

	// If this field was previously bound, delete the old binding
	_deleteOutputFieldBinding( fieldNumber );

	// Set the input pipe and output field icons to their new color
        XtVaSetValues( (Widget) inputPipeGraphics->getElement(pipeNumber),
		       XmNbackground, newBindingColor,
		          NULL );
        XtVaSetValues( (Widget) outputFieldGraphics->getElement( fieldNumber ),
		       XmNbackground, newBindingColor,
		          NULL );

	// Update the pipe and port binding tables
	inputPipeBindingTable->setElement( pipeNumber,
			    inputPipeBindingTable->getElement(pipeNumber)+1 );
	outputFieldToPipeBindingTable->setElement(fieldNumber, pipeNumber );

	configBoardDetail->resetDetails();
}



void ConfigBoard::addInputPipePortBinding( int pipeNumber, int portNumber)
{
        if ( ( (pipeNumber < 0 ) || (pipeNumber >= numberInputPipes) ) ||
             ( (portNumber < 0 ) || (portNumber >= numberInputPorts) ) ) {
             return;
	}

	// If a binding exists, use the current pipe color 
        Pixel newBindingColor = _getNewBindingColor( 
			 inputPipeBindingTable->getElement(pipeNumber),
                         (Widget)inputPipeGraphics->getElement(pipeNumber) ); 

	// If this port was previously bound, delete the old binding
	int previousPipeForPort =inputPortBindingTable->getElement(portNumber);

	if (previousPipeForPort >= 0) {
	   int bindsOnPreviousPipe =
                        inputPipeBindingTable->getElement(previousPipeForPort);
	   if (bindsOnPreviousPipe == 1 ) {
              Display *display = XtDisplay(boardShell);
              XtVaSetValues(
		(Widget)inputPipeGraphics->getElement(previousPipeForPort),
	        XmNbackground, WhitePixel( display, DefaultScreen(display)),
		NULL );
	   }
	   inputPipeBindingTable->setElement( previousPipeForPort,
	         inputPipeBindingTable->getElement(previousPipeForPort) - 1 );
	} 

	// Set the input pipe and port icons to their new color
        XtVaSetValues( (Widget)inputPipeGraphics->getElement( pipeNumber ),
                       XmNbackground, newBindingColor,
		          NULL );
        XtVaSetValues( (Widget) inputPortGraphics->getElement( portNumber ),
		       XmNbackground, newBindingColor,
		           NULL );
	// Update the pipe and port binding tables
	inputPipeBindingTable->setElement( pipeNumber, 	
		           inputPipeBindingTable->getElement(pipeNumber)+1 );
	inputPortBindingTable->setElement( portNumber, pipeNumber );

	configBoardDetail->resetDetails();
}



void ConfigBoard::addOutputPortFieldBinding( int portNumber, 
					     const char *fieldName )
{
        int fieldNumber = _convertFieldNameToNumber( fieldName );

        if ( ( (portNumber < 0 ) || (portNumber >= numberInputPorts) ) ||
             ( (fieldNumber < 0 ) || (fieldNumber >= numberOutputFields) ) ) {
             return;
	}

      	// If a binding exists for the port, use the current port color 
        Pixel newBindingColor = _getNewBindingColor(
                          outputPortBindingTable->getElement(portNumber),
			  (Widget)outputPortGraphics->getElement(portNumber) );

	// If this field was previously bound, delete the old binding
	_deleteOutputFieldBinding( fieldNumber );

	// Set the output port and field icons to their new color
        XtVaSetValues( (Widget)outputPortGraphics->getElement( portNumber ),
                       XmNbackground, newBindingColor,
		          NULL );

        XtVaSetValues( (Widget) outputFieldGraphics->getElement( fieldNumber ), 
		       XmNbackground, newBindingColor, NULL );

	// Update the port and field binding tables
	outputPortBindingTable->setElement( portNumber, 	
		         outputPortBindingTable->getElement(portNumber)+1 );
	outputFieldToPortBindingTable->setElement( fieldNumber, portNumber );

	configBoardDetail->resetDetails();
}



void ConfigBoard::addOutputRecordField( const char *outputFieldName )
{
        outputFieldList->addElement( strdup((char *)outputFieldName) );

        Widget outputFieldContainer = XtVaCreateWidget(
					      "outputFieldIconContainer",
					      xmFormWidgetClass,
					      outputFieldWorkWindow,
						 NULL );
        outputFieldContainers->addElement( (void *)outputFieldContainer );
	
	char *currentWidgetName =(char *)malloc( 
 			  (_INTLEN(numberOutputFields)+1)* sizeof(char) );
        sprintf( currentWidgetName, "%d", numberOutputFields );
	Widget outputFieldGraphic = XtVaCreateWidget(
					    currentWidgetName,
					    xmPushButtonWidgetClass,
					    outputFieldContainer,
					    XmNlabelPixmap, fieldPixmap,
					    XmNrightAttachment, XmATTACH_NONE,
					      NULL );
	free(currentWidgetName);
	outputFieldGraphics->addElement( (void *)outputFieldGraphic );
	addCallback( outputFieldGraphic, XmNactivateCallback,
		     &Callback::callback1, this, 
		     outputFieldWorkWindow );
	
	XmString outputFieldIconLabelString = 
	                         XmStringCreateSimple((char *)outputFieldName);
	Widget outputFieldIconLabel =  XtVaCreateWidget(
			      "outputFieldIconLabel",
			      xmLabelWidgetClass,
			      outputFieldContainer,
			      XmNlabelString, outputFieldIconLabelString,
			      XmNtopAttachment, XmATTACH_WIDGET,
			      XmNtopWidget, outputFieldGraphic,
			      XmNbottomAttachment, XmATTACH_FORM,
				 NULL );
	XmStringFree( outputFieldIconLabelString );

	outputFieldToPipeBindingTable->setElement( numberOutputFields, -1 );
	outputFieldToPortBindingTable->setElement( numberOutputFields, -1 );

        _centerGraphicAboveLabel( outputFieldGraphic, outputFieldIconLabel );

	XtManageChild( outputFieldContainer );
        XtManageChild( outputFieldGraphic );
	XtManageChild( outputFieldIconLabel );
	
	fieldExtractedOutputField->addElement( 0 );
	configBoardDetail->registerOutputFieldAddition();
        numberOutputFields++;
}




void ConfigBoard::callback1(Widget graphic,
			    XtPointer workWindow, XtPointer /* ptr2 */)
{
        Pixel background;

        XtVaGetValues( (Widget)workWindow,
		        XmNbackground, &background,
			   NULL );
	
	int indexNumber = atoi( XtName( graphic ) );

        if ( (Widget)workWindow == inputPipeWorkWindow) {
           configBoardDetail->setDescriptor( "Input Pipe Detail", background );
	   configBoardDetail->setName(
	                     interfaceClass->getInputPipeName( indexNumber ) );
	   configBoardDetail->setType( "None" );
           configBoardDetail->fillRecordScrollAreaFromPipe( indexNumber );

	} else if ( (Widget)workWindow == inputPortWorkWindow) {
           configBoardDetail->setDescriptor( "Input Port Detail", background );
	   configBoardDetail->setName(
			     interfaceClass->getInputPortName( indexNumber ) );
           configBoardDetail->setType(
                        interfaceClass->getTraitsForInputPort( indexNumber ) );
	   configBoardDetail->fillRecordScrollAreaFromPort( indexNumber,
    			    inputPortBindingTable->getElement(indexNumber),
			    fieldExtractedInputPort->getElement(indexNumber) );

	} else if ( (Widget)workWindow == outputPortWorkWindow) {
           configBoardDetail->setDescriptor( "Output Port Detail", background);
	   configBoardDetail->setName(
			     interfaceClass->getOutputPortName( indexNumber ));
	   configBoardDetail->setType(
                     interfaceClass->getTraitsForOutputPort( indexNumber ) );
	   configBoardDetail->clearRecordScrollArea();

	} else if ( (Widget)workWindow == outputFieldWorkWindow) {
	   indexNumber = _convertGraphicToFieldNumber( graphic );
	   configBoardDetail->setDescriptor("Output Field Detail", background);
	   configBoardDetail->setName( _convertFieldNumberToName(indexNumber));
           configBoardDetail->setType( 
                    interfaceClass->getTraitsForOutputField( (const char *) 
                                 _convertFieldNumberToName(indexNumber) ) );
           configBoardDetail->fillRecordScrollAreaFromOutputField(indexNumber,
		        outputFieldToPipeBindingTable->getElement(indexNumber),
			fieldExtractedOutputField->getElement(indexNumber) );
	}
        configBoardDetail->realize();
}



void ConfigBoard::callback2( Widget /* callbackWidget */,
			     XtPointer /* ptr1 */, XtPointer /* ptr2 */ )
{
        XtUnrealizeWidget( boardShell );
	configBoardDetail->unrealize();
}


void ConfigBoard::helpCallback( Widget /* callbackWidget */,
			     XtPointer /* ptr1 */, XtPointer /* ptr2 */ )
{
        Pablo::HelpSystem()->giveHelpOn( "ConfigBoard" );
}


void ConfigBoard::clearAllInputPipePortBindings()
{
        Display *display = XtDisplay( boardShell );
        int i;

	for (i=0; i<interfaceClass->getNumberInputPorts(); i++) {
	    // Reset color on this port
            XtVaSetValues( (Widget)inputPortGraphics->getElement(i),
			   XmNbackground, WhitePixel( display,
						      DefaultScreen(display) ),
			      NULL );

	    // Determine if the corresponding pipe has other bindings
	    int pipeNumberForPort = inputPortBindingTable->getElement(i);

	    // First, check if a binding even exists for this port
	    if ( pipeNumberForPort >= 0 ) {
               int pipeBindCount =
		     inputPipeBindingTable->getElement( pipeNumberForPort );

	       // If so, check if the current binding is the only binding
               if ( pipeBindCount == 1) {
                   XtVaSetValues( 
		      (Widget)inputPipeGraphics->getElement(pipeNumberForPort),
		       XmNbackground, WhitePixel( display,
			                          DefaultScreen(display) ),
		       NULL );
	           inputPipeBindingTable->setElement( pipeNumberForPort, 0 );
	       }
	       // Update the binding count for the pipe
	       inputPipeBindingTable->setElement( pipeNumberForPort,
						  pipeBindCount - 1 );
	    }
	    // Finally, update the binding table for this port
	    inputPortBindingTable->setElement( i, -1 );
	}
	configBoardDetail->resetDetails();
}



void ConfigBoard::clearAllInputPipeOutputFieldBindings()
{
        Display *display = XtDisplay( boardShell );
        int i;

	for (i=0; i<numberOutputFields; i++) {
	    // Clear the color for the output record field if it does not
            // also have a binding to an output port
	    if ( outputFieldToPortBindingTable->getElement(i) < 0 ) {
               XtVaSetValues( (Widget)outputFieldGraphics->getElement(i),
			      XmNbackground, WhitePixel( display,
						     DefaultScreen(display) ),
			      NULL );
   	    }
	    int pipeNumberForField =
                                 outputFieldToPipeBindingTable->getElement(i);
	    // Check if this field was actually bound to a pipe
	    if (pipeNumberForField >= 0 ) {
	       int pipeBindCount = 
                      inputPipeBindingTable->getElement( pipeNumberForField );
	       // If so, clear the pipe color only if this was the only 
	       //    binding for that pipe
               if ( pipeBindCount == 1 ) {
                 XtVaSetValues( 
		    (Widget)inputPipeGraphics->getElement(pipeNumberForField),
		    XmNbackground, WhitePixel( display,
					       DefaultScreen(display) ),
		    NULL );
	       }
	       // Update the binding count for the pipe
	       inputPipeBindingTable->setElement( pipeNumberForField, 
						  pipeBindCount-1);  
	    }
	    // Finally, update the binding table for this field
	    outputFieldToPipeBindingTable->setElement( i, -1 );
	}
	configBoardDetail->resetDetails();
}



void ConfigBoard::clearAllOutputPortFieldBindings()
{
        Display *display = XtDisplay( boardShell );
        int i;

	for (i=0; i<numberOutputFields; i++) {
	    // Clear the color for the output record field if it does not
            // also have a binding to an input pipe
	    if ( outputFieldToPipeBindingTable->getElement(i) < 0 ) {
               XtVaSetValues( (Widget)outputFieldGraphics->getElement(i),
			       XmNbackground, WhitePixel( display,
						      DefaultScreen(display) ),
			          NULL );
	    }
   	    int portNumberForField = 
                                  outputFieldToPortBindingTable->getElement(i);
	    // Check if this field was actually bound to a port
	    if ( portNumberForField >= 0 ) {
	       int portBindCount = 
                     outputPortBindingTable->getElement( portNumberForField );
	       // If so, clear the pipe color only if this was the only 
	       //    binding for that pipe
               if ( portBindCount == 1 ) {
                 XtVaSetValues( 
		    (Widget)outputPortGraphics->getElement(portNumberForField),
		    XmNbackground, WhitePixel( display,
					       DefaultScreen(display) ),
		    NULL );
	       }
	       // Update the binding count for the port
	       outputPortBindingTable->setElement( portNumberForField, 
						   portBindCount-1);  
	    }
	    // Finally, update the binding table for this field
	    outputFieldToPortBindingTable->setElement( i, -1 );
	}
	configBoardDetail->resetDetails();
}



void ConfigBoard::clearAllPipePortExtract()
{
        int i;
       	for (i=0; i<interfaceClass->getNumberInputPorts(); i++) {
            configBoardDetail->clearAllPipePortExtract( i );
	}
	configBoardDetail->resetDetails();
}


void ConfigBoard::clearAllPipeOutputExtract()
{
 	int i;
	for (i=0; i<numberOutputFields; i++) {
            configBoardDetail->clearAllPipeOutputExtract( i );
	}
	configBoardDetail->resetDetails();
}


void ConfigBoard::clearSingleOutputPortFieldBinding( int portNumber,
						     const char *fieldName )
{
	int fieldNumber = _convertFieldNameToNumber( fieldName );
	if ( ( (portNumber < 0 ) || (portNumber >= numberInputPorts) ) ||
	     ( (fieldNumber < 0 ) || (fieldNumber >= numberOutputFields) ) ) {
	     return;
	}

        Display *display = XtDisplay( boardShell );

	// Clear the color for the output record field 
        XtVaSetValues( (Widget)outputFieldGraphics->getElement( fieldNumber ),
		       XmNbackground, 
		       WhitePixel( display, DefaultScreen(display) ),
		       NULL );

	// If this is the last field bound to the output port, then clear
	// the port color.
	int portBindCount = outputPortBindingTable->getElement( portNumber );
	if (  portBindCount == 1 ) {
            XtVaSetValues( (Widget)outputPortGraphics->getElement( portNumber ),
		           XmNbackground, 
			   WhitePixel( display, DefaultScreen(display) ),
		    	   NULL );
	}

	// Update the binding count for the port
	outputPortBindingTable->setElement( portNumber, portBindCount-1 );  

	// Finally, update the binding table for this field
        outputFieldToPortBindingTable->setElement( fieldNumber, -1 );

	configBoardDetail->resetDetails();
}



void 
ConfigBoard::clearSinglePipePortExtract( int portNumber, const char *recordName)
{
        configBoardDetail->clearSinglePipePortExtract( portNumber, recordName);

	configBoardDetail->resetDetails();
}


void 
ConfigBoard::clearSinglePipeOutputExtract( const char *fieldName,
					   const char *recordName )
{
        int fieldNumber = _convertFieldNameToNumber( fieldName );
	configBoardDetail->clearSinglePipeOutputExtract( fieldNumber,
						         recordName );
	configBoardDetail->resetDetails();
}


void ConfigBoard::deleteOutputRecordField( char *fieldName )
{
        // Find fieldNumber
        int fieldNumber = _convertFieldNameToNumber( fieldName );

        Widget doomedGraphic = 
                    (Widget)outputFieldContainers->getElement( fieldNumber );
        XtUnrealizeWidget( doomedGraphic );
        XtDestroyWidget( doomedGraphic );

	free( outputFieldList->getElement( fieldNumber ) );
	outputFieldList->deleteElementAt( fieldNumber );
	outputFieldContainers->deleteElementAt( fieldNumber );
	outputFieldGraphics->deleteElementAt( fieldNumber );

	outputFieldToPipeBindingTable->deleteElementAt( fieldNumber );
	outputFieldToPortBindingTable->deleteElementAt( fieldNumber );
	fieldExtractedOutputField->deleteElementAt( fieldNumber );

	configBoardDetail->registerOutputFieldDeletion( fieldNumber );
	numberOutputFields--;

	configBoardDetail->resetDetails();
}


void ConfigBoard::fieldExtractPipeField( const char *fieldName, 
					 const char *recordName,
				         const char *fieldNameForRecord )
{
	int fieldNumber = _convertFieldNameToNumber( fieldName );
        configBoardDetail->fieldExtractPipeField( fieldNumber, recordName,
						  fieldNameForRecord );
	fieldExtractedOutputField->setElement( fieldNumber, 1 );
	configBoardDetail->resetDetails();
}


void ConfigBoard::fieldExtractPipePort( int portNumber, 
					const char *recordName, 
					const char *fieldName )
{
        configBoardDetail->fieldExtractPipePort( portNumber, recordName,
						 fieldName );
	fieldExtractedInputPort->setElement( portNumber, 1 );
	configBoardDetail->resetDetails();
}



void ConfigBoard::realize()
{
      	XtRealizeWidget( boardShell );
}



void ConfigBoard::registerInputPipes()
{
        char *pipeName;
        int i;
	for (i=0; i<interfaceClass->getNumberInputPipes(); i++) {
    	    pipeName = (char *)interfaceClass->getInputPipeName(i);
            _addInputPipe( pipeName );
	    inputPipeBindingTable->addElement( 0 );
	}
	_addInputPipe( "User Input" );
	inputPipeBindingTable->addElement( 0 );

	configBoardDetail->registerNumberInputPipes( numberInputPipes );
}



void ConfigBoard::registerInputPorts()
{
        int i;

	for (i=0; i<interfaceClass->getNumberInputPorts(); i++) {
            _addInputPort();
            inputPortBindingTable->addElement( -1 );
	    configBoardDetail->registerPort(i);
	}
}



void ConfigBoard::registerOutputPorts()
{
        int i;
	for (i=0; i<interfaceClass->getNumberOutputPorts(); i++) {
            _addOutputPort();
	    outputPortBindingTable->addElement( 0 );
	}
}


