/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Keith Shields (shields@cs.uiuc.edu)
 * Contributing Author:	Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InterfaceClass.h:  The interface between a Funtional Unit Wrapper and
 *		      the Binding Dialog.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/InterfaceClass.h,v 1.22 1994/02/25 04:28:23 aydt Exp $
 */

#ifndef InterfaceClass_h
#define InterfaceClass_h

#include "ActionRecord.h"
#include "FUWrapper.h"
#include "Obj.h"

class	ConfigBoard;

#define MAX_NUMBER_INPUT_PORTS 20
#define MAX_NUMBER_OUTPUT_PORTS 20
#define MAX_NUMBER_OUTPUT_FIELDS 20
#define MAX_NUMBER_USER_INPUTS 20
#define MAX_NUMBER_FIELD_BINDINGS 200

enum Sources { Unbound = 0,
      	       Input_Pipe = 1,
	       User_Input = 2,
	       Output_Port = 3
	     };

class InterfaceClass : public Obj {
/*
 * Class that manages communication between the FUWrapper and the 
 * Binding dialog and Scoreboard used to configure the FU Module.
 * Duplicates much of FUWrapper information because we want to keep
 * that constant until configuration changes are 'posted'.  Many of the
 * structures are allocated to fixed size due to problems with dtors
 * in g++ not correctly freeing space allocated 'on the fly'.
 */

private:
 	static const char *const MY_CLASS;      // My class name

	Boolean_		     notInitialized;
	FUWrapper		    *fuWrapper;		
	ExecutionMode_		     runMode;
	Boolean_		     previouslyConfigured;	
	StructureDictionaryInquirer *dictionaryLookup;
	ConfigBoard		    *configBoard;

	/*
	 * recordStruct is used to keep track of records on an input pipe.
	 */
	struct recordStruct {
	    RecordDossier *dossier;
	    int		   tag;
	    ActionType     action;
	} ;

	/*
	 * inputPipeStruct is used to keep track of the information for our
	 * input pipes.  The array inputPipeInfo has an additional entry at
	 * cell inputPipeCnt which is used to track UserInput information.
	 */
	struct inputPipeStruct {
	    const char    *name;
	    int	           inputPortsFilled;
	    int		   outputFieldsFilled;
	    int		   recordCnt;
	    recordStruct* *record;
	} ;
	int		   inputPipeCnt;
	inputPipeStruct   *inputPipeInfo;

	/* 
	 * fieldBindingStruct is used to keep track of bindings from
	 * pipe/record/field to input ports output fields.
	 */
	struct fieldBindingStruct {
	    int		   pipe;
	    int		   record;
	    int		   field;
	    int		   inPort;
	    int		   outField;
	} ;
	int		   fieldBindingCnt;
	fieldBindingStruct fieldBindingInfo[ MAX_NUMBER_FIELD_BINDINGS ];

	/* 
	 * inputPortStruct is used to keep track of the information for our
	 * input ports.  
	 */
	struct inputPortStruct {
	    const char    *name;			 
	    DataTraits     initialTraits;
	    int		   initialConstraintsCnt;
	    Sources	   source;	
	    int		   srcIndex;
	} ;
	int	           inputPortCnt;
	inputPortStruct    inputPortInfo[ MAX_NUMBER_INPUT_PORTS ];

	/*
	 * outputPortStruct is used to keep track of our output port
	 * information.
	 */
	struct outputPortStruct {
	    const char    *name;
	    DataTraits     initialTraits;
	} ;
	int 		   outputPortCnt;
	outputPortStruct   outputPortInfo[ MAX_NUMBER_OUTPUT_PORTS ];

	/*
	 * outputFieldStruct is used to keep track of our output fields and
	 * what is bound to them.   Note that the name here is a CString
	 * so that space is allocated for it.  This is not the case for most
	 * of the structures used by the InterfaceClass.
	 */
	struct outputFieldStruct {
	    CString 	   name;
	    Sources        source; 
	    int		   srcIndex;
	    int		   origIndex;
	    DataTraits	   traits;
	    DataTraits	   origTraits;
	    int		   bindCnt;
	    int		   origCnt;
	} ;
	int	           outputFieldCnt;
	outputFieldStruct  outputFieldInfo[ MAX_NUMBER_OUTPUT_FIELDS ];

	/*
	 * userInputStruct is used to keep track of the user input data
	 * values.
	 */
	struct userInputStruct {
	    Value	   dataValue;
	    int		   inputPort;
	    int		   outputField;
	} ;
	int		   userInputCnt;
	userInputStruct	   userInputInfo[ MAX_NUMBER_USER_INPUTS ];
	Value*		  *shadowConstants;	

	char 	           outputRecordName[256];
	RecordDossier	  *outputDossier;

	/* Method _addStructureToSystem:	Build a structure descriptor
	*					from the output fields & add
	*					it to the global structure
	*					dictionary.  Return the system
	*					tag for the structure         */
	int _addStructureToSystem();

	/* Method _getRealInputPortNum:		Return the absolute input port
	*					number for the given port on 
	*					pipe.                         */
	int _getRealInputPortNum( int pipe, int portIdx ) const;

	/* Method _getRealOutputFieldNum:	Return the absolute output field
	*					number for the given output 
	*					field on pipe.                */
	int _getRealOutputFieldNum( int pipe, int fieldIdx ) const;

	/* Method _inputPortSetValid:		Returns TRUE_ if the set of
	*					input ports is acceptable to 
	*					FU and sets the traits of
	*					output fields satisfied by 
	*					output ports. Otherwise prints 
	*					message	and returns FALSE_    */
	Boolean_ _inputPortSetValid();

	/* Method _nextFieldBindingIdx:		Returns the next field binding
	*					index; aborts if out of range */
	int _nextFieldBindingIdx();

	/* Method _nextOutputFieldIdx:		Returns the next output field 
	*					index; aborts if out of range */
	int _nextOutputFieldIdx();

	/* Method _nextUserInputIdx:		Returns the next user input
	*					index; aborts if out of range */
	int _nextUserInputIdx();

	/* Method _setValueFromString:		Set the value from the string
	*					input based on the traits
	*					given.                        */
	void _setValueFromString( char *string, Value& value,
				  const DataTraits& traits );

	/* Method _showModuleResultBindings:	Display the information on
	*					ModuleResult to Output Field
	*					bindings on the status board  */
	void _showModuleResultBindings();

	/* Method _showInputPortBindings:	Display the information on 
	*					Input Pipe/Record/Field to
	*					Input Port and Output Field
	*					bindings to the status board  */
	void _showInputPortBindings();

	/* Method _showValueAsString:		Format the value and return
	*					a pointer to the string.      */
	const char * _showValueAsString( const Value& value) const;


public:
        /* Method InterfaceClass: 		The constructor.              */
	InterfaceClass( FUWrapper *fuw );

        /* Method ~InterfaceClass: 		The destructor.               */
	~InterfaceClass();

/*********** General-purpose methods *****************************************/

	/* Method getModuleName:		Get the name of the module    */
	const char * getModuleName() const;

	/* Method initialize:			Initialize the internal state
	*					of the class from the FUWrapper
	*					information.                  */
	void initialize();

	/* Method moduleConfiguredPreviously:	Returns TRUE_ if this is a
	*					reconfiguration.              */
	Boolean_ moduleConfiguredPreviously();

	/* Method runDialog:			Create and run the Binding and
	*					Scoreboard dialogs. Called by
	*					FUWrapper. Updates FUWrapper
	*					from internal state before 
	*					returning.                    */
	void runDialog(); 

	/* Method updateConfiguration:		Called to update the FUWrapper
	*					InputPort and InputPipeSockets
	*					with new configuration info.  */
	void updateConfiguration();

/*********** Methods used by Screen1 of the Binding Dialog ******************/

	/* Method bindPipeToInputPort:		Bind input pipe to input port.
	*					If pipe == numPipes, then port
	*					is bound to user input.       */
	void bindPipeToInputPort( int pipe, int port );

	/* Method clearPipeToInputPortBindings:	Clear all pipe to port bindings
	*					bindings.                     */
	void clearPipeToInputPortBindings();

	/* Method getInputPipeName:		Get the name of the pipe      */
	const char * getInputPipeName( int pipe ) const;

	/* Method getInputPortName:		Get the name of the port      */
	const char * getInputPortName( int port ) const;

	/* Method getNumberInputPipes:		Get the number of input pipes */
	int getNumberInputPipes() const;

	/* Method getNumberInputPorts:		Get the number of input ports */
	int getNumberInputPorts() const;	

	/* Method getPipeBoundToInputPort:	Get the pipe bound to port    */
	int getPipeBoundToInputPort( int port ) const;

/********** Methods used first by Screen2 of the Binding Dialog *************/

	/* Method addOutputField:		Add an output field with the
	*					given name to the output record.
	*					Mark that it is satisfied by
	*					the indicated pipe. If pipe is
	*					-1, then it is bound to output
	*					port.                         */
	void addOutputField( char *fieldName, int pipe );

	/* Method createOutputRecord:		Set the output record name, and
	*					build a StructureDescriptor
	*					from the output fields that
	*					have been entered.            */
	Boolean_ createOutputRecord( char *name );

	/* Method getNameOfOutputFieldBoundToInputPipe: Get the name of the
	*					output field bound to the 
	*					indicated pipe.               */
	const char * getNameOfOutputFieldBoundToInputPipe( int pipe, 
							   int fieldIdx ) const;

	/* Method getNumberInputPortsBoundToInputPipe:	Get the number of
	*					input ports bound to specified
	*					pipe.  	                      */
	int getNumberInputPortsBoundToInputPipe( int pipe ) const;

	/* Method getNumberOutputFieldsBoundToInputPipe: Get the number of
	*					output fields bound to specified
	*					pipe. If pipe == -1, report
	*					number bound to output ports. */
	int getNumberOutputFieldsBoundToInputPipe( int pipe ) const;

	/* Method getOutputRecordName:		Get name of the output record */
	const char * getOutputRecordName();

	/* Method isPipeBound:		        Determine if the pipe has been
	*					bound to an input port        */
	Boolean_ isPipeBound( int pipe ) const;

	/* Method removeAllOutputFields:	Removes all output fields and
	*					clears bindings from output 
	*					ports, input pipes, and user
	*					input to output fields.  Only
	*					valid if not previously 
	*					configured.                   */
	void removeAllOutputFields();

/****** Methods used first by Screen3  of the Binding Dialog *******/

	/* Method getInputPortValue:		Returns value of given input
	*					port bound to constant.       */
	const char * getInputPortValue( int portNumber ) const;

	/* Method getNumberOutputFields:	Returns the number of output
	*					fields.                       */
	int getNumberOutputFields() const;

	/* Method getOutputFieldName:		Returns name of given output
	*					field.                        */
	const char * getOutputFieldName( int fieldNumber ) const;

	/* Method getOutputFieldValue:		Returns value of given output
	*					field.                        */
	const char * getOutputFieldValue( int fieldNumber ) const;

	/* Method isInputPortBoundToConstant:	Returns TRUE_ or FALSE_.      */
	Boolean_ isInputPortBoundToConstant( int portNumber ) const;

	/* Method isOutputFieldBoundToConstant:	Returns TRUE_ or FALSE_.      */
	Boolean_ isOutputFieldBoundToConstant( int fieldNumber ) const;

	/* Method restoreOriginalConstants: 	Restores User Input values
	*					to what they were when Screen3
	*					originally entered.  Also
	*					adjusts traits on input ports
	*					and output fields as needed.  */
	void restoreOriginalConstants();

	/* Method setInputPortValue:		Sets value of input port.     */
	const char * setInputPortValue( int portNumber, char *userValue );

	/* Method setOutputFieldValue:		Sets value of output field.   */
	const char * setOutputFieldValue( int fieldNumber, char *userValue );

/********** Methods used first by Screen4 of the Binding Dialog *************/

	/* Method allOutputPortTraitsSet:	Returns true if all the output
	*					ports have their traits set.  */
	Boolean_ allOutputPortTraitsSet();

	/* Method getNameOfFieldInInputPipeRecord: Return name of specified
	*					field.                        */
	const char * getNameOfFieldInInputPipeRecord( int pipe, int record,
						      int field ) const;

	/* Method getNameOfInputPortBoundToInputPipe:	Return the port name
	*					for specified input port      */
	const char * getNameOfInputPortBoundToInputPipe( int pipe, 
							 int port ) const;

	/* Method getNameOfRecordOnInputPipe:	Return the name of record
	*					on input pipe.                */
	const char * getNameOfRecordOnInputPipe( int pipe, int record ) const;

	/* Method getNumberFieldsInInputPipeRecord: Return the number of fields
	*					in specifed record.           */
	int getNumberFieldsInInputPipeRecord( int pipe, int record ) const;

	/* Method getNumberOutputPorts:		Returns the number of output
	*					ports.                        */
	int getNumberOutputPorts() const;

	/* Method getNumberRecordsOnInputPipe:	Return the number of records
	*					on the given input pipe.      */
	int getNumberRecordsOnInputPipe( int pipe ) const;

	/* Method getOutputPortName:		Return the name of the output
	*					port.                         */
	const char * getOutputPortName( int oport ) const;

	/* Method getStatusOfRecordOnInputPipe:	Return the status of the
	*					specified record.             */
	ActionType getStatusOfRecordOnInputPipe( int pipe, int record ) const;

	/* Method isInputRecordFieldBoundToInputPort:  Returns TRUE_ or FALSE_.
	*					record, field, and port are
	*					offsets relative to pipe.     */
	Boolean_ isInputRecordFieldBoundToInputPort( int pipe, 
						     int record, 
						     int field, 
						     int port ) const;
	
	/* Method isInputRecordFieldBoundToOutputField: Returns TRUE_ or FALSE_.
	*					record, field, and port are 
	*					offsets relative to pipe.     */
	Boolean_ isInputRecordFieldBoundToOutputField( int pipe, 
						       int record,
						       int field, 
						       int ofield ) const;

	/* Method isOutputPortBoundToOutputField:	Returns TRUE_ or FALSE_.
	*					ofield is relative to output 
	*					ports.                        */
	Boolean_ isOutputPortBoundToOutputField( int oport, int ofield ) const;

	/* Method linkInputRecordFieldToInputPort:  Returns TRUE_ or FALSE_.
	*					record, field, and port are
	*					offsets relative to pipe.     */
	Boolean_ linkInputRecordFieldToInputPort( int pipe, int record, 
						  int field, int port ); 
	
	/* Method linkInputRecordFieldToOutputField: Returns TRUE_ or FALSE_.
	*					record, field, and port are 
	*					offsets relative to pipe.     */
	Boolean_ linkInputRecordFieldToOutputField( int pipe, 
						    int record,
						    int field, 
						    int ofield );

	/* Method linkOutputPortToOutputField: Returns TRUE_ or FALSE_. ofield
	*				       relative to output ports       */
	Boolean_ linkOutputPortToOutputField( int oport, int ofield );

	/* Method restoreOriginalFieldBindings:	Restore the state to what
	*					it was prior to calling screen
	*					4 routines.                   */
	void restoreOriginalFieldBindings();

	/* Method setStatusOfRecordOnInputPipe:	Set the status of the
	*					specified record.             */
	void setStatusOfRecordOnInputPipe( int pipe, int record, 
					   	     ActionType status ) ;

	/* Method unlinkInputRecordFieldFromInputPort:  Removes the specified
	*					binding.                      */
	void unlinkInputRecordFieldFromInputPort( int pipe, int record, 
						  int field, int port ); 
	
	/* Method unlinkInputRecordFieldFromOutputField:  Removes the specified
	*					binding.                      */
	void unlinkInputRecordFieldFromOutputField( int pipe, int record, 
						    int field, int ofield ); 
	
	/* Method unlinkOutputPortFromOutputField:  Removes the specified
	*					binding.                      */
	void unlinkOutputPortFromOutputField( int oport, int ofield );
	
/********** Methods used first by Screen5 of the Binding Dialog *************/

	/* Method getModuleExecutionMode:	Returns execution mode        */
	ExecutionMode_ getModuleExecutionMode() const;

	/* Method setModuleExecutionMode:	Sets execution mode           */
	void setModuleExecutionMode( ExecutionMode_ mode );

/********** Methods used by the Scoreboard *********************************/

	/* Method getTraitsForInputPort:	Returns traits for specified
	*					input port.                   */
	const char * getTraitsForInputPort( int portNumber ) const;

	/* Method getTraitsForOutputPort:	Returns traits for specified
	*					output port.                  */
	const char * getTraitsForOutputPort( int portNumber ) const;

	/* Method getTraitsForOutputField:	Returns traits for specified
	*					output port.                  */
	const char * getTraitsForOutputField( const char* fieldName ) const;
};

#endif
