/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Author: Ruth Aydt (aydt@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 */
 /*
 * InfrastructureManager.h: Implements InfrastructureManager.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/InfrastructureManager.h,v 1.16 1994/02/25 04:29:41 aydt Exp $
 *	
 */

#ifndef InfrastructureManager_h
#define InfrastructureManager_h

#include <stdio.h>

#include "Obj.h"
#include "IntList.h"
#include "CString.h"

class ConnectionId;
class ConnectionIdPtrList;
class ModuleId;
class ModuleIdPtrList;

class InfrastructureManager : public Obj {
/*
 * The InfrastructureManager handles all the details of creating and 
 * connecting wrappers and pipes; and generally keeps an eye on things.
 */
private:
	static const char *const MY_CLASS;	// Class name

	ModuleIdPtrList     *moduleIds; 
	ModuleIdPtrList     *sortedModuleIds;
	ConnectionIdPtrList *connectionIds;

	Boolean_	modulesConfigured;

	/* Method _deleteMyObjects:		Delete the object instances
	*					associated with this IM. 
	*					Called by the destructor and
	*					reset methods.                */
	void _deleteMyObjects();

	/* Method _reSort:			Regenerate the list of sorted
	*					modules after and addition or
	*					deletion.                     */
	void _reSort();

	/* Method _topSortAux:			Helper method for sorting the
	*					list of module ids.           */
	void _topSortAux( ModuleId *id, ModuleIdPtrList *ids,
			  IntList &marked, ModuleIdPtrList *sorted, 
			  int &sortIdx, IntList &stack );

public:
	/* Method InfrastructureManager:	The constructor               */
	InfrastructureManager();

	/* Method ~InfrastructureManager:	The destructor                */
	~InfrastructureManager();

	/* Method configure:			Configure the modules in the
	*					system in sorted order.       */
	void configure();

	/* Method isConfigured:			Returns TRUE_ if the modules 
	*					are configured.               */
	Boolean_ isConfigured() const;

	/* Method loadConfigurationFromDir:	Load our configuration from
	*					the specified directory.      */
	Boolean_ loadConfigurationFromDir( const CString& dir );

	/* Method lookupModule:			Return a pointer to the 
	*					specified module.             */
	ModuleId * lookupModule( int moduleIndex ) const;

	/* Method reset:			Resets the state to what it 
	*					was just after the ctor.      */
	void reset();

	/* Method restart:			Restarts the current execution
	*					graph from the beginning, but
	*					maintains current configuration
	*					parameters, display placements,
	*					etc.			      */
	void restart();

	/* Method run:				Run the graph the specified
	*					number of times, singleStepping
	*					if noted. Returns TRUE_ if 
	*					any module ran the last type
	*					through the cycle.            */
	Boolean_ run( int nPasses = 100, Boolean_ singleStepping = FALSE_ );

	/* Method saveConfigurationToDir:	Save the configuration 
	*					information to the specified
	*					directory.                    */
	Boolean_ saveConfigurationToDir( const CString& dir ) const;

	/* Method writeLayoutToFP: 		Write the layout of the nodes 
	*					and edges to the given file   */
	Boolean_ writeLayoutToFP( FILE *fp );


	/*------------ Methods Related to Modules and Connections ------------*/
	
	/* Method addConnection:	Add a pipe connecting the two modules.
	*				Update the associated wrappers with the
	*				pipe information. Return FAILURE_ 
	*				if unable to add the connection.      */
	Boolean_ addConnection( int fromIdx, int toIdx );

	/* Method addFileInputModule:	Create a file input wrapper and the
	*				module associated with it.  Return 
	*				the index of the module.              */
	int addFileInputModule( const CString& moduleName, 
				const CString& fileName );

	/* Method addFileOutputModule:	Create a file output wrapper and the
	*				module associated with it.  Return 
	*				the index of the module.              */
	int addFileOutputModule( const CString& moduleName, 
				 const CString& fileName );

	/* Method addModule:		Create a FunctionalUnit of the 
	*				specified type and also the wrapper
	*				and module associated with it.  Return
	*				the index of the module or -1 if an
	*				error occurs.                         */
	int addModule( const CString& fuName, const CString& moduleName );

	/* Method deleteModule:		Delete the specified module, adjusting
	*				the index values of other modules as
	*				necessary to adjust for deleted module.
	*				Also deletes any incoming pipes and
	*				adjusts the wrappers of the modules
	*				where they originate. Prints warning 
	*				and returns FAILURE_ if module can't 
	*				be deleted.                           */
	Boolean_ deleteModule( int moduleIndex );
};

inline Boolean_
InfrastructureManager::isConfigured() const
{
	return modulesConfigured;
}

#endif
