/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                      Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613 DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FunctionalUnit.h: The base class for all functional units.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/FunctionalUnit.h,v 1.24 1994/03/15 16:25:35 aydt Exp $
 */

#ifndef FunctionalUnit_h
#define FunctionalUnit_h

#include "CString.h"
#include "Pablo.h"
#include "Obj.h"
#include "WarningMsgs.h"

class InputPort;
class OutputPort;
class FUWrapper;
class Wrapper;

class FunctionalUnit : public Obj {
/*
 * The base class for all functional units (display, math, etc).
 */

private:
	FUWrapper *wrapper;
	int 	   baseClassConfigCnt;	// used by updateParameters()

protected:
	/* Method _addInputPort:	Associate input port with Functional
	*				Unit.				      */
	void _addInputPort( InputPort *port ) const;

	/* Method _addOutputPort:	Associate output port with Functional
	*				Unit.			              */
	void _addOutputPort( OutputPort *port ) const;

        /* Method _writeOutput:         Method called by FunctionalUnit::run()
        *                               to write the output port data to
        *                               the output pipe.  Really just an
        *                               interface to the wrapper's 
	*				writeOutputRecord() method.           */
        void _writeOutput() const;

public:
        /* Method FunctionalUnit: 	The FunctionalUnit constructor.       */
	FunctionalUnit();

        /* Method ~FunctionalUnit: 	The FunctionalUnit destructor.        */
	~FunctionalUnit();

	/* Method configure:		Method to configure things specific to
	*				a particular Functional Unit.	      */
	virtual void configure() { 
		baseClassConfigCnt++;
		} ;

	/* Method configureOperation:	Method to configure the operation for
	*				a functional unit that may perform more
	*				than one task. Called before ports are
	*				bound.                                */
	virtual void configureOperation() { 
		baseClassConfigCnt++;
		} ;

        /* Method copy:                 Method to make a copy of a functional
	*                               unit.  This should be implemented by
	*                               the subclasses.                       */
	virtual FunctionalUnit * copy() { 
		subclassResponsibility( "FunctionalUnit::copy" ); 
		return NULL;
	} ;

	/* Method getName:		Get name associated with this instance
	*				of the Functional Unit.		      */
	CString getName() const;

	/* Method loadConfigFromFile:	Method to load configuration information
	*				for a Functional Unit from the file
	*				specified.  If problems reading file,
	*				the configureOperation() and configure()
	*				methods are usually called to allow 
	*				interactive input.                    */
	virtual Boolean_ loadConfigFromFile( const CString& fileName );


	/* Method init:			Method to initialize a functional unit.
	*				This should be implemented by the
	*				subclasses.			      */
	virtual void init() { 
		subclassResponsibility( "FunctionalUnit::init" ); 
	 	} ;

	/* Method inputTraitsValid: 	Called by the wrapper to verify that
	*				the input types are acceptable.  If
	*				OK, then this sets the currentTraits
	*				fields for the output ports.  Returns
	*				FAILURE_ if the input traits are not 
	*				OK.			              */
	virtual Boolean_ inputTraitsValid() { 
		return TRUE_; 
		} ;

	/* Method restart:		Clear any internal data state in the
	*				functional unit but not any 
	*				configuration parameters.  Display FUs
	*				should also clear their widgets.      */
	virtual void restart(){
		} ;

	/* Method run:			Run the functional unit.  Should be
	*				implemented by the subclasses. If
	*				an error occurs that causes a dialog
	*				to be displayed (like a bad data type)
	*				errorFlag is set to TRUE_ by the 
	*				method.                               */
	virtual void run( Boolean_& errorFlag ) { 
		subclassResponsibility( "FunctionalUnit::run" ); 
		errorFlag = TRUE_;
		} ;

	/* Method saveConfigToFile:	Save enough configuration information
	*				in the file named to restore FU to
	*				current configuration.                */
	virtual Boolean_ saveConfigToFile( const CString& fileName ) const;

	/* Method setWrapper:		Associate given wrapper with this
	*				instance of the Functional Unit.      */
	void setWrapper( Wrapper *wrapper );

	/* Method updateParameters:	Displays and updates the FU parameters.
	*				Calls configureOperation() and 
	*				configure() and prints message if 
	*				neither are overloaded by the derived
	*				FU class.	                      */
	void updateParameters() ;


	/* Method printOn: 		Stream output function 		      */
	virtual void printOn( ostream& os = cout ) const ;
};

inline ostream& operator<<( ostream& os, FunctionalUnit& obj )
{
	obj.printOn(os);
	return os;
}

#endif FunctionalUnit_h
