/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FUWrapper.h: A wrapper that encloses all functional units.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/FUWrapper.h,v 1.22 1994/03/15 16:27:36 aydt Exp $
 */

#ifndef FUWrapper_h
#define FUWrapper_h

#include "Wrapper.h"

#include "Assert.h"
#include "PacketHeader.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"
#include "StructureDescriptorResolver.h"

class FunctionalUnit;
class InputPipeSocket;
class InputPort;
class ModuleId;
class OutputPort;

enum ExecutionMode_ { SYNCHRONOUS_ = 0,  ASYNCHRONOUS_= 1  }; 

class FUWrapper : public Wrapper {
/*
 * FUWrapper is an enclosing class for all functional units (FUs).
 * It provides basic support for ports and data access.
 */
	class UserInputData {
	    friend class FUWrapper;
	    friend class InterfaceClass;

	private:
	    Value	dataValue;		// "constant" value
	    int		inputPort;		// input port bound to
	    int		outputField;		// output field bound to

	    UserInputData( const Value& value, int inPort, int outField )
                { dataValue=value; inputPort=inPort; outputField=outField; }
	} ;

private:
 	static const char * const MY_CLASS;             // My class name

	CString 		fuClassName;		// Functional Unit info
	FunctionalUnit 		*functionalUnit;

	int			inputPortCnt;
	InputPort*	 	*inputPorts;		// Input Ports

	int			outputPortCnt;
	OutputPort*		*outputPorts;		// Output Ports

	int			inputSocketCnt;
	InputPipeSocket*	*inputSockets;		// Input Pipe Sockets

	RecordDossier		*outputDossier;		// Output Record Dossier
	int			outputBindCnt;	
	int			*outputBindings;	

	int			   userInputCnt;	// User-Input data
	FUWrapper::UserInputData*  *userInputs;

	StructureDescriptorResolver resolver;

	ExecutionMode_		runMode;

public:
//	***** Constructors and Destructors *****

        /* Method FUWrapper: 			The constructor.              */
	FUWrapper( FunctionalUnit *fu, const CString& className );

        /* Method ~FUWrapper: 			The destructor.               */
	~FUWrapper();

//	***** Virtual Functions *****

	/* Method addInputPipe:			Add an input pipe and create an
	* 					InputPipeSocket for it.       */
	virtual Boolean_ addInputPipe( StreamPipe *pipe, ModuleId *source );

	
	/* Method configure:			Perform operations necessary to 
	*					configure this module.        */
	virtual void configure( Boolean_ singleModule );

	/* Method configureParams:		Display FU params and allow
	*					changes to them.  IF module not
	*					yet configured, display message
	*					and call configure().         */
	virtual void configureParams( );

	/* Method flushOutputDictionary:  	Make sure output dictionary is
	*					complete and send it down the
	*					output pipe(s).               */
	virtual void flushOutputDictionary();

	/* Method getFunctionalUnit:		Return pointer to FU associated
	*					with this Wrapper.            */
	virtual FunctionalUnit * getFunctionalUnit() const;

	/* Method getWrapperClass:              Returns the class of the
        *                                       Wrapper instance for which it
        *                                       is called.                    */
        virtual WrapperClass getWrapperClass() const {
		return FUWrapperClass;
		} ;

	/* Method init:				Initialize Wrapper state. Also
	*					initializes FU.  Input and  
	*					Output Pipes not reset -- all
	*					else is (bindings, dict, ...) */
	virtual void init();

	/* Method loadConfigurationFromDir:	Load saved config information */
	virtual Boolean_ loadConfigurationFromDir( const CString& dir,
						   int moduleIndex );

	/* Method ready:			Returns TRUE_ if data is 
	*					avaliable on all input pipes.
	*					Processes Discard and 
	*					CopyThrough Data packets.  
	*				        Also, Attribute, Command, and 
	*					Descriptor packets 	      */
	virtual Boolean_ ready();

 	/* Method restart:                      Clears any internal data state
        *                                       associated with the module while
        *                                       maintaining binding and 
        *                                       configuration parameters. This
	*					basically calls the FU method
        *					"restart" to do the work.     */
        virtual void restart();

	/* Method run:				Cause the FU to run, processing
	*					data on all input pipes. 
	*					errorFlag is set if an error
	*					was seen when running the FU. */
	virtual void run( Boolean_& errorFlag );

	/* Method saveConfigurationToDir:	Save the configuration info.  */
	virtual Boolean_ saveConfigurationToDir( const CString& dir,
						 int moduleIndex ) const;

	/* Method writeLayoutToFP:		Save layout information.      */
	virtual void writeLayoutToFP( FILE *fp ) const;

//	***** Non-Virtual Functions *****

	/* Method addInputPort:       		Add an input port	      */
	void addInputPort( InputPort *port );

	/* Method addOutputPort:       		Add an output port	      */
	void addOutputPort( OutputPort *port );

	/* Method addUserInput:			Add a user-input constant bound
	*					to input port or output field */
	void addUserInput( const Value& value, int inPort, int outField );

	/* Method bindOutputPort:		Bind the indicated output port
	*					to the indicated field in the
	*					outputDossier record. If oport 
	*                                       is -1, the call removes a
	*					previous output port binding  */
	void bindOutputPort( int oport, int ofield );

	/* Method getExecutionMode:		Return run mode.              */
	ExecutionMode_ getExecutionMode() const;

	/* Method getInputPort:        		Retrieve input port 'n'	      */
	InputPort * getInputPort( int n ) const;

	/* Method getInputPort:        		Retrieve input port 'pname'   */
	InputPort * getInputPort( const CString& pname ) const;

	/* Method getInputSocket:		Retrieve InputPipeSocket 'n'  */
	InputPipeSocket * getInputSocket( int n ) const;

	/* Method getDossier:			Returns pointer to output 
	*				        dossier.                      */
	RecordDossier * getDossier() const;

	/* Method getOutputPort:        	Retrieve output port 'n'      */
	OutputPort * getOutputPort( int n ) const;

	/* Method getOutputPort:        	Retrieve output port 'pname'  */
	OutputPort * getOutputPort( const CString& pname ) const;

	/* Method getOutputPortBindingForField: Returns the port bound to the
	*					output field.  -1 ==> none    */
	int getOutputPortBindingForField( int field )const;

	/* Method getUserInput:			Return pointer to UserInput
	*					structure.                    */
	FUWrapper::UserInputData * getUserInput( int i ) const;

	/* Method inputPortCount:		Return number of input ports  */
	int inputPortCount() const;

	/* Method inputSocketCount:		Return number of input pipes. */
	int inputSocketCount() const;

	/* Method outputPortCount: 		Return number of output ports */
	int outputPortCount() const; 

	/* Method resetConfiguration:		Called by InterfaceClass to
	*					clear configuration parameters
	*					before resetting them.        */
	void resetConfiguration();

	/* Method setExecutionMode:		Sets the mode of execution.   */
	void setExecutionMode( ExecutionMode_ newMode );

	/* Method setOutputRecord:		Sets outputDossier to the
	*					record indicated by the tag.
	*					Inits outputBindings to -1's.
	*					Returns FAILURE_ if record not
	*					in global structure dict.     */
	Boolean_ setOutputRecord( int sysTag );

	/* Method userInputCount:		Return number of user input
	*					values.                       */
	int userInputCount() const;

	/* Method writeOutputRecord:		Update any fields in the output
	*					record that are bound to Output
	*					Ports and write the completed 
	*					record to the Output Pipe(s).
	*					Called by FU.                 */
	void writeOutputRecord();

	/* Method writePacket:			Write indicated packet to the
	*					output pipe(s).  Called by
	*					InputPipeSocket.              */
	void writePacket( const PacketHeader& pktHdr, 
					            const char* pktData ) const;

//	***** Functions Defined for all Classes *****

	/* Method printOn: 			Stream output function        */
	virtual void printOn( ostream& os = cout ) const ;
};

//	***** Inline Function Definitions *****

inline ExecutionMode_ 
FUWrapper::getExecutionMode() const
{
	return runMode;
}
	
inline InputPort *
FUWrapper::getInputPort( int n ) const		
{
	Assert(  (n >= 0) && (n < inputPortCnt) );
	return inputPorts[n];
}

inline InputPipeSocket *
FUWrapper::getInputSocket( int n ) const
{
	Assert(  (n >= 0) && (n < inputSocketCnt) );
	return inputSockets[n];
}

inline RecordDossier *
FUWrapper::getDossier() const
{
	return outputDossier;
}

inline OutputPort *
FUWrapper::getOutputPort( int n ) const
{
	Assert(  (n >= 0) && (n < outputPortCnt) );
	return outputPorts[n];
}

inline int
FUWrapper::getOutputPortBindingForField( int field ) const
{
	Assert( field < outputBindCnt );
	return outputBindings[ field ];
}

inline FUWrapper::UserInputData *
FUWrapper::getUserInput( int i ) const
{
	Assert( (i >= 0) && (i < userInputCnt) );
	return userInputs[i];
}
	
inline int
FUWrapper::inputPortCount() const
{
	return inputPortCnt; 
}

inline int
FUWrapper::inputSocketCount() const	
{
	return inputSocketCnt; 
}
	
inline int
FUWrapper::outputPortCount() const
{
	return outputPortCnt; 
}

inline void
FUWrapper::setExecutionMode( ExecutionMode_ newMode ) 
{
	runMode = newMode;
}

inline int
FUWrapper::userInputCount() const
{
	return userInputCnt;
}

inline void
FUWrapper::writePacket( const PacketHeader &pktHdr, const char* pktData ) const
{
	if ( binaryOutput != NULL ) {
	    binaryOutput->putPacket( pktHdr, pktData );
	}
}

//	***** General Operators *****

inline ostream& operator<< ( ostream& os, FUWrapper& obj )
{
	obj.printOn(os);
	return os;
}

#endif
