/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BooleanArray.h: Class used to maintain arrays of True/False values.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/BooleanArray.h,v 1.7 1994/02/25 04:31:12 aydt Exp $
 */

#ifndef BooleanArray_h
#define BooleanArray_h

#include "Defines.h"
#include "Obj.h"

class BooleanArray : public Obj {
/*
* The BooleanArray Class is used to represent and manipulate True/False values
* in variable-length, multidimensional arrays.  The array is of unsigned chars
* to save space, but values communicated through the methods are all type
* Boolean_ Once a BooleanArray instance is created, the only way it's 
* dimension can be changed is with the = operator.
*/

private:
	static const char *const MY_CLASS;	// My class name

	int	 	dimension;		// Number of dimensions
	int 	 	cellCount;		// Total number of cells
	int 	 	*cellsInDim;		// Cells in each dimension
	int	 	*strides;		// Offsets for each dimension
	unsigned char 	*data;			// Points to actual t/f array

public:
	/* Method BooleanArray:     Constructor with dimension specified. No
	*			    space is allocated for the array cells    */
	BooleanArray( int _dimension );

	/* Method BooleanArray:      Constructor with copy initialization     */
	BooleanArray( const BooleanArray& array );

	/* Method BooleanArray:      Constructor with pointer to existing 
	*			     BooleanArray instance as an argument.    */
	BooleanArray( const BooleanArray *arrayP );

	/* Method ~BooleanArray:     The Destructor.			      */
	~BooleanArray();

	/* Method getCellCount:	     Return total number of cells in array.   */
	int getCellCount() const;

	/* Method getCellValue:	     Return value of specified cell. 
	*			     Specialized for vectors to enhance 
	*			     performance.                             */
	Boolean_ getCellValue( int dim0Index ) const;

 	/* Method getCellValue:      Return value of specified cell. 
	*			     Specialized for 2-D arrays to enhance 
	*			     performance.			      */
        Boolean_ getCellValue( int dim0Index, int dim1Index ) const;

 	/* Method getCellValue:      Return value of specified cell. General
	*			     method for arrays with >2 dimensions.    */
        Boolean_ getCellValue( int dim0Index, int dim1Index, 
			       int dim2Index ... ) const;

	/* Method getDimSizes: 	     Return pointer to integer array giving
	*			     the number of cells in each dimension    */
	const int * getDimSizes() const;

	/* Method getDimension:	     Return the dimension of the array	      */
	int getDimension() const;

	/* Method getTheCellValue:   Return value of the cell, where index
	*			     is the absolute index into the array
	*			     rather than an index into each dimension */
	Boolean_ getTheCellValue( int index ) const;

	/* Method setCellValue:	   Set specified cell to given value. Returns
	*			   FAILURE_ if index out of bounds or if
	*			   dimension sizes of array not yet set.      */
	Boolean_ setCellValue( Boolean_ value, int dim0Index ... );

	/* Method setDimSizes:     Set number of cells in each dimension from 
	*			   the array of integers pointed to by intArray.
	*			   The data area is resized and initialized 
	*			   to FALSE_.		                      */
	void setDimSizes( const int *intArray );

	/* Method setTheCellValue: Set cell to given value, where the absolute
	*			   index into the array is given rather than
	*			   an index for each dimension. Returns 
	*			   FAILURE_ if index out of bounds or if 
	*			   dimension sizes of array not yet set.      */
	Boolean_ setTheCellValue( int index, Boolean_ value );

	/* Method operator=:	   BooleanArray to Boolean Array assignment.  
	*			   Upon completion, both BooleanArray instances
	*		           have copies of the data.  	              */
	BooleanArray& operator=( const BooleanArray& array );

	/* Method printOn:	   Helper function for BooleanArray output    */
	virtual void printOn( ostream& strm = cout ) const;

};

inline int 
BooleanArray::getCellCount() const
{ 
	return cellCount; 
}

inline const int* 
BooleanArray::getDimSizes() const
{ 
	return cellsInDim; 
} 

inline int 
BooleanArray::getDimension() const
{ 
	return dimension; 
}

inline ostream& operator<<( ostream& os, BooleanArray& array)
{
	array.printOn( os );
	return os;
}

#endif

