/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ActionRecord.h: Specifies an action to be taken for a given record.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/ActionRecord.h,v 1.7 1994/02/25 04:28:16 aydt Exp $
 */

#ifndef ActionRecord_h
#define ActionRecord_h

#include <stdio.h>

#include "Obj.h"

enum ActionType {
        ActionDiscard,
        ActionCopyThru,
        ActionExtract,
        ActionCopyThruAndExtract,
        ActionInvalid,
        LastAndUnusedActionType,
};

static const int NActionTypes = LastAndUnusedActionType;

static const char *ActionTypeNames[] = {
        "Discard",
        "Copy_Through",
        "Field_Extract",
        "Copy_Through_And_Extract",
        "Invalid",
};

#define ActionDoesExtraction( atype ) \
        ( ((atype) == ActionExtract) || ((atype) == ActionCopyThruAndExtract) )

#define ActionDoesCopy( atype )  \
        ( ((atype) == ActionCopyThru) || ((atype) == ActionCopyThruAndExtract) )

#define ActionIsSet( atype )  \
        ( (atype) != ActionInvalid )


class ActionRecord : public Obj {
/*
 * 	An ActionRecord specifies the action to be taken on Record Data packets
 *      with a given tag.  The tag is used as an index into a table of
 *	ActionRecords maintained by the ActionTable class - each InputPipeSocket
 * 	has and ActionTable associated with it.  The ActionRecord also contains
 *	two arrays of integers -- the first has an entry for each InputPort
 *	to be filled by the record; the second has an entry for each field
 *	in the FU's Output Record to be filled by the record.  The integer
 *	value at cell N in one of these arrays says that field N from the
 *	input Record is bound to the corresponding Input Port or Output
 *	Record field.   The fieldToInput and fieldToOutput arrays are only
 *	guaranteed to be valid if the ActionType is ActionExtract or 
 *	ActionCopyThruAndExtract.   Only the ActionList and InputPipeSocket
 *	classes may access the member variables and methods of the ActionRecord
 *	class.
 */
	friend class ActionTable;
	friend class InputPipeSocket;

private:
	ActionType	action;
	int		inputCnt;
	int		*fieldForInput;
	int		outputCnt;
	int		*fieldForOutput;

	/* Method ActionRecord: 	The constructor. Defaults to 
	*				ActionInvalid.			      */
	ActionRecord( );

	/* Method ~ActionRecord: 	The destructor.		 	      */
	~ActionRecord();

	/* Method _actionName:		char* representation of Action        */
	const char * _actionName() const;

	/* Method _bindToInputPort:	Bind the field to the input port.     */
	void _bindToInputPort( int fieldId, int inputPortIdx );

	/* Method _bindToOutputField:	Bind the field to the output field.   */
	void _bindToOutputField( int fieldId, int outputFieldIdx );

        /* Method _loadConfiguration:   Load configuration from filestream    */
        void _loadConfiguration( FILE *fp );

	/* Method _reset:		Resets the action.  For Extraction
	*				actions, inputs and outputs reset &
	*				fieldForInput and fieldForOutput 
	*				entries all set to -1's.  For 
	*				non-Extraction types, expect inputs 
	*				and outputs to be zero.               */
	void _reset( ActionType type, int inputs = 0, int outputs = 0 );

        /* Method _saveConfiguration:   Save the table entries to the indicated
        *                               file stream.                          */
        void _saveConfiguration( FILE *fp ) const;

	/* Method _setInputCount:	For Extraction types, adjust the size 
	*				of fieldForInput to count and reset 
	*				cells to -1's.  For non-Extraction,
	*				set to zero size.                     */
	void _setInputCount( int count );

	/* Method _setOutputCount:	For Extraction types, adjust the size 
	*				of fieldForOutput to count and reset 
	*				cells to -1's.  For non-Extraction,
	*				set to zero size.                     */
	void _setOutputCount( int count );

public:
        /* Method printOn:         	Helper function for output	      */
	virtual void printOn( ostream& strm = cout ) const;
};

inline const char*
ActionRecord::_actionName() const
{
	return ActionTypeNames[ action ];
}

inline ostream& operator<<( ostream& os, ActionRecord& ar )
{
	ar.printOn( os );
	return os;
}

#endif
