/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Keith A. Shields (shields@cs.uiuc.edu) 
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * XYGraphFU.cc - A polymorophic functional unit for a dynamic line plot
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/XYGraphFU.C,v 1.32 1994/03/15 16:42:13 aydt Exp $
 */

#include "XYGraphFU.h"
#include "XYGraphFormWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define NormX(x) ( ( ((double) x) - minX ) / (maxX - minX) )
#define NormY(y) ( ( ((double) y) - minY ) / (maxY - minY) )
#define Round(x) ( (int)(x + .5) )

XYGraphFU::XYGraphFU()
{
	_setClassName( MY_CLASS );

	input = NULL;
	dialog = NULL;
	xygraphForm = NULL;

        currentxptr = NULL;
        currentyptr = NULL;
}


XYGraphFU::~XYGraphFU()
{
      delete input;
      delete xygraphForm;
      delete dialog;
     
      /* free up space holding x and y values */
      if (currentxptr != NULL) {
           delete currentxptr;
           delete currentyptr;
	}
}



void 			/* virtual */
XYGraphFU::configure()
{
#ifndef XtNxticks
#define XtNxticks "xticks"
#define XtNyticks "yticks"
#endif

#ifndef XtNminx
#define XtNminx                 "minx"
#define XtNminy                 "miny"
#define XtNmaxx                 "maxx"
#define XtNmaxy                 "maxy"
#endif

#ifndef XtNshowRange
#define XtNconnect	"connect"
#define XtNshowRange	"showRange"
#define XtNshowGrid	"showGrid"
#define XtNmarkerSize	"markerSize"
#endif

	double 	fuMinX, fuMaxX, fuMinY, fuMaxY;
	const char	*labelLeft, *labelBottom;
	int	fuMarkerSize;

	xygraphForm->getXYGraphValue( XtNxticks, (XtArgVal) &xticks );
       	xygraphForm->getXYGraphValue( XtNyticks, (XtArgVal) &yticks );
       	xygraphForm->getXYGraphValue( XtNmarkerSize, (XtArgVal) &fuMarkerSize );

        xygraphForm->getXYGraphValue( XtNconnect, (XtArgVal) &connect );

        if ( connect == TRUE_ ) {
           connectOption = 1;
        } else {
           connectOption = 0;
	}

	if ( ! isConfigured ) {
		int wMinX, wMaxX, wMinY, wMaxY;
			
		xygraphForm->getXYGraphValue( XtNminx, (XtArgVal) &wMinX );
		xygraphForm->getXYGraphValue( XtNmaxx, (XtArgVal) &wMaxX );
		xygraphForm->getXYGraphValue( XtNminy, (XtArgVal) &wMinY );
		xygraphForm->getXYGraphValue( XtNmaxy, (XtArgVal) &wMaxY );

		/* defaults for min and max initially match widget values */
		fuMinX = wMinX;		
		fuMaxX = wMaxX;
		fuMinY = wMinY;
		fuMaxY = wMaxY;

		/* the offset & range values never change as the widget min */
		/* and max never change.  These are used for scaling data.  */
		xOffset = wMinX;	
		yOffset = wMinY;
		xRange  = wMaxX - wMinX;
		yRange  = wMaxY - wMinY;

		isConfigured = TRUE_;
	} else {
		fuMarkerSize = markerSize;
		fuMinX = minX;
		fuMaxX = maxX;
		fuMinY = minY;
		fuMaxY = maxY;
	}


	// --- CONFIGURATION PARAMETERS --------------------------------------
	FUParams params;

	params.addRadioButtonsParam( "Display Option",
			             BaseFUParamEntry::Integer, connectOption,
			             ConnectOptionList );
        params.addTextParam( "Data marker size",
				     BaseFUParamEntry::Integer, fuMarkerSize );

 	params.addTextParam( "X Ticks", BaseFUParamEntry::Integer, xticks );
        params.addTextParam( "Y Ticks", BaseFUParamEntry::Integer, yticks );
        params.addTextParam( "Minimum X", BaseFUParamEntry::Real, fuMinX );
        params.addTextParam( "Maximum X", BaseFUParamEntry::Real, fuMaxX );
        params.addTextParam( "Minimum Y", BaseFUParamEntry::Real, fuMinY );
        params.addTextParam( "Maximum Y", BaseFUParamEntry::Real, fuMaxY );

	// Labelling
        xygraphForm->getXYGraphLabel(L_LEFT_VERT, &labelLeft );
        xygraphForm->getXYGraphLabel(L_BOTTOM_HORIZ, &labelBottom );

        params.addTextParam( "Vertical axis label",
			     BaseFUParamEntry::Str, labelLeft );
        params.addTextParam( "Horizontal axis label",
			     BaseFUParamEntry::Str, labelBottom );

        BaseFUParamEntry& connectEntry =
                params.getEntry( "Display Option" );

        if ( connectOption != -1 ) {
           connectEntry.setValue ( connectOption );
        }

        BaseFUParamEntry& markerEntry = params.getEntry( "Data marker size");

	CString configPanelName = "XYGraph FU Config: " + getName();
        ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

 	BaseFUParamEntry& xTicEntry = params.getEntry( "X Ticks");
        BaseFUParamEntry& yTicEntry = params.getEntry( "Y Ticks");
 	BaseFUParamEntry& xMinEntry = params.getEntry( "Minimum X");
 	BaseFUParamEntry& xMaxEntry = params.getEntry( "Maximum X");
 	BaseFUParamEntry& yMinEntry = params.getEntry( "Minimum Y");
 	BaseFUParamEntry& yMaxEntry = params.getEntry( "Maximum Y");

        BaseFUParamEntry &vertLabelEntry =
		              params.getEntry("Vertical axis label");
        BaseFUParamEntry &horizLabelEntry = 
		              params.getEntry("Horizontal axis label");

	// --- CONFIGURATION PARAMETERS  --------------------------------------

	// === PARAMETER ERROR-CHECKING =======================================
        if (xTicEntry.valueIsValid()) {
		xticks = xTicEntry.getValue().getInteger();
	} else {
		warning( "X tick count was not valid.\n") ;
	}
	if (yTicEntry.valueIsValid()) {
		yticks = yTicEntry.getValue().getInteger();
	} else {
		warning( "Y tick count was not valid.\n") ;
	}
	if (xMinEntry.valueIsValid()) {
		fuMinX = xMinEntry.getValue().getReal();
	} else {
		warning( "X Minimum Value was not valid.\n") ;
	}
	if (xMaxEntry.valueIsValid()) {
		fuMaxX = xMaxEntry.getValue().getReal();
	} else {
		warning( "X Maximum Value was not valid.\n") ;
	}
	if (yMinEntry.valueIsValid()) {
		fuMinY = yMinEntry.getValue().getReal();
	} else {
		warning( "Y Minimum Value was not valid.\n") ;
	}
	if (yMaxEntry.valueIsValid()) {
		fuMaxY = yMaxEntry.getValue().getReal();
	} else {
		warning( "Y Maximum Value was not valid.\n") ;
	}
        if ( connectEntry.valueIsValid() ) {
           connectOption = connectEntry.getValue().getInteger();
        } else {
           warning( "Invalid display option in XYGraph" );
	   connectOption = -1;
        }
	if (markerEntry.valueIsValid()) {
		fuMarkerSize = markerEntry.getValue().getInteger();
	} else {
		warning( "Scatter3D data marker size was not valid.\n") ;
	}
	// === PARAMETER ERROR-CHECKING ======================================

	// --- FU AND WIDGET CONFIGURATION DATA SETUPS -----------------------
	minX = fuMinX;
	maxX = fuMaxX;
	minY = fuMinY;
	maxY = fuMaxY;
	markerSize = fuMarkerSize;

        xygraphForm->setXYGraphLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
        xygraphForm->setXYGraphLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        xygraphForm->setXYGraphValue( XtNxticks, xticks );
	xygraphForm->setXYGraphValue( XtNyticks, yticks );

	if ( connectOption == 0 ) {
	   xygraphForm->setXYGraphValue( XtNconnect, FALSE ); 
	} else {
	   xygraphForm->setXYGraphValue( XtNconnect, TRUE );
	}

	xygraphForm->setXYGraphValue( XtNmarkerSize, markerSize );
        xygraphForm->setPerfWidgetColors();
}

FunctionalUnit *			/* virtual */
XYGraphFU::copy()
{
	XYGraphFU *copy = new XYGraphFU();
	return copy;
}


void 
XYGraphFU::fuCallback( int /* nearestXValue */, int /* nearestYValue */,
		       int index )
{
	if ( inputValue.isUndefined() ) {
	   return;	
	}

	FUParams params;
	Array *a = (Array *) inputValue;
        int curDim = inputValue.getTraits().getDimension();

        if ( curDim == 1 ) {
           params.addDisplayParam( "Closest Point Vector Position",
			     		index );
	   params.addDisplayParam( "Closest Point Vector Value",
				        a->getCellValue( index, 0 ) );

        } else if ( curDim == 2 ) {
           params.addDisplayParam( "Closest Point X Value", 
				    a->getCellValue( index, 0 ) );
           params.addDisplayParam( "Closest Point Y Value", 
				    a->getCellValue( index, 1 ) );
	}

        CString callbackTitle = "Callback: " + getName();
	ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
					     callbackTitle );
}


void			/* virtual */ 
XYGraphFU::init()
{
	/* 
	 * ConnectOptionList is static so only initialize it for first 
	 * instance of Class.
	 */
	if ( ConnectOptionList.isEmpty() ) {
            ConnectOptionList.addElement( "Scatter" );
            ConnectOptionList.addElement( "Line" );
	}

	/*
	 * Clear up any leftovers if this isn't the first time init() is
	 * called for this instance.
	 */
	if ( input != NULL ) {
	    delete input;
	}
	if ( xygraphForm != NULL ) {
	    delete xygraphForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	/*
	 * Real code to initialize the FU
	 */
	connectOption = -1;
	isConfigured = FALSE_;
	inputValue = Value::NOVALUE;

	input = new InputPort( "Input" );
        input->addTraits( INTEGER, 1 );
        input->addTraits( FLOAT, 1 );
        input->addTraits( DOUBLE, 1 );
	input->addTraits( INTEGER, 2 );
	input->addTraits( FLOAT, 2 );
	input->addTraits( DOUBLE, 2 );
	_addInputPort( input );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	xygraphForm = new XYGraphFormWrapper( dialog, this, NullArgs, 
							    "Contents" );
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}

Boolean_ 		/* virtual */
XYGraphFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
	    int theOption;
	    int labelSize;
	    int markSize;
	    char buf[LABEL_MAX_LEN];

  	    // ----- Comment line and numeric parameters -----
	    fscanf( fp, "%*[^\n]\n" );

            if ( fscanf( fp, "%lf %lf %lf %lf %d %d %d %d %d %d %d %d\n", 
			   &minX, &maxX, &minY, &maxY, &xticks, &yticks,
			   &theOption, &markSize, &x,
			   &y, &width, &height ) != 12 ) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
	    	dialog->unmanage();
		connectOption = theOption;
		markerSize = markSize;

                // ----- Comment Line; Horizontal Label Size and String
                labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                    strcpy( buf, "\0" );
                }
                xygraphForm->setXYGraphLabel( L_BOTTOM_HORIZ, buf );

                // ----- Comment Line; Vertical Label Size and String
                labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                    strcpy( buf, "\0" );
                }
                xygraphForm->setXYGraphLabel(L_LEFT_VERT, buf );

                /*
 	 	 * We configure based on parameters we loaded from the file.
	 	 */
                if ( connectOption == 0 ) {
                    xygraphForm->setXYGraphValue( XtNconnect, FALSE );
                } else {
                    xygraphForm->setXYGraphValue( XtNconnect, TRUE );
                }

		xygraphForm->setXYGraphValue( XtNmarkerSize, markerSize );

		/* The widget's understanding of maximum and minimum are
		 * configured at a higher level, so it is necessary to get
		 * these values and calculate an offset and range for scaling
		 * purposes
      		 */
                int wMinX, wMaxX, wMinY, wMaxY;
			
		xygraphForm->getXYGraphValue( XtNminx, (XtArgVal) &wMinX );
		xygraphForm->getXYGraphValue( XtNmaxx, (XtArgVal) &wMaxX );
		xygraphForm->getXYGraphValue( XtNminy, (XtArgVal) &wMinY );
		xygraphForm->getXYGraphValue( XtNmaxy, (XtArgVal) &wMaxY );

	        xOffset = wMinX;	
		yOffset = wMinY;
		xRange  = wMaxX - wMinX;
		yRange  = wMaxY - wMinY;

		// Set values in the widget
	        xygraphForm->setXYGraphValue( XtNxticks, xticks );
	        xygraphForm->setXYGraphValue( XtNyticks, yticks );

                /* Place widget at its configured position with proper height
		   and width */
                xygraphForm->setPerfWidgetColors();

	    	dialog->manage();
                xygraphForm->setPerfWidgetPosition( x, y, width, height );

		isConfigured = TRUE_;
	    } 
	    fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured;
}

Boolean_ 
XYGraphFU::ready()
{
	return TRUE_;
}

void 
XYGraphFU::run(  Boolean_& errorFlag )
{
	Assert( input->valueAvailable() );
	inputValue = input->getValue();

	Array *a = (Array *) inputValue;
	const int *dimSizes = a->getDimSizes();
	
	if ( ( inputValue.getTraits().getDimension() == 2 ) &&
	      ( dimSizes[1] != 2 ) ) {
           warning("run():  XYGraph requires X-Y pairs to run properly.\n");
	   errorFlag = TRUE_;
	   return;
        }

       /* currentxptr and currentyptr are currently pointing to the space used
          for the previous run's x and y data values. This space is freed, and
          then currentxptr and currentyptr are assigned to the current run's
          x and y value locations. We can't just use one set of private vars
          since input size changes dynamically, and we must keep track of 
          the x and y values outside of the run method since the widget may
          need the values to redraw the graph before the next time the
          run method is called (i.e. if graph is obscured during single
          step execution, the values are needed)*/
        
       if (currentxptr != NULL) {
           delete currentxptr;
           delete currentyptr;
	}
	numPairs = dimSizes[0];
       	int *xptr = new int [ numPairs ];
	int *yptr = new int [ numPairs ];
   
        currentxptr = xptr;
        currentyptr = yptr;
 
 	int p;
        double xin, yin;
	double dvalScaled;

	if (inputValue.getTraits().getDimension() == 1){
       	    for ( p = 0; p < numPairs; p++ ) {
    	       xin = (double)p;
	       dvalScaled = warnings.fuDataRangeWarningCheck( this, xin,
							      minX, maxX );
	       xptr[p] = Round( ( NormX(dvalScaled) *  xRange ) ) + xOffset;

	       yin = a->getCellValue(p,0);
	       dvalScaled = warnings.fuDataRangeWarningCheck( this, yin,
							      minY, maxY );
	       yptr[p] = Round( ( NormY(dvalScaled) *  yRange ) ) + yOffset;
      	     }
        } 
        else {
	    for ( p = 0; p < numPairs; p++ ) {
    	       xin = a->getCellValue(p,0);
	       dvalScaled = warnings.fuDataRangeWarningCheck( this, xin,
							      minX, maxX );
	       xptr[p] = Round( ( NormX(dvalScaled) *  xRange ) ) + xOffset;

	       yin = a->getCellValue(p,1);
	       dvalScaled = warnings.fuDataRangeWarningCheck( this, yin,
							      minY, maxY );
	       yptr[p] = Round( ( NormY(dvalScaled) *  yRange ) ) + yOffset;
      	     }
	 }

	xygraphForm->setPerfWidgetValues(currentxptr, currentyptr,
                                         &numPairs, 1 );

}

Boolean_		/* virtual */
XYGraphFU::saveConfigToFile( const CString& fileName ) const
{
	Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

	if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
	    result = FAILURE_;
	} else {
	    int	x, y, width, height; 
	    const char *label;

	    xygraphForm->getPerfWidgetPosition( &x, &y, &width, &height );

	    fprintf( fp, 
	        "# MinX MaxX MinY MaxY XTicks YTicks Connect MarkSize X Y Wdth Hght\n" );
	    fprintf( fp, "%lf %lf %lf %lf %d %d %d %d %d %d %d %d\n", 
			  minX, maxX, minY, maxY, xticks, yticks,
			  connectOption, markerSize, x, y, width, height );

            xygraphForm->getXYGraphLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
            fprintf( fp, "%d %s\n", strlen( label), label );

            xygraphForm->getXYGraphLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

	    fclose( fp );
	    result = SUCCESS_;
        }
        return result;
}

// Global class definition
const char *const XYGraphFU::MY_CLASS = "XYGraphFU";
CStringObjList XYGraphFU::ConnectOptionList;

