/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/**********************************************
 * 
 * UtilDisplayManager.C
 *
 * Manager class for UtilDisplay types.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/UtilDisplayManager.C,v 1.10 1994/02/25 04:33:19 aydt Exp $
 *
 ***********************************************/
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>

#include "Defines.h"
#include "UtilDisplayManager.h"
#include "BarGraphUtilDisplay.h"
#include "ChartUtilDisplay.h"
#include "Selection.h"
#include "StripUtilDisplay.h"
#include "KiviatUtilDisplay.h"



/*
 * #define some constants for array dimensions
 * g++ allows 'const int' to do this, but CC doesn't
 */
#define	BUF_LEN	60



/*
 * initialize static class data
 * only executed once
 */
const int UtilDisplayManager::numTypes = 4;
const int UtilDisplayManager::numAggrTypes = 3;
const char* UtilDisplayManager::typeList[] = {
	"Bar Graph",
	"Chart",
	"Strip",
	"Kiviat",
};


UtilDisplayManager::UtilDisplayManager( Widget par, UtilFU *ufu )
{

	/*
	 * initialize other object data
	 */
	parent = par;
	fuParent = ufu;
}


/*
 * getAggregateTypeList()
 * Returns a list of available types of UtilDisplays suitable for displaying
 * aggregate statistics
 */
const char **
UtilDisplayManager::getAggregateTypeList( int& num )
{
	num = numAggrTypes;
	return typeList;
}



/*
 * getTypeList()
 * Returns a list of available types of UtilDisplays
 */
const char **
UtilDisplayManager::getTypeList( int& num )
{
        num = numTypes;
	return typeList;
}



/*
 * makeNewDisplay()
 *
 * This method creates a new display, given its name and type.
 */
UtilDisplay *
UtilDisplayManager::makeNewDisplay( char *name, DisplayTypeConstant type,
					UtilsDisplayedType utype )
{
        UtilDisplay *temp = NULL;


        switch( type ) {
                case bargraph:
			temp = new BarGraphUtilDisplay( name, fuParent, 
						parent );
                        break;
		case chart:
			temp = new ChartUtilDisplay( name, fuParent, parent );
			break;

		case strip:
			temp = new StripUtilDisplay( name, fuParent, parent );
			break;

		case kiviat:
			temp = new KiviatUtilDisplay( name, fuParent, parent );
			break;

		case unknown:
			break;
        }

	temp->setDisplayedType( utype );
        return temp;
}



/*
 * nameToType()
 *
 * Converts a character string containing a type name to its type constant
 */
UtilDisplayManager::DisplayTypeConstant
UtilDisplayManager::nameToType( char *name )
{
        int i = 0;


        if( name ) {
                /* find index of name into the typeList */
                while( i < numTypes ) {

                        if( !strcmp( typeList[i], name )) {
					return (DisplayTypeConstant)i;
			}
                        i++;
                }
        }

        return unknown;
}



/*
 * typeToName()
 *
 * Converts a type constant to a character string containing its type name
 */
const char* const
UtilDisplayManager::typeToName( DisplayTypeConstant type )
{
        return typeList[ type ];
}




/*******************
 * standard methods
 *******************/

UtilDisplay *
UtilDisplayManager::readDisplayListFromFile( FILE *fp, int numDisplays )
{
        UtilDisplay * list = NULL;              // the list which is created
        UtilDisplay *lend = NULL;                 // end of the display list
        UtilDisplay *temp;
        char    buf1[BUF_LEN];
        char    buf2[BUF_LEN];
        int     buf1size, buf2size;
	UtilsDisplayedType disptype;
        int     i;
	int	ix, iy, iwid, ihei;
	Position	x, y;
	Dimension	width, height;


        for( i = 0; i < numDisplays; i++ ) {

                /*
                 * get name and type to create a new display
                 */
		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d%*c%[^\n]", &buf1size, buf1 );
		fscanf( fp, "\n" );

		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d%*c%[^\n]", &buf2size, buf2 );
		fscanf( fp, "\n" );

		/* read screen chax */
		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d %d %d %d\n", &ix, &iy, &iwid, &ihei );
		fscanf( fp, "\n" );

		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d\n", &disptype );
		fscanf( fp, "\n" );

		/*
		 * make the new display
		 */
                temp = makeNewDisplay( buf1, nameToType( buf2 ), disptype );

                /* get horizontal and vertical labels */
		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d%*c%[^\n]", &buf1size, buf1 );
                if( buf1size ) {
                        temp->setHoriLabel( buf1 );
                }
		fscanf( fp, "\n" );

		fscanf( fp, "%*[^\n]\n" );	// skip comment line
		fscanf( fp, "%d%*c%[^\n]", &buf1size, buf1 );
                if( buf1size ) {
                        temp->setVertLabel( buf1 );
                }
		fscanf( fp, "\n" );

                /* 
		 * get temp's list of Selections
		 */
                temp->setSelection( Selection::readSelectionList( fp ));
		temp->updatePerfWidgets();

		/* update screen chax */
		x = ix;
		y = iy;
		width = iwid;
		height = ihei;
		XtVaSetValues( temp->getDialog(),
			XmNx,		x,
			XmNy,		y,
			XmNwidth,	width,
			XmNheight,	height,
			NULL );


		fscanf( fp, "%*[^\n]\n" );	// skip separator line

                /* 
		 * link temp at end of list 
		 */
                if( list ) {
                        lend->next = temp;
                }
                else {
                        list = temp;
                }
                lend = temp;
        }
        return list;
}


void
UtilDisplayManager::saveDisplayListToFile( ofstream &ofs, 
					UtilDisplay * list ) const
{
	UtilDisplay *curr = list;

	while( curr ) {
		curr->printOn( ofs );
		curr = (UtilDisplay *)curr->next;
	}
}


