/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/************************************************************
 * StripUtilDisplay.C
 *
 * Implementation of a UtilDisplay showing utilization on an 
 * array of Gantt charts.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/StripUtilDisplay.C,v 1.10 1994/02/25 04:33:16 aydt Exp $
 *
 *************************************************************/
#include <stream.h>
#include <strstream.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include "StripUtilDisplay.h"
#include "Selection.h"
#include "UtilDisplayManager.h"

extern "C" {
#include "Gantt.h"
}




/*
 * #defines for array dimensions
 * g++ allowed this to be done with 'const int,' but CC complains...
 */
#define MAX_LABEL_SIZE	16



StripUtilDisplay::StripUtilDisplay( char *name, UtilFU *ufu, Widget par ) 
	: UtilDisplay( name, ufu, par )
{
	int i, j;




	y = NULL;
	for( i = 0; i < numAggregates; i++ ) {
		for( j = 0; j < GANTT_WINDOW_SIZE; j++ ) {
			ay[i][j] = 0;
		}
	}

	/* initialize the x array to show values at regular intervals */
	for( j = 0; j < GANTT_WINDOW_SIZE; j++ ) {
		x[j] = j + 1;
	}

	XtManageChild( getDialog() );
}


StripUtilDisplay::~StripUtilDisplay( void )
{
	XtDestroyWidget( getDialog() );
}



/*
 * _handleResize()
 *
 * Resize the performance widgets so they fill their dialog, unless
 * resizing them would make them too small.
 * The parameter w is the ScrolledWindow widget.
 */
/* virtual */
void
StripUtilDisplay::_handleResize( Widget w )
{
	Widget cwind, wwind;	    /* clip and work windows for w */
	Dimension cwidth, cheight;      /* dimensions of clip window */
	Dimension natWidth, natHeight;  /* natural size of nonscrolled dialog */
	Dimension labelWidth, labelHeight,/* RowColumn Dimension attributes */
		marginWidth, marginHeight, spacing;
	int numSetups;
	int i;


	/*
	 * Check that there is a minimum size
	 */
	getNaturalDimensions( natWidth, natHeight );
	if( natWidth == 0 || natHeight == 0 ) {
		return;			 // no, so do nothing
	}


	XtUnmanageChild( getWorkWindow() );


	/*
	 * get new size of clip window
	 */
	XtVaGetValues( w,
			XmNworkWindow,  &wwind,
			XmNclipWindow,  &cwind,
			NULL );
	XtVaGetValues( cwind,
			XtNwidth,       &cwidth,
			XtNheight,      &cheight,
			NULL );


	/*
	 * check whether the clip window is larger or smaller than the
	 * natural size of the workWindow
	 */
	Dimension width = ( cwidth > natWidth ? cwidth : natWidth );
	Dimension height = ( cheight > natHeight ? cheight : natHeight );

	XtVaSetValues( getWorkWindow(),
			XtNwidth,       width,
			XtNheight,      height,
			NULL );


	/*
	 * find out the amount of width actually available to the performance
	 * widgets ( i.e., not taken up by labels, margins, etc. )
	 */
	UtilsDisplayedType disp = getDisplayedType();
	if( disp == IndividualsShown ) {
		if( !(numSetups = getNumIndividuals()) ) {
			return;
		}
	}
	else {
		numSetups = numAggregates;
	}

	/*
	 * compute new height of performance widgets
	 */
	XtVaGetValues( getHoriLabelWidget(),
			XtNheight,       &labelHeight,
			NULL );
	height -= labelHeight;
	XtVaGetValues( getRowCol(),
			XmNmarginWidth, &marginWidth,
			XmNmarginHeight, &marginHeight,
			XmNspacing,     &spacing,
			NULL );
	height -= ( 2 * marginHeight + numSetups * spacing );
	height -= 20;		    // to ensure the ScrollBars go away

	/*
	 * compute new width of underlying RowColumn widget
	 * ( This handles setting height of performance widgets )
	 */
	XtVaGetValues( getVertLabelWidget(),
			XtNwidth,      &labelWidth,
			NULL );
	width -= ( 2*marginWidth + labelWidth + getPerfLabelDim() );
	width -= 20;		   // to ensure the ScrollBars go away


	/*
	 * Compute proposed height of performance widgets.
	 * Note we checked for numSetups == 0 above.
	 */
	Dimension proposedHeight = height / numSetups;

	if( disp == IndividualsShown ) {
		for( i = 0; i < numSetups; i++ ) {
			XtVaSetValues( individualUtils[i],
				XtNheight,       proposedHeight,
				NULL );
		}
	}
	else {
		for( i = 0; i < numSetups; i++ ) {
			XtVaSetValues( aggregateUtils[i],
				XtNheight,       proposedHeight,
				NULL );
		}
	}
	XtManageChild( getWorkWindow() );
}



/*
 * _handleSelection()
 *
 * Actual callback to handle reporting the XtNselect callback correctly.
 * This is necessary because the call data reported by the Gantt widget
 * is different than that of some other performance widgets
 */
void
StripUtilDisplay::_handleSelection( Widget w )
{
	UtilsDisplayedType disp = getDisplayedType();
	int procnum;			// processor represented by w
	int i;


	/* determine which processor Widget w represents */
	switch( disp ) {
		case AggregateShown:
			for( i = 0; i < numAggregates; i++ ) {
				if( w == aggregateUtils[i] ) {
					procnum = i;
					break;
				}
			}

			UtilDisplay::selectCB( w, this,
					(XtPointer)ay[i][GANTT_WINDOW_SIZE-1] );
			break;

		case IndividualsShown:
			for( i = 0; i < getNumIndividuals(); i++ ) {
				if( w == individualUtils[i] ) {
					procnum = map[i];
					break;
				}
			}
			UtilDisplay::selectCB( w, this, 
					(XtPointer)y[i][GANTT_WINDOW_SIZE-1] );

			break;
	}
}




/*
 * changeValue()
 *
 * update the utilization value for PE proc to utilization util
 */
/* virtual */
void
StripUtilDisplay::changeValue( int proc, float util )
{
	int i;


	switch( getDisplayedType() ) {
		case AggregateShown: {
			GanttUnMapLine( (GanttWidget)aggregateUtils[proc], 0 );

			/* compute new line */
			for( i = 0; i < GANTT_WINDOW_SIZE-1; i++ ) {
				ay[proc][i] = ay[proc][i + 1];
			}
			ay[proc][GANTT_WINDOW_SIZE-1] = (int)(util*100);

			GanttAddLine( (GanttWidget)aggregateUtils[proc], 0, 
				GANTT_WINDOW_SIZE, 0, x, ay[proc] );

			GanttShowGantt( (GanttWidget)aggregateUtils[proc] );
		}
		break;

		case IndividualsShown: {
			
			int indx = _indexIntoMap( proc );

			if( indx != notDisplayed ) {
				GanttUnMapLine( 
					(GanttWidget)individualUtils[indx], 0 );

				/* compute new line */
				for( i = 0; i < GANTT_WINDOW_SIZE-1; i++ ) {
					y[indx][i] = y[indx][i + 1];
				}
				y[indx][GANTT_WINDOW_SIZE-1] = (int)(util*100);

				GanttAddLine( 
					(GanttWidget)individualUtils[indx], 0, 
					GANTT_WINDOW_SIZE, 0, 
					x, y[indx] );
				GanttShowGantt( 
					(GanttWidget)individualUtils[indx] );
			}

		}
		break;

	}
}





/*
 * createPerfSetup()
 *
 * Creates a widget subtree for displaying utilization on a strip
 * performance widget
 */
Widget
StripUtilDisplay::createPerfSetup( const char *label ) 
{
	Pixel	pixel;
	Dimension width;


	Widget form = XtVaCreateWidget( "form",
			xmFormWidgetClass, getRowCol(),
			NULL );

	/* create the widget's label */
	XmString xstr = XmStringCreateLtoR( (char*)label,
				XmSTRING_DEFAULT_CHARSET );

	Widget labl = XtVaCreateManagedWidget( "striplabel",
			xmLabelWidgetClass, form,
			XmNlabelString,		xstr,
			NULL );
	XtVaGetValues( labl,
			XtNwidth,	&width,
			NULL );
	setPerfLabelDim( width );
	

	Widget retval = XtVaCreateManagedWidget( "strip",
			ganttWidgetClass, form,
			XtNxmax, GANTT_WINDOW_SIZE,
			NULL );
	XtAddCallback( retval, XtNselect, 
			(XtCallbackProc)StripUtilDisplay::selectCB, this );

	/* set line color for Gantt chart */
	/* take the color from the foreground color of form */
	XtVaGetValues( form,
		XmNforeground,	&pixel,
		NULL );
	GanttSetColorMap( (GanttWidget)retval, 1, &pixel );
		

	XtManageChild( form );
	return retval;
}



/*
 * getType()
 *
 * Returns the character string indicating the type of this UtilDisplay
 */
/* virtual */
const char* const
StripUtilDisplay::getType( void ) const
{
	return "Strip";
}




/*
 * selectCB()
 *
 * callback to handle XtNselect callback supported by the Strip performance
 * widget.
 */
/* static */
void
StripUtilDisplay::selectCB( Widget w, StripUtilDisplay *disp, 
						XtPointer /* cd */ )
{
	disp->_handleSelection( w );
}



/*
 * updatePerfWidgets()
 *
 * This method is called whenever the performance widget selection is
 * changed.
 * All deallocation of prior widgets is assumed to be complete before the
 * start of this method.
 */
/* virtual */
void
StripUtilDisplay::updatePerfWidgets( void )
{
	int	perfWCount = 0;			// # of strip widgets displayed
	Dimension height = 0;			// Dimensions of strip widgets
	Dimension natWidth, natHeight;		// natural Dimensions of dialog
	Position xpos, ypos;                    // screen loc of this display

	/* save screen location of this display */
	/* no use saving dimensions, they probably should be changed */
	XtVaGetValues( getDialog(),
		XmNx,   &xpos,
		XmNy,   &ypos,
		NULL );

	/* pop down the display */
	XtUnmanageChild( getDialog() );


	/* lay out displays horizontally */
	XtVaSetValues( getRowCol(),
		XmNorientation,	XmVERTICAL,
		NULL );

	/*
	 * Create performance widget setups
	 */
	if( getDisplayedType() == AggregateShown ) {
		perfWCount += numAggregates;

		/* create performance widget subtree for average utilization */
		aggregateUtils[Average] = createPerfSetup( averageLabelString );
		aggregateUtils[Minima] = createPerfSetup( minimumLabelString );
		aggregateUtils[Maxima] = createPerfSetup( maximumLabelString );

		XtVaGetValues( aggregateUtils[Average],
			XmNheight,       &height,
			NULL );
	}

	else {
		/* create performance widgets for individual PEs */

		/* deterine how many individuals to show utilizations */
		int icount = 0;
		Selection* sel = getSelection();
		_makeMap( sel, icount );
		setNumIndividuals( icount );
		perfWCount += icount;

		if( icount ) {
			int i;

			delete[] y;
			y = new int[icount][GANTT_WINDOW_SIZE];

			/* allocate enough space for individual strip widgets */
			individualUtils = (Widget*)XtMalloc( icount *
							sizeof( GanttWidget ) );

			/* create widget subtrees for each individual */
			for( i = 0; i < icount; i++ ) {
				char buf[ MAX_LABEL_SIZE ];	// for labels


				/* create label string for this PE */
				ostrstream temp( buf, MAX_LABEL_SIZE );
				temp << map[i] << '\0';

				individualUtils[i] = createPerfSetup( buf );

				/* init line to all 0s */
				int j;
				for( j = 0; j < GANTT_WINDOW_SIZE; j++ ) {
					y[i][j] = 0;
				}

				GanttAddLine( (GanttWidget)individualUtils[i], 
						0, GANTT_WINDOW_SIZE, 
						0, x, y[i] );
				GanttShowGantt((GanttWidget)individualUtils[i]);

			}

			XtVaGetValues( individualUtils[0],
				XmNheight, &height,
				NULL );

		}
	}

	/* find natural Dimensions for the new work window */
	XtVaGetValues( getWorkWindow(),
		XtNwidth,	&natWidth,
		XtNheight,	&natHeight,
		NULL );
	setNaturalDimensions( natWidth, natHeight );

	/* resize the dialog for the number of setups */
	initialResizeDialog();

	/* pop up the dialog */
	XtManageChild( getDialog() );

	/* reset the screen location */
	XtVaSetValues( getDialog(),
		XmNx,  	xpos,
		XmNy,   ypos,
		NULL );

}


