/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SimpleStatistics.cc - A functional unit for synthesizing simple
 *			statistics (.i.e., mean, variance, etc) from
 *			a sequence of scalar data values
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/SimpleStatistics.C,v 1.8 1994/03/15 16:42:04 aydt Exp $
 */

#include <math.h>

#include "SimpleStatistics.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

SimpleStatistics::SimpleStatistics()
{
	_setClassName( MY_CLASS );

	dataValue = NULL;
	outMean = NULL;
	outVariance = NULL;
	outSD = NULL;
	outCILow = NULL;
	outCIHigh = NULL;
}

SimpleStatistics::~SimpleStatistics()
{
	delete dataValue;
	delete outMean;
   	delete outVariance;
	delete outSD;
	delete outCILow;
	delete outCIHigh;
}

void				/* virtual */
SimpleStatistics::configure()
{
	FUParams params;
 	params.addRadioButtonsParam( "Confidence Interval Width",
				     BaseFUParamEntry::Integer, 1,
				     CIPercentList );
	ParamConfig pc( Pablo::TopLevel(), params, getName() );
 	BaseFUParamEntry& ciEntry = params.getEntry( 
				     "Confidence Interval Width" );

	if ( ciOption != -1 ) {
	    ciEntry.setValue( ciOption );
	}
	pc.run();

	if ( ciEntry.valueIsValid() ) {
	    ciOption = ciEntry.getValue().getInteger();
	} else {
	    warning( "Invalid confidence interval width; Using default value" );
	    ciOption = 1;
	}
	isReady = TRUE_;
}

FunctionalUnit *		/* virtual */
SimpleStatistics::copy()
{
	SimpleStatistics *copy = new SimpleStatistics();
	return copy;
}

void 				/* virtual */
SimpleStatistics::init()
{
	/* 
	 * CIPercentList is a static that is only initialized once
	 */
	if ( CIPercentList.isEmpty() ) {
	    CIPercentList.addElement( "0.80" );
	    CIPercentList.addElement( "0.90" );
	    CIPercentList.addElement( "0.95" );
	    CIPercentList.addElement( "0.98" );
	    CIPercentList.addElement( "0.99" );
	}

	/*
	 * Start by clearing up any leftovers that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */

	if ( dataValue != NULL ) {
	    delete dataValue;
	}
	if ( outMean != NULL ) {
	    delete outMean;
	}
        if ( outVariance != NULL ) {
            delete outVariance;
        }
	if ( outSD != NULL ) {
	    delete outSD;
	}
	if ( outCILow != NULL ) {
	    delete outCILow;
	}
        if ( outCIHigh != NULL ) {
            delete outCIHigh;
        }

	dataValue = new InputPort( "Data Value" );
	dataValue->addTraits( INTEGER, 0 );
	dataValue->addTraits( FLOAT, 0 );
	dataValue->addTraits( DOUBLE, 0 );
	_addInputPort( dataValue );

	outMean = new OutputPort( "Mean" );
	_addOutputPort( outMean );

	outSD = new OutputPort( "Standard Deviation" );
	_addOutputPort( outSD );

 	outVariance = new OutputPort( "Variance" );
	_addOutputPort( outVariance );

 	outCILow = new OutputPort( "CI (Low)" );
	_addOutputPort( outCILow );

 	outCIHigh = new OutputPort( "CI (High)" );
	_addOutputPort( outCIHigh );

	dataSum = 0.0;
	dataSquares = 0.0;
	dataCount = 0;

	ciOption = -1;
	isReady = FALSE_;
}

Boolean_			/* virtual */
SimpleStatistics::inputTraitsValid()
{
	/* 
	 * SimpleStatistics will always output a group of scalars
	 * each with Traits=( DOUBLE, 0 ).  Our input port has a type
	 * that is restrictive enough to insure any binding is valid 
	 * without further verification.
	 */
	DataTraits elementTraits( DOUBLE, 0 );

	outMean->setTraits( elementTraits );
	outSD->setTraits( elementTraits );
	outVariance->setTraits( elementTraits );
	outCILow->setTraits( elementTraits );
	outCIHigh->setTraits( elementTraits );

	return TRUE_;
}
	
Boolean_			/* virtual */
SimpleStatistics::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );
        if ( fp == NULL ) {
            warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
            fscanf(fp, "%*[^\n]\n" );		// Comment line

            if ( fscanf( fp, "%d\n", &ciOption ) == 1 ) {
	        isReady = TRUE_;
            } else {
                warning( "\nUnable to read configuration information from %s\n",
                          fileName.getValue() );
		ciOption = -1;
            }
            fclose( fp );
        }

	if ( isReady == FALSE_ ) {	    // last chance to configure!
	    configure();		    // sets isReady if successful
	}

	return ( isReady );
}

Boolean_ 			/* virtual */
SimpleStatistics::ready()
{
	return ( isReady );
}

void 				/* virtual */
SimpleStatistics::run(  Boolean_& /* errorFlag */ )
{
	Assert( dataValue->valueAvailable() );
	
	double newValue = (double) dataValue->getValue();

	dataSum += newValue;
	dataSquares += newValue * newValue;
	dataCount++;

	double newMean = dataSum / dataCount;
	
	double newVar;
	if ( dataCount > 1 ) {
	   newVar = (dataCount * dataSquares - (dataSum * dataSum)) /
				(dataCount * (dataCount - 1));
	} else {
	   newVar = 0.0;
	}

	double newSD = sqrt(newVar); 

	tmpValue = newMean;
	outMean->setValue( tmpValue );

	tmpValue = newVar;
	outVariance->setValue( tmpValue );

	tmpValue = newSD;
	outSD->setValue( tmpValue );

	int	idx;
	if ( dataCount > 30 ) {
	   idx = 30;
	} else {
	   idx = dataCount - 1;
	}

	double newLow = newMean - studentT[idx][ciOption] / sqrt(dataCount); 
	double newHigh = newMean + studentT[idx][ciOption] / sqrt(dataCount);

	tmpValue = newLow;
	outCILow->setValue( tmpValue );

	tmpValue = newHigh;
	outCIHigh->setValue( tmpValue );

	_writeOutput();
}

Boolean_			/* virtual */
SimpleStatistics::saveConfigToFile( const CString& fileName ) const
{
	Boolean_ retCode = FAILURE_;

        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                              errorString() );
        } else {
            fprintf( fp, "# CI Option\n" );
            fprintf( fp, "%d\n", ciOption );

            fclose( fp );
            retCode = SUCCESS_;
        }

        return retCode;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const SimpleStatistics::MY_CLASS = "SimpleStatistics";
CStringObjList SimpleStatistics::CIPercentList;
const double SimpleStatistics::studentT[30][5] = {
		3.078, 6.314, 12.706, 31.821, 63.657,
		1.886, 2.920,  4.303,  6.965,  9.925,
		1.638, 2.353,  3.182,  4.541,  5.841,
		1.533, 2.132,  2.776,  3.747,  4.604,
		1.476, 2.015,  2.571,  3.365,  4.032,

		1.440, 1.943,  2.447,  3.143,  3.707, 
		1.415, 1.895,  2.364,  2.998,  3.499,
		1.397, 1.860,  2.306,  2.896,  3.355,
		1.383, 1.833,  2.262,  2.821,  3.250,
		1.372, 1.812,  2.228,  2.764,  3.169,

		1.363, 1.796,  2.201,  2.718,  3.106,
		1.356, 1.782,  2.179,  2.681,  3.055,
		1.350, 1.771,  2.160,  2.650,  3.012,
		1.345, 1.761,  2.145,  2.624,  2.997,
		1.341, 1.753,  2.131,  2.602,  2.947,

		1.337, 1.746,  2.120,  2.583,  2.921,
		1.333, 1.740,  2.110,  2.567,  2.898,
		1.330, 1.734,  2.101,  2.552,  2.878,
		1.328, 1.729,  2.093,  2.539,  2.861,
		1.325, 1.725,  2.086,  2.528,  2.845,

		1.323, 1.721,  2.080,  2.518,  2.831,
		1.321, 1.717,  2.074,  2.508,  2.819,
		1.319, 1.714,  2.069,  2.500,  2.807,
		1.318, 1.711,  2.064,  2.492,  2.797,
		1.316, 1.708,  2.060,  2.485,  2.787,

		1.315, 1.706,  2.056,  2.479,  2.779,
		1.314, 1.703,  2.052,  2.473,  2.771,
		1.313, 1.701,  2.048,  2.467,  2.763,
		1.311, 1.669,  2.045,  2.462,  2.756,
		1.282, 1.645,  1.960,  2.326,  2.576
};

