/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ScaleFU.cc - A polymorophic functional unit for scaling data
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/ScaleFU.C,v 1.15 1994/03/15 16:41:59 aydt Exp $
 */
#include <values.h>

#include "ScaleFU.h"
#include "Addition.h"
#include "Subtraction.h"
#include "Multiplication.h"
#include "Division.h"
#include "Maximum.h"
#include "Minimum.h"
#include "ReductionMinimum.h"
#include "ReductionMaximum.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

ScaleFU::ScaleFU()
{
	_setClassName( MY_CLASS );

	operand = NULL;
	result = NULL;

        addOperator = NULL;
        multOperator = NULL;
        subOperator = NULL;
        divideOperator = NULL;
        minOperator = NULL;
        maxOperator = NULL;
        maxReductionOperator = NULL;
        minReductionOperator = NULL;
}

ScaleFU::~ScaleFU()
{
	delete operand;
	delete result;
	delete addOperator;
	delete subOperator;
	delete multOperator;
	delete divideOperator;
	delete maxOperator;
	delete minOperator;
	delete maxReductionOperator;
	delete minReductionOperator;
}

void                            /* virtual */
ScaleFU::configureOperation()
{
	float	scale;
	float	offset;
	float	min, max;

  	FUParams params;

	if ( isConfigured ) {
	    offset = (float) offsetValue;
	    scale = (float) scaleValue;
	    min = (float) rangeMin;
	    max = (float) rangeMax;
	} else {
	    offset = 0.0;
	    scale = 1.0;
	    min = 0.0;
	    max = 100.0;
	    isConfigured = TRUE_;
	}

	FUParams buttonParams;
  	buttonParams.addRadioButtonsParam( "Scaling Option", 
				      BaseFUParamEntry::Integer, 0, ScaleList );
  	ParamConfig pc( Pablo::TopLevel(), buttonParams, getName() );
  	BaseFUParamEntry& opEntry = buttonParams.getEntry( "Scaling Option" );

	if ( operationType != -1 ) {
	    opEntry.setValue( operationType );
	}
        pc.run();

  	if ( opEntry.valueIsValid() ) {
     	    operationType = opEntry.getValue().getInteger();
  	} else {
     	    warning( "Scaling option was not valid - defaults to Static." );
	    operationType = STATIC;
  	}

	if ( operationType == STATIC ) {

	    params.addTextParam( "Offset", BaseFUParamEntry::Real, offset );
	    params.addTextParam( "Scale Factor", BaseFUParamEntry::Real, scale);

	    ParamConfig pc( Pablo::TopLevel(), params, getName() );
	    pc.run();

	    BaseFUParamEntry& offsetEntry = params.getEntry( "Offset" );
	    BaseFUParamEntry& scaleEntry = params.getEntry( "Scale Factor" );

	    if ( offsetEntry.valueIsValid() ) {
	        offset = offsetEntry.getValue().getReal();
	    } else {
	        warning( "Invalid scale offset\n");
	    }

  	    if ( scaleEntry.valueIsValid() ) {
                scale = scaleEntry.getValue().getReal();
            } else {
                warning( "Invalid scale factor\n");
            }

	    scaleValue = (double) scale;
	    offsetValue = (double) offset;

            if ( addOperator == NULL ) {
	        addOperator = new Addition;
	    }
	    if ( multOperator == NULL ) {
	        multOperator = new Multiplication;
	    }
	    isReady = TRUE_;

	 } else if ( operationType == DYNAMIC ) {

	    params.addTextParam( "Minimum", BaseFUParamEntry::Real, min );
	    params.addTextParam( "Maximum", BaseFUParamEntry::Real, max );

	    ParamConfig pc( Pablo::TopLevel(), params, getName() );
	    pc.run();

	    BaseFUParamEntry& minimumEntry = params.getEntry( "Minimum" );
	    BaseFUParamEntry& maximumEntry = params.getEntry( "Maximum" );

	    if ( minimumEntry.valueIsValid() ) {
	        min = minimumEntry.getValue().getReal();
	    } else {
	        warning( "Invalid scale maximum\n");
	    }

  	    if ( maximumEntry.valueIsValid() ) {
                max = maximumEntry.getValue().getReal();
            } else {
                warning( "Invalid scale minimum\n");
            }

	    rangeMin = (double)min;
	    rangeMax = (double)max;
	    staticRangeValue = rangeMax - rangeMin;

	    minValue = MAXDOUBLE;
	    maxValue = -MAXDOUBLE;

            if ( addOperator == NULL ) {
	        addOperator = new Addition;
	    }
            if ( subOperator == NULL ) {
	        subOperator = new Subtraction;
	    }
            if ( multOperator == NULL ) {
	        multOperator = new Multiplication;
	    }
            if ( divideOperator == NULL ) {
	        divideOperator = new Division;
	    }
            if ( maxOperator == NULL ) {
	        maxOperator = new Maximum;
	    }
            if ( minOperator == NULL ) {
	        minOperator = new Minimum;
	    }
            if ( maxReductionOperator == NULL ) {
	        maxReductionOperator = new ReductionMaximum;
	    }
            if ( minReductionOperator == NULL ) {
	        minReductionOperator = new ReductionMinimum;
	    }
	    isReady = TRUE_;

	} else {

	    isReady = FALSE_;

	}
}


FunctionalUnit *		/* virtual */
ScaleFU::copy()
{
	ScaleFU *copy = new ScaleFU();
	return copy;
}

void 				/* virtual */
ScaleFU::init()
{
	/*
	 * ScaleList is a static that is only initialized once
	 */
	if ( ScaleList.isEmpty() ) {
	    ScaleList.addElement("Static");
	    ScaleList.addElement("Dynamic");
	}

	/*
	 * Continue by clearing up any 'leftovers' that will be around
	 * if init() isn't being called for the first time.  The goal is
	 * to start with a 'virgin' FU.
	 */
	if ( operand != NULL ) {
	    delete operand;
	}
	if ( result != NULL ) {
	    delete result;
	}
        if ( addOperator != NULL ) {
	    delete addOperator;
	    addOperator = NULL;
	}
        if ( multOperator != NULL ) {
	    delete multOperator;
	    multOperator = NULL;
	}
        if ( subOperator != NULL ) {
	    delete subOperator;
	    subOperator = NULL;
	}
        if ( divideOperator != NULL ) {
	    delete divideOperator;
	    divideOperator = NULL;
	}
        if ( minOperator != NULL ) {
	    delete minOperator;
	    minOperator = NULL;
	}
        if ( maxOperator != NULL ) {
	    delete maxOperator;
	    maxOperator = NULL;
	}
        if ( maxReductionOperator != NULL ) {
	    delete maxReductionOperator;
	    maxReductionOperator = NULL;
	}
        if ( minReductionOperator != NULL ) {
	    delete minReductionOperator;
	    minReductionOperator = NULL;
	}

	operationType = -1;
	isConfigured = FALSE_;
	isReady = FALSE_;

	operand = new InputPort( "Operand" );
	operand->addTraits( INTEGER, -1 );
	operand->addTraits( FLOAT, -1 );
	operand->addTraits( DOUBLE, -1 );
	_addInputPort( operand );

	result = new OutputPort( "Result" );
	_addOutputPort(result);
}

Boolean_ 			/* virtual */
ScaleFU::inputTraitsValid()
{
	DataTraits dataT = operand->getTraits();

	/*
	* Because many scaling operations produce double precision
	* values, we simply force all data types to be double precision
	*/

	DataTraits resultT( DOUBLE, dataT.getDimension() );
	result->setTraits( resultT );

	return SUCCESS_;
}


Boolean_                        /* virtual */
ScaleFU::loadConfigFromFile( const CString& fileName )
{
	float	val1, val2;
	char	opName[256];

  	FILE *fp = fopen( fileName.getValue(), "r" );
  	if ( fp == NULL ) {
    	    warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
  	} else {
	    fscanf( fp, "%*[^\n]\n" );  // Comment line

    	    if ( fscanf( fp, "%s %f %f", opName, &val1, &val2) != 3 ) {
                warning( "unable to read scale configuration file" );
		operationType = -1;
            } else {
                if ( ( operationType = ScaleList.lookup( opName ) ) < 0 ) {
                    warning( "\n%s is not a valid operation name", opName );
                }
	    }

	   if ( operationType == STATIC ) {
	        scaleValue = (double)val1;
	        offsetValue = (double)val2;

	        addOperator = new Addition;
	        multOperator = new Multiplication;

                isReady = TRUE_;
           } else if ( operationType == DYNAMIC ) {
	        rangeMin = (double)val1;
	        rangeMax = (double)val2;
	        staticRangeValue = rangeMax - rangeMin;
	        minValue = MAXDOUBLE;
	        maxValue = -MAXDOUBLE;
 
 	        addOperator = new Addition;
 	        subOperator = new Subtraction;
 	        multOperator = new Multiplication;
 	        divideOperator = new Division;
	        maxOperator = new Maximum;
	        minOperator = new Minimum;
	        maxReductionOperator = new ReductionMaximum;
	        minReductionOperator = new ReductionMinimum;

	        isReady = TRUE_;
	    }

    	    fclose( fp );
  	}

  	if ( isReady == FALSE_ ) { // one last chance to configure
	    configureOperation();
            configure();
	} else {
	    isConfigured = TRUE_;
	}

  	return( isReady );
}

    
Boolean_ 			/* virtual */
ScaleFU::ready()
{
	return isReady;
}

void 				/* virtual */
ScaleFU::run( Boolean_& errorFlag )
{
	Assert( operand->valueAvailable() );

	if ( operationType == STATIC ) {
	    multOperator->getBinaryResult( operand->getValue(), scaleValue, 
								tmpV );
	    addOperator->getBinaryResult( tmpV, offsetValue, resultV );
	} else {
    	    if ( operand->getValue().getTraits().isScalar() ) {
	        maxOperator->getBinaryResult( operand->getValue(), maxValue, 
								   tmpV );
	        maxValue = tmpV;
 	        minOperator->getBinaryResult( operand->getValue(), minValue, 
								   tmpV );
	        minValue = tmpV;
	    } else {
	        maxReductionOperator->getReductionResult( operand->getValue(),
						          newV );
	        maxOperator->getBinaryResult( newV, maxValue, tmpV );
	        maxValue = tmpV;

	        minReductionOperator->getReductionResult( operand->getValue(),
                                                          newV );
                minOperator->getBinaryResult( newV, minValue, tmpV );
	        minValue = tmpV;
	    }

	    dynamicRangeValue = maxValue - minValue;

	    if ( ( (double) dynamicRangeValue ) == 0.0 ) {
	        dynamicRangeValue = 1.0;
	    } 

	    subOperator->getBinaryResult( operand->getValue(), minValue, tmpV );
	    multOperator->getBinaryResult( tmpV, staticRangeValue, newV );
	    divideOperator->getBinaryResult( newV, dynamicRangeValue, tmpV );
	    addOperator->getBinaryResult( tmpV, rangeMin, resultV );
	}

	if ( resultV.isDefined() ) {
	    result->setValue( resultV );
	    _writeOutput();
	} else {
	    errorFlag = TRUE_;
	}
}

Boolean_ 
ScaleFU::saveConfigToFile( const CString& fileName ) const
{
  	Boolean_ retCode = FAILURE_;

  	FILE *fp = fopen( fileName.getValue(), "w" );

  	if ( fp == NULL ) {
     	    error( "\nUnable to open %s: %s\n", fileName.getValue(),
	  		errorString() );
  	} else {
	    if ( operationType == STATIC ) {
	        fprintf( fp, "# OperationType ScaleValue OffsetValue \n");
    	        fprintf( fp, "%s %f %f\n",
			(const char *)ScaleList.getElement( operationType ),
			(float)scaleValue, (float)offsetValue );
	    } else {
                fprintf( fp, "# OperationType RangeMin RangeMax\n");
		fprintf( fp, "%s %f %f\n",
                        (const char *) ScaleList.getElement( operationType ),
                        (float)rangeMin, (float)rangeMax );
	    }

    	    retCode = SUCCESS_;
    	    fclose( fp );
  	}

  	return retCode;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const ScaleFU::MY_CLASS = "ScaleFU";
CStringObjList ScaleFU::ScaleList;
