/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
#ifndef __GNUG__
#include <malloc.h>
#else
#include <stdlib.h>
#endif

#include <math.h>
#include <Xm/List.h>
#include <Xm/Text.h>

#include "ParallelCoordsFU.h"
#include "ParallelCoordsFormWrapper.h"

#include "PabloMainInterface.h"
#include "InputPort.h"
#include "SystemErrors.h"
#include "WidgetBase.h"

#define BaseWidget       ( Pablo::TopLevel()->getWidget() )
#define Round(x)         ( x >= 0.0 ? (int)(x + .5) : (int)(x - .5) )
#define Norm(i, p)       ( ( ((double)p) - fuMinValues.getElement(i))/ \
               (fuMaxValues.getElement(i) - fuMinValues.getElement(i) ) )

#define REALLOC_INTERVAL 10


ParallelCoordsFU::ParallelCoordsFU()
{
       portOption = ARRAY_ELEMENT;
}


ParallelCoordsFU::~ParallelCoordsFU()
{
}


Widget
ParallelCoordsFU::_setupPCPerCategoryDialog( int dimension )
{
       static PCConfigPerDimDataStruct data;

       Widget container = _setupPerCategoryDialog(  BaseWidget,
					  (ConfigPerDimDataStruct *)&data,
					  "Config");
       /* Initialize the areas */
       char label[STRING_MAX_LEN];
       int i;
       for (i=0; i<dimension; i++) {
          /* Category label */
	  XmString currentLabelString = XmStringCreateSimple( 
				   parallelCoordsForm->getCategoryLabel(i) );
	  XmListAddItemUnselected( data.scrolledLists[0],
				   currentLabelString, i+1 );
	  /* Min value */
	  sprintf(label, "%lf", fuMinValues.getElement(i) ); 
	  currentLabelString = XmStringCreateSimple( label ); 
	  XmListAddItemUnselected( data.scrolledLists[1],
				   currentLabelString, i+1 );

	  /* Max value */
 	  sprintf(label, "%lf", fuMaxValues.getElement(i) ); 
	  currentLabelString = XmStringCreateSimple( label ); 
	  XmListAddItemUnselected( data.scrolledLists[2],
				   currentLabelString, i+1 );
	  XmStringFree( currentLabelString );
       }

       return( container );
}


Widget
ParallelCoordsFU::_setupPCConfigWindow()
{
        static PCConfigDataStruct pcConfigDataStruct;

        pcConfigDataStruct.dialog = XmCreateFormDialog( BaseWidget,
						    "PCConfigDialogContainer",
						    NULL, 0 );
	Widget dialogActionArea = XtVaCreateManagedWidget(
						 "PCConfigActionArea",
						 xmFormWidgetClass,
						 pcConfigDataStruct.dialog,
						    NULL );
	/* Data dimension option */
	Widget dialogDimensionArea = 
	       multivariateWrapper.addTextFieldToDialog( dialogActionArea,
						         NULL,
					  "ConfigDialogDataDimension",
				    &(pcConfigDataStruct.totalDimension) );

	char dimensionString[STRING_MAX_LEN];
	sprintf(dimensionString, "%d", dimension );
	XmTextSetString( pcConfigDataStruct.totalDimension, dimensionString );
	
	/* Dataset name */
	Widget dialogDatasetNameArea =
	       multivariateWrapper.addTextFieldToDialog( dialogActionArea,
						         dialogDimensionArea,
					  "ConfigDialogDatasetName",
	  			         &(pcConfigDataStruct.datasetName) );
	XtVaSetValues( pcConfigDataStruct.datasetName,
		       XmNvalue, parallelCoordsForm->getPointSetName(),
		          NULL );

	 /* Data input option */
        Widget dialogDataInputOptionArea =
	        multivariateWrapper.addRadioToggleListToDialog(
						  dialogActionArea,
						  dialogDatasetNameArea,
					   "ConfigDialogDataInputOption",
					   2, 1,
			     &(pcConfigDataStruct.inputOptionRadioBox) );

	/* Additional options */
	Widget dialogAdditionalOptionsArea = 
	        multivariateWrapper.addToggleToDialog( dialogActionArea,
					     dialogDataInputOptionArea,
					     "ConfigDialogAdditionalOptions",
			     &(pcConfigDataStruct.perCategoryToggle) );

	/* separator */
	Widget paneSeparator = XtVaCreateManagedWidget(
	                            "PCConfigDialogPaneSeparator",
				     xmSeparatorWidgetClass,
				     pcConfigDataStruct.dialog,
				     XmNtopWidget, dialogActionArea,
					 NULL );
	/* Control area */
	Widget dialogControlArea = XtVaCreateManagedWidget( 
					       "DialogControlArea",
						xmFormWidgetClass,
				                pcConfigDataStruct.dialog,
					        XmNtopWidget, paneSeparator,
						XmNfractionBase, 1,
						   NULL );
	Widget dialogOKButton = XtVaCreateManagedWidget(
					       "DialogOKButton",
					        xmPushButtonWidgetClass,
						dialogControlArea,
        					XmNleftPosition, 0,
						XmNrightPosition, 1,
						    NULL );
        addCallback( dialogOKButton, XmNactivateCallback, &Callback::callback1,
		       this, &pcConfigDataStruct );
 	// Fix the size of the control button area
        Dimension controlHeight, marginHeight;
        XtVaGetValues( dialogOKButton,
		      XmNheight, &controlHeight,
		         NULL );
        XtVaGetValues( dialogControlArea,
		      XmNmarginHeight, &marginHeight,
		         NULL );
        XtVaSetValues( paneSeparator,
		      XmNbottomOffset, (int)(controlHeight + marginHeight),
		         NULL );

	return( pcConfigDataStruct.dialog );
}

void
ParallelCoordsFU::callback1( Widget /* callbackWidget */,
			     XtPointer ptr1, XtPointer /* ptr2 */ )
{
        PCConfigDataStruct *data = (PCConfigDataStruct *)ptr1;
	XtUnmanageChild( data->dialog );
	parallelCoordsForm->realize();

	/* Extract data dimension */
	char *dimensionText = XmTextFieldGetString( data->totalDimension );
	sscanf( dimensionText, "%d", &dimension );
        XtFree( dimensionText );

        parallelCoordsForm->setDimension( dimension );

	/* Dataset name */
	char *datasetName = XmTextFieldGetString( data->datasetName );
	parallelCoordsForm->setPointSetName( datasetName );

	/* additional options */
	static Widget perCategoryDialog = NULL;
	if ( XmToggleButtonGetState(data->perCategoryToggle) ){
	   if ( perCategoryDialog == NULL ) {
              perCategoryDialog = _setupPCPerCategoryDialog( dimension );
	   }
	   XtManageChild( perCategoryDialog );
	}

	/* Data allocation */
	if ( savedValues != NULL ) {
	   free( savedValues );
	}
	savedValues = (double *)malloc( REALLOC_INTERVAL * dimension *
				        sizeof(double) );
 	int i;
	for (i=0; i<dimension; i++) {
            double currentMinValue = 0.0, currentMaxValue = 0.0;
            fuMinValues.setElement(i, currentMinValue);
	    fuMaxValues.setElement(i, currentMaxValue); 		   
	}

	/* Extract data run mode */
	int numDataRunChoices;
	WidgetList childList;
	XtVaGetValues( data->inputOptionRadioBox,
		       XmNchildren, &childList,
		       XmNnumChildren, &numDataRunChoices,
		          NULL );
	for (i=0; i<numDataRunChoices; i++) {
            if ( XmToggleButtonGetState( childList[i] ) ) {
               portOption = i;
	       break;
	    }
        }

	/* Update fu min/max information */
	int dimMinValue, dimMaxValue;
	parallelCoordsForm->getPCAttr( XtNminValue, 
				      (XtArgVal)&dimMinValue );
	parallelCoordsForm->getPCAttr( XtNmaxValue,
				      (XtArgVal)&dimMaxValue );
	widgetMinValue = dimMinValue;
        widgetRange = dimMaxValue - dimMinValue;

	parallelCoordsForm->setPerfWidgetColors();
}

void
ParallelCoordsFU::callback2( Widget /* callbackWidget */,
			     XtPointer /* ptr1 */, XtPointer /* ptr2 */)
{
        _completePP();
	parallelCoordsForm->setPointCount( pointCount );
	
	int i,j;

	// Draw data
	int *scaledDataPoints = new int[ (pointCount+1) * dimension ];
        for (i=0; i<pointCount; i++) {
           for (j=0; j<dimension; j++) {
        	scaledDataPoints[i*dimension+j] = 
		                 Round( Norm(j,savedValues[i*dimension+j]) *
				      widgetRange ) + widgetMinValue;
	   }
       }     
       int pointsPerLine[1];
       pointsPerLine[0] = pointCount;
       parallelCoordsForm->setPCValues(scaledDataPoints, pointCount );
       delete scaledDataPoints;
}


void
ParallelCoordsFU::configure()
{
        static Widget configureShell;

	if ( configureShell == NULL ) {
           configureShell = _setupPCConfigWindow();
	}
	XtManageChild( configureShell );
} 


FunctionalUnit *
ParallelCoordsFU::copy()
{
         ParallelCoordsFU *copy = new ParallelCoordsFU();
	 return copy;
}


void
ParallelCoordsFU::fuCallback2( PCPreprocessOptions *data )
{
         preprocessMask = 0;
	 int i;
	 for (i=0; i<data->numberOptions; i++) {
             if (XmToggleButtonGetState(data->toggleOptions[i])) {
                 preprocessMask |= (int)pow(2,i);
	     }
	 }
}


void
ParallelCoordsFU::init()
{
         // Input Port
         inputPort = new InputPort( "Input" );
	 inputPort->addTraits( DOUBLE, 2 );
	 _addInputPort( inputPort );
	 
	 // Display Wrapper
	 parallelCoordsForm = new ParallelCoordsFormWrapper( BaseWidget,this );

	 // Tell the main interface to signal us when teh run is complete
	 Widget signalWidget = XtVaCreateWidget( "PCSignalWidget",
		                       xmPushButtonWidgetClass,
				       Pablo::TopLevel()->getWidget(),
				         NULL );
 	 Pablo::MainInterface()->addSignalWidget( signalWidget );
	 addCallback( signalWidget, XmNactivateCallback,
		      &Callback::callback2, this );
}
	 

Boolean_
ParallelCoordsFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            configure();
        } else {
	    int x, y, width, height;
	    char buf[STRING_MAX_LEN];

	    // ----- Comment line and numeric parameters -----
            fscanf(fp, "%[^\n]\n", buf );
	    
	    if ( fscanf( fp, "%d %d %d %d %d %d\n",
                      &x, &y, &width, &height, &dimension, &portOption) != 6) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
		parallelCoordsForm->setPCAttr( x,y, width, height );
	     	parallelCoordsForm->realize();
	        parallelCoordsForm->setDimension( dimension );
                parallelCoordsForm->setPerfWidgetColors(); 

		// Allocate initial space for data
		if ( savedValues != NULL ) {
		   free( savedValues );
		}
		savedValues = (double *)malloc(REALLOC_INTERVAL*dimension *
					       sizeof(double) );

		// Set per-dim widget ranges
		int dimMinValue, dimMaxValue;
		parallelCoordsForm->getPCAttr( XtNminValue,
					       (XtArgVal) &dimMinValue );
		parallelCoordsForm->getPCAttr( XtNmaxValue,
					       (XtArgVal) &dimMaxValue );
		widgetMinValue = dimMinValue;
		widgetRange = dimMaxValue - dimMinValue;

                isConfigured = TRUE_;

		// ----- Comment Line; Point set name
		fscanf( fp, "%[^\n]\n", buf );
		int labelSize;
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
                   strcpy( buf, "\0" );
		}
		fscanf( fp, "\n", buf );
		parallelCoordsForm->setPointSetName( buf );

		// ----- Comment Line; Dimension categories
		fscanf( fp, "%[^\n]\n", buf );
		int i;
		for (i=0; i<dimension; i++) {
                    fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		    if ( labelSize == 0 ) {
                       strcpy( buf, "\0" );
		    }
                    parallelCoordsForm->setCategoryLabel( buf, i );
		    fscanf( fp, "\n", buf );
	        }
		
		// ----- Comment Line; Min/Max Info
		fscanf( fp, "%[^\n]\n", buf );
		double currentMinValue, currentMaxValue;
		for (i=0; i<dimension; i++) {
                    fscanf(fp, "%lf %lf\n",&currentMinValue,&currentMaxValue );
		    fuMinValues.setElement(i, currentMinValue );
		    fuMaxValues.setElement(i, currentMaxValue );
		}
            } 
            fclose( fp );
        }

	if ( portOption == -1 ) {
	    configureOperation();
	}

	if ( ! isConfigured ) {
	    configure();
	}
	return isConfigured;
}


Boolean_
ParallelCoordsFU::ready()
{
        return isConfigured;
}


void
ParallelCoordsFU::run(  Boolean_& /* errorFlag */ )
{
        Assert( inputPort->valueAvailable() );
	pointCount++;

	inputValue = inputPort->getValue();
	Array *a = (Array *)inputValue;
	int *dimSizes = (int *)a->getDimSizes();
	int nPoints = dimSizes[0];
	
	double *dataPoint = new double[dimension];
	double *pDataPoint;
        int *scaledDataPoints = new int[ (nPoints+1)*dimension];

	switch ( portOption ) {
	    case FULL_ARRAY:
	          int i, j;
	          for (i=0; i<nPoints; i++) {
		      for (j=0; j<dimension; j++) {
		          dataPoint[j] = a->getCellValue(i,j);
		          dataPoint[j] = warnings.fuDataRangeWarningCheck( 
						this,
						dataPoint[j],
						fuMinValues.getElement(j),
						fuMaxValues.getElement(j) );
		          scaledDataPoints[i*dimension+j] =
			               Round( Norm(j,dataPoint[j]) *
					      widgetRange ) +  widgetMinValue;
		      }
	          }  
	          parallelCoordsForm->setPCValues( scaledDataPoints, nPoints );
	          break;

	    case ARRAY_ELEMENT:
	       if ( preprocessMask ) {
		  if ( ! ppData.isInitialized ) {
                     pDataPoint = _initializePPData(a);
		     ppData.isInitialized = True;
		  } else {
		     pDataPoint = _addPPData( a, nPoints );
		  }
	       // Otherwise do normal processing
               } else {
                  int *scaledDataPoint = new int[dimension];
               
                  for (j=0; j<dimension; j++) {
                     dataPoint[j] = a->getCellValue(nPoints-1,j);
		     dataPoint[j] = warnings.fuDataRangeWarningCheck( this,
		                                              dataPoint[j],
		        fuMinValues.getElement(j), fuMaxValues.getElement(j));

		     // Keep data within existing range
		     // Map point to value within the integer range
		     scaledDataPoint[ j ] =  Round( Norm(j,dataPoint[j]) *  
		                               widgetRange) + widgetMinValue;
	          }
	          parallelCoordsForm->setPCValues( scaledDataPoint );
		  pDataPoint = (double *)dataPoint;
                  delete scaledDataPoint;
               }
	       // Save the complete history
	       if ( (pointCount % REALLOC_INTERVAL) == 0 ) {
                  savedValues = (double *) realloc( (double *)savedValues, 
         	     (pointCount+REALLOC_INTERVAL)*dimension*sizeof(double) );
	       }	
	       for (i=0; i<dimension; i++) {
		  savedValues[ (pointCount-1)*dimension + i ] = pDataPoint[i];
	       }
	}
	delete dataPoint;
	delete scaledDataPoints;
}




Boolean_
ParallelCoordsFU::saveConfigToFile( const CString &fileName ) const
{
	Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
	} else {
            int x, y, width, height;
	    parallelCoordsForm->getPerfWidgetPosition( x, y,
						       width, height);

            fprintf( fp, "# X Y Wdth Hght DataDim PortOpt\n" );
            fprintf( fp, "%d %d %d %d %d %d \n", 
			 x, y, width, height, dimension, portOption );

	    fprintf( fp, "# Dataset Name\n" );
	    fprintf( fp, "%d %s\n",
		         strlen(parallelCoordsForm->getPointSetName() ),
		         parallelCoordsForm->getPointSetName() );

	    fprintf( fp, "# Dimension Categories\n");
	    int i;
	    for (i=0; i<dimension; i++) {
                fprintf(fp," %d %s\n",
			strlen(parallelCoordsForm->getCategoryLabel(i)),
			parallelCoordsForm->getCategoryLabel(i) );
	    }

	    fprintf( fp, "# Min/Max Values\n");
	    for (i=0; i<dimension; i++) {
	        fprintf(fp, "%lf %lf\n",
		    fuMinValues.getElement(i), fuMaxValues.getElement(i));
	    }
            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}


// Global class definition
const char *const ParallelCoordsFU::MY_CLASS = "ParallelCoordsFU";
