/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *          Daniel A. Reed (reed@cs.uiuc.edu)
 *          Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * LedFU.cc - A polymorophic functional unit for a simple LED that
 *            shows a specified number of states.  It is the discrete
 *            analog of a bargraph
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/LedFU.C,v 1.31 1994/03/15 16:41:44 aydt Exp $
 */

#include "LedFU.h"
#include "LedFormWrapper.h"	

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define Normalize(x) ( ( ((double) x) - minValue ) / (maxValue - minValue) )
#define Round(x)     ( (int)(x + .5) )

LedFU::LedFU()
{
	_setClassName( MY_CLASS );
	inputPort = NULL;
	dialog = NULL;
	ledForm = NULL;
}

LedFU::~LedFU()
{
	delete inputPort;
	delete ledForm;
	delete dialog;
}

void 
LedFU::configure()
{
#ifndef XtNminValue
#define	XtNminValue "minValue"
#define	XtNmaxValue "maxValue"
#endif

#ifndef XtNstickyHold
#define XtNstickyHold "stickyHold"
#endif

#ifndef XtNlevels
#define XtNlevels "levels"
#endif
	/*
	 * The expected minimum and maximum data values are selected by
	 * the user, and these are used to scale the actual value seen
	 * by the LedFU into the range of integer values accepted
	 * by the Led widget.  We never change the widget's minimum
	 * and maximum values.
	 */
	double 	    fuMin, fuMax;
	int	    fuHold, fuLevels;
	const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    // ----- Input dimension for scalar/vector/array support ------
            dimension = inputPort->getTraits().getDimension();

	    int wMin, wMax;

            // ---- Obtain the Led attributes from the Form child ---- 
	    ledForm->getLedAttr( XtNminValue, (XtArgVal) &wMin );
	    ledForm->getLedAttr( XtNmaxValue, (XtArgVal) &wMax );
	    ledForm->getLedAttr( XtNstickyHold, (XtArgVal) &fuHold );
	    ledForm->getLedAttr( XtNlevels, (XtArgVal) &fuLevels );
 
	    fuMin = wMin;
	    fuMax = wMax;
	    valueOffset = wMin;
	    valueRange = wMax - wMin;
	    isConfigured = TRUE_;
	} else {
	    fuMin = minValue;
	    fuMax = maxValue;
	    fuHold = stickyHold;
	    fuLevels = levels;
	}

	// ----- Configuration Phase ------ 
        ledForm->getLedLabel( L_BOTTOM_HORIZ, &labelBottom );
        ledForm->getLedLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	params.addTextParam( "Historical marker interval", 
				          BaseFUParamEntry::Integer, fuHold );
	params.addTextParam( "LED levels", BaseFUParamEntry::Integer, fuLevels);
 	params.addTextParam( "Minimum value", BaseFUParamEntry::Real, fuMin );
 	params.addTextParam( "Maximum value", BaseFUParamEntry::Real, fuMax );
        params.addTextParam( "Vertical axis label", 
					  BaseFUParamEntry::Str, labelLeft );
        params.addTextParam( "Horizontal axis label", 
					  BaseFUParamEntry::Str, labelBottom );

        CString configPanelName = "LED FU Config: " + getName();
        ParamConfig pc( Pablo::TopLevel(), params, configPanelName ); 
	pc.run();

	BaseFUParamEntry& stickyHoldEntry = 
				params.getEntry( "Historical marker interval" );
	BaseFUParamEntry& levelsEntry = params.getEntry( "LED levels" );
 	BaseFUParamEntry& minEntry = params.getEntry( "Minimum value" );
        BaseFUParamEntry& maxEntry = params.getEntry( "Maximum value" );
        BaseFUParamEntry& vertLabelEntry =
		                params.getEntry( "Vertical axis label" );
        BaseFUParamEntry& horizLabelEntry = 
		                params.getEntry( "Horizontal axis label" );
	
	if ( stickyHoldEntry.valueIsValid() ) {
	    fuHold = stickyHoldEntry.getValue().getInteger();
	} else {
	    warning( "Led historical marker was not valid.\n" );
	}
	if ( levelsEntry.valueIsValid() ) {
	    fuLevels = levelsEntry.getValue().getInteger();
	} else {
	    warning( "Levels value was not valid.\n" );
	}
	if ( minEntry.valueIsValid() ) {
            fuMin = minEntry.getValue().getReal();
        } else {
            warning( "Led minimum value was not valid.\n" );
	}
        if ( maxEntry.valueIsValid() ) {
            fuMax = maxEntry.getValue().getReal();
        } else {
            warning( "Led maximum value was not valid.\n" );
	}

	if ( fuMin >= fuMax ) {
	    warning( "Minimum >= Maximum.  Resetting Max to Min+1.\n");
	    fuMax = fuMin + 1;
	}

	// ----- Set Data to Configured Values for FU and Widget ------ 
	stickyHold = fuHold;
	levels = fuLevels;
	minValue = fuMin;
	maxValue = fuMax;

	ledForm->setLedLabel( fuMin, fuMax );
        ledForm->setLedLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
	ledForm->setLedLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        ledForm->setLedAttr( XtNstickyHold, fuHold );
	ledForm->setLedAttr( XtNlevels, fuLevels );
	ledForm->setPerfWidgetColors();
}

FunctionalUnit *
LedFU::copy()
{
	LedFU *copy = new LedFU();
	return copy;
}

void 
LedFU::fuCallback( int iRow, int iCol )
{
	if ( inputValue.isUndefined() ) {
	    return;
	}

	FUParams params;
        if ( dimension == 0 ) {
            params.addDisplayParam( "Current LED Value", inputValue );
        } else {
	    Value cValue;
	    Array *aInputData = (Array *)inputValue;
            const int *dimSizes = aInputData->getDimSizes();

            if ( dimension == 1 ) {
                if ( iCol < dimSizes[0] ) {
                   cValue = aInputData->getCellValue( iCol ); 
                   params.addDisplayParam( "Current LED Value", cValue );
                } else {
                   params.addDisplayParam( "Current LED Value", 
					    noDataMSG );
                }        
	        params.addDisplayParam( "LED Column", iCol );
            } else if (dimension == 2) {
                if ( ( iRow < dimSizes[0] ) && ( iCol < dimSizes[1] ) ) {
                   cValue = aInputData->getCellValue( iRow, iCol );
                   params.addDisplayParam( "Current LED Value", cValue );
                } else {
                   params.addDisplayParam( "Current LED Value", 
					    noDataMSG );
                }
                params.addDisplayParam( "LED Row", iRow );
                params.addDisplayParam( "LED Column", iCol );
            }
        }

        CString callbackTitle = "Callback: " + getName();
	ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params, 
					     callbackTitle );

}

void 
LedFU::init()
{
	/*
	 * Start by clearing up any 'leftovers' that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( ledForm != NULL ) {
	    delete ledForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	/*
	 * Here's the real code to initialize the FU
	 */
	isConfigured = FALSE_;
	inputValue = Value::NOVALUE;

	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 0 );
	inputPort->addTraits( INTEGER, 1 );
	inputPort->addTraits( INTEGER, 2 );
	inputPort->addTraits( FLOAT, 0 );
	inputPort->addTraits( FLOAT, 1 );
	inputPort->addTraits( FLOAT, 2 );
	inputPort->addTraits( DOUBLE, 0 );
	inputPort->addTraits( DOUBLE, 1 );
        inputPort->addTraits( DOUBLE, 2 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	ledForm= new LedFormWrapper( dialog, this, NullArgs, "Contents" );
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}

Boolean_		/* virtual */
LedFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );
        if ( fp == NULL ) {
            warning( "\nUnable to open %s:\n%s", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
            int labelSize;
	    char buf[LABEL_MAX_LEN];

	    // ---- Comment line and numeric paramaters ---- 
	    fscanf( fp,"%*[^\n]\n" ); 
            if ( fscanf( fp, "%lf %lf %d %d %d %d %d %d %d\n",
                           &minValue, &maxValue, &stickyHold, &levels,
			   &x, &y, &width, &height, &dimension ) != 9 ) {
                warning( "\nUnable to read configuration information from\n%s",
                          fileName.getValue() );
	    } else {
		dialog->unmanage();

		// ---- Comment Line; Horizontal Label Size and String ---- 
		labelSize = 0;
		fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
	        if ( labelSize == 0 ) {
                   strcpy( buf, "\0");
                }

                ledForm->setLedLabel( L_BOTTOM_HORIZ, buf );

		// ---- Comment Line; Vertical Label Size and String ---- 
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
		   strcpy( buf, "\0" );
                }
                ledForm->setLedLabel( L_LEFT_VERT, buf );

		/* 
		 * Recalculate the valueOffset and valueRange based on the 
		 * widget's min and max values. Use loaded parameters to set 
		 * other widget values.
		 */
     		int wMin, wMax;
                ledForm->getLedAttr( XtNminValue, (XtArgVal) &wMin );
                ledForm->getLedAttr( XtNmaxValue, (XtArgVal) &wMax );
                valueOffset = wMin;
                valueRange = wMax - wMin;

		ledForm->setLedLabel( minValue, maxValue );
		ledForm->setLedAttr( XtNstickyHold, stickyHold );
		ledForm->setLedAttr( XtNlevels, levels );
		ledForm->setPerfWidgetColors();

		dialog->manage();
		ledForm->setPerfWidgetPosition( x, y, width, height );

                isConfigured = TRUE_;
            } 
            fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured;
}

Boolean_ 
LedFU::ready()
{
	return TRUE_;
}

void 
LedFU::run(  Boolean_& /* errorFlag */ )
{
	Assert( inputPort->valueAvailable() );
	inputValue = inputPort->getValue();
       
	/* 
	 * We interactively vary the Led layout depending upon
	 * the incoming data size; dimension is fixed.
	 */
        switch ( dimension ) {
          case SCALAR:
	    {
            double inputValueScaled = warnings.fuDataRangeWarningCheck( this,
				       (double)inputValue, minValue, maxValue );
	    int val = Round( Normalize( inputValueScaled ) * valueRange );
	    val += valueOffset;    
	    ledForm->setDisplayValues( val );
	    }
            break;

          case VECTOR: 
          case ARRAY_2D:
	    {
            Array *aInputData = (Array *) inputValue;

	    // ---- Get the number of elements in the vector or array.
            const int *aDimSizes = aInputData->getDimSizes();
            int nRows, nCols;

	    if ( dimension == VECTOR ) {
          	nRows = 1;
		nCols = aDimSizes[0];
	    } else if ( dimension == ARRAY_2D) {
           	nRows = aDimSizes[0]; 
		nCols = aDimSizes[1];
	    }

	    // ---- Fill the vector/array structure with input data 
            double tmpValueScaled;
	    int nValues = nRows * nCols;
            int *vals = new int[ nValues ];

            for ( int i = 0; i < nValues; i++ ) { 
               tmpValueScaled = warnings.fuDataRangeWarningCheck( this,
                       			 (double)aInputData->getTheCellValue(i),
					 minValue, maxValue );
               vals[i] = Round( Normalize( tmpValueScaled ) * valueRange );
               vals[i] += valueOffset;
            }

            ledForm->setDisplayValues( nCols, nRows, vals );

            delete[] vals;
	    }
            break;
        }
}

Boolean_		/* virtual */
LedFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;

        FILE *fp = fopen( fileName.getValue(), "w" );
        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            int x, y, width, height;
	    const char *label;

	    ledForm->getPerfWidgetPosition( &x, &y, &width, &height );
	    fprintf( fp, "# Min Max Markers Levels X Y Width Height Dimen\n" );
	    fprintf( fp, "%lf %lf %d %d %d %d %d %d %d\n",
			 minValue, maxValue, stickyHold, levels, x, y, 
			 width, height, dimension );

            ledForm->getLedLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
            fprintf( fp, "%d %s\n", strlen( label), label );

            ledForm->getLedLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

	    fclose( fp );
            result = SUCCESS_;
        }

        return result;
}


/*
 *      Initialize the static data.   Only executed once.
 */
const char *const LedFU::MY_CLASS = "LedFU";
