/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/******************************************************************
 * KiviatUtilDisplay.C
 *
 *
 * Implementation of a UtilDisplay showing utilization on a
 * Kiviat chart.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/KiviatUtilDisplay.C,v 1.10 1994/02/25 04:33:13 aydt Exp $
 *
 *******************************************************************/
#include <stream.h>
#include <strstream.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include "KiviatUtilDisplay.h"
#include "Selection.h"
#include "UtilDisplayManager.h"


extern "C" {
#include "Kiviat.h"
}





KiviatUtilDisplay::KiviatUtilDisplay( char *name, UtilFU *ufu, Widget par ) 
	: UtilDisplay( name, ufu, par )
{

	vals = NULL;
	numAxes = 0;

	XtManageChild( getDialog() );
}



KiviatUtilDisplay::~KiviatUtilDisplay( void )
{
	delete[] vals;

	XtDestroyWidget( getDialog() );
}


/*
 * _handleResize()
 *
 * Resize the performance widgets so they fill their dialog, unless
 * resizing them would make them too small.
 * The parameter w is the ScrolledWindow widget.
 */
/* virtual */
void
KiviatUtilDisplay::_handleResize( Widget w )
{
        Widget cwind, wwind;            /* clip and work windows for w */
        Dimension cwidth, cheight;      /* dimensions of clip window */
        Dimension natWidth, natHeight;  /* natural size of nonscrolled dialog */
        Dimension labelWidth, labelHeight,/* RowColumn Dimension attributes */
                marginWidth, marginHeight, spacing;
        int numSetups;
        int i;


        /*
         * Check that there is a minimum size
         */
        getNaturalDimensions( natWidth, natHeight );
        if( natWidth == 0 || natHeight == 0 ) {
                return;                         // no, so do nothing
        }


        XtUnmanageChild( getWorkWindow() );


        /*
         * get new size of clip window
         */
        XtVaGetValues( w,
                        XmNworkWindow,  &wwind,
                        XmNclipWindow,  &cwind,
                        NULL );
        XtVaGetValues( cwind,
                        XtNwidth,       &cwidth,
                        XtNheight,      &cheight,
                        NULL );

        /*
         * check whether the clip window is larger or smaller than the
         * natural size of the workWindow
         */
        Dimension width = ( cwidth > natWidth ? cwidth : natWidth );
        Dimension height = ( cheight > natHeight ? cheight : natHeight );

        XtVaSetValues( getWorkWindow(),
                        XtNwidth,       width,
                        XtNheight,      height,
                        NULL );


        /*
         * find out the amount of width actually available to the performance
         * widgets ( i.e., not taken up by labels, margins, etc.
         */
        UtilsDisplayedType disp = getDisplayedType();
        if( disp == IndividualsShown ) {
                if( !(numSetups = getNumIndividuals()) ) {
                        return;
                }
        }
        else {
                numSetups = numAggregates;
        }

        /*
         * compute new width of performance widgets
         */
        XtVaGetValues( getVertLabelWidget(),
                        XtNwidth,       &labelWidth,
                        NULL );
        width -= labelWidth;
        XtVaGetValues( getRowCol(),
                        XmNmarginWidth, &marginWidth,
                        XmNmarginHeight, &marginHeight,
                        XmNspacing,     &spacing,
                        NULL );
        width -= ( 2 * marginWidth + numSetups * spacing );
        width -= 20;                    // to ensure the ScrollBars go away

        /*
         * compute new height of underlying RowColumn widget
         * ( This handles setting height of performance widgets )
         */
        XtVaGetValues( getHoriLabelWidget(),
                        XtNheight,      &labelHeight,
                        NULL );
        height -= ( 2*marginHeight + labelHeight + getPerfLabelDim() );
        height -= 20;                   // to ensure the ScrollBars go away


        /*
         * Compute proposed width of performance widgets.
         * Note we checked for numSetups == 0 above.
         */
        Dimension proposedWidth = width / numSetups;

        if( disp == IndividualsShown ) {
                for( i = 0; i < numSetups; i++ ) {
                        XtVaSetValues( individualUtils[i],
                                XtNwidth,       proposedWidth,
                                NULL );
                }
        }
        else {
                for( i = 0; i < numSetups; i++ ) {
                        XtVaSetValues( aggregateUtils[i],
                                XtNwidth,       proposedWidth,
                                NULL );
                }
        }
        XtManageChild( getWorkWindow() );
}



/*
 * changeValue()
 *
 * update the utilization value for PE proc to utilization util
 */
/* virtual */
void
KiviatUtilDisplay::changeValue( int proc, float util )
{
	/*
	 * compute which axis is used to represent proc
	 * don't use _indexIntoMap(), since it depends on number of
	 * individual display widgets to compute its index
	 */
	int i;
	int index = notDisplayed;
	for( i = 0; i < numAxes; i++ ) {
		if( map[i] == proc ) {
			index = i;
			break;
		}
	}



	if( index != notDisplayed ) {
		vals[index] = (int)(util*100);
		KiviatSetValues( (KiviatWidget)individualUtils[0], 
					numAxes, vals );
	}
}



/*
 * createPerfSetup()
 *
 * Creates a widget subtree for displaying utilization on a kiviat
 * performance widget
 */
/* virtual */
Widget
KiviatUtilDisplay::createPerfSetup( const char* /* label */ )
{
	Widget form = XtVaCreateWidget( "form",
			xmFormWidgetClass,	getRowCol(),
			NULL );

	/* no PE label to create... */
	setPerfLabelDim( 0 );
	
	Widget retval = XtVaCreateManagedWidget( "kiviat",
			kiviatWidgetClass,	form,
			XtNnumaxes,		4,
			NULL );
	XtAddCallback( retval, XtNselect,
		(XtCallbackProc)selectCB, this );

	XtManageChild( form );
	return retval;
}



/*
 * getType()
 *
 * Returns the character string indicating the type of this UtilDisplay
 */
/* virtual */
const char* const
KiviatUtilDisplay::getType( void ) const
{
	return "Kiviat";
}







/*
 * updatePerfWidgets()
 *
 * This method is called whenever the performance widget selection is
 * changed.
 * All deallocation of prior widgets is assumed to be complete before the
 * start of this method.
 */
/* virtual */
void
KiviatUtilDisplay::updatePerfWidgets( void )
{
	int i;
	Dimension width = 0;
	Dimension natWidth, natHeight;		// natural dimensions of dialog
	Position x, y;                          // screen loc of this display

	/* save screen location of this display */
	/* no use saving dimensions, they probably should be changed */
	XtVaGetValues( getDialog(),
			XmNx,   &x,
			XmNy,   &y,
			NULL );

	/* pop down the dialog */
	XtUnmanageChild( getDialog() );

	XtVaSetValues( getRowCol(),
			XmNorientation, XmHORIZONTAL,
			NULL );
	delete[] vals;


	/* determine how many individuals to show */
	int icount = 0;
	Selection *sel = getSelection();
	_makeMap( sel, icount );
	numAxes = icount;


	if( icount ) {
		setNumIndividuals( 1 );			// only one kiviat

		vals = new int[icount];
		for( i = 0; i < icount; i++ ) {
			vals[i] = 0;
		}
	
		/* allocate space for the kiviat widget */
		individualUtils = (Widget*)XtMalloc( sizeof( KiviatWidget ));

		individualUtils[0] = createPerfSetup( NULL );

		XtVaSetValues( individualUtils[0],
			XtNnumaxes,	icount,
			NULL );
		KiviatSetValues( (KiviatWidget)individualUtils[0], 
					icount, vals );
		
		XtVaGetValues( individualUtils[0],
				XtNwidth, &width,
				NULL );

	}
	else {
		setNumIndividuals( 0 );
	}

	/* find natural Dimensions for new work window */
	/* note that, we only set the natural dimensions for this work */
	/* window once, since it doesn't change ( i.e., changing the */
	/* range of PEs shown doesn't change the number of kiviats shown, */
	/* but rather the number of axes shown. */
	getNaturalDimensions( natWidth, natHeight );
	if( (natWidth == 0) || (natHeight == 0) ) {
		XtVaGetValues( getWorkWindow(),
			XtNwidth, 	&natWidth,
			XtNheight, 	&natHeight,
			NULL );
		setNaturalDimensions( natWidth, natHeight );
	}

	/* resize dialog based on natural size */
	initialResizeDialog();

	/* pop up the dialog */
	XtManageChild( getDialog() );

	/* reset screen location */
	XtVaSetValues( getDialog(),
			XmNx,   x,
			XmNy,   y,
			NULL );

}




