/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Ruth A. Aydt (aydt@cs.uiuc.edu )
 *          Daniel A. Reed (reed@cs.uiuc.edu)
 *          Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Contributing Author:	 Keith A. Shields (shields@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * KiviatFU.cc - A polymorophic functional unit for a dynamic kiviat graph
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/KiviatFU.C,v 1.29 1994/03/15 16:41:42 aydt Exp $
 */

#include "KiviatFU.h"
#include "KiviatFormWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"


#define Normalize(x)	( ( ((double) x) - minValue) / (maxValue - minValue) )
#define Round(x)     ( (int)(x + .5) )

KiviatFU::KiviatFU()
{
        _setClassName( MY_CLASS );
	inputPort = NULL;
	dialog = NULL;
	kiviatForm = NULL;
}

KiviatFU::~KiviatFU()
{
	delete inputPort;
	delete kiviatForm;
	delete dialog;
}

void 
KiviatFU::configure()			/* virtual */
{
#ifndef XtNminValue
#define XtNminValue "minValue"
#define XtNmaxValue "maxValue"
#endif

#ifndef XtNnumaxes
#define XtNnumaxes "numaxes"
#endif

	/*
	 * The expected minimum and maximum data values are selected by 
	 * the user, and these are used to scale the actual value seen
	 * by the Kiviat FU into the range of integer values accepted
	 * by the Kiviat widgets.  We never change the widget's minimum 
	 * and maximum values.
	 */
	
  	double      fuMin, fuMax; 
	int	    fuAxes;
	const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    // ----- Input dimension for scalar/vector/array support ------
	    dimension = inputPort->getTraits().getDimension();

	    // ----- Obtain the Kiviat attributes from the Form child -----
            int wMin, wMax;
	    kiviatForm->getKiviatAttr( XtNminValue, (XtArgVal) &wMin );
            kiviatForm->getKiviatAttr( XtNmaxValue, (XtArgVal) &wMax );
 	    kiviatForm->getKiviatAttr( XtNnumaxes, (XtArgVal) &fuAxes );

	    fuMin = wMin;
	    fuMax = wMax;
	    valueOffset = wMin;
	    valueRange = wMax - wMin;
	    isConfigured = TRUE_;
	} else {
	    fuMin = minValue;
            fuMax = maxValue;
	    fuAxes = numAxes;
	}

	// ----- Configuration Phase -----
	kiviatForm->getKiviatLabel( L_BOTTOM_HORIZ, &labelBottom );
	kiviatForm->getKiviatLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	params.addTextParam( "Number of kiviat axes", BaseFUParamEntry::Integer,
				fuAxes );
 	params.addTextParam( "Minimum value", BaseFUParamEntry::Real, fuMin );
        params.addTextParam( "Maximum value", BaseFUParamEntry::Real, fuMax );
	params.addTextParam( "Vertical axis label",
				BaseFUParamEntry::Str, labelLeft );
	params.addTextParam( "Horizontal axis label",
				BaseFUParamEntry::Str, labelBottom );

	CString configPanelName = "Kiviat FU Config: " + getName();
	ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

	BaseFUParamEntry& axisEntry = params.getEntry( "Number of kiviat axes");
 	BaseFUParamEntry& minEntry = params.getEntry( "Minimum value" );
        BaseFUParamEntry& maxEntry = params.getEntry( "Maximum value" );
        BaseFUParamEntry &vertLabelEntry =
		                     params.getEntry( "Vertical axis label" );
        BaseFUParamEntry &horizLabelEntry = 
		                     params.getEntry( "Horizontal axis label" );

	if ( axisEntry.valueIsValid() ) {
	    fuAxes = axisEntry.getValue().getInteger();
	} else {
	    warning( "Number of axes was not valid.\n" );
	}
	if ( minEntry.valueIsValid() ) {
            fuMin = minEntry.getValue().getReal();
        } else {
             warning( "Minimum value was not valid.\n" );
	}
        if ( maxEntry.valueIsValid() ) {
            fuMax = maxEntry.getValue().getReal();
        } else {
            warning( "Maximum value was not valid\n" );
	}

	if ( fuMin >= fuMax ) {
  	    warning( "Minimum >= Maximum.  Resetting Max to Min+1.\n");
	    fuMax = fuMin + 1;
	}
       
     	// ------ Set Data to Configured Values for FU and Widget ----- 
	numAxes = fuAxes;
        minValue = fuMin;
	maxValue = fuMax;

	kiviatForm->setKiviatLabel( fuMin, fuMax );
        kiviatForm->setKiviatLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
	kiviatForm->setKiviatLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
	kiviatForm->setKiviatAttr( XtNnumaxes, numAxes );
}

FunctionalUnit *		/* virtual */
KiviatFU::copy()
{
	KiviatFU *copy = new KiviatFU();
	return copy;
}

void 
KiviatFU::fuCallback( int axisSelected )
{
	if ( inputValue.isUndefined() ) {
	    return;
	}

        FUParams params;

	Value  cValue;
	Array* aInputData = (Array *)inputValue;
	const int *dimSizes = aInputData->getDimSizes();

	if ( axisSelected < dimSizes[0] ) {
	    aInputData->getCellValue( cValue, axisSelected );
 	    params.addDisplayParam( "Kiviat Value", cValue );
        } else {
	    params.addDisplayParam( "Kiviat Value", noDataMSG );
        }
        params.addDisplayParam( "Kiviat Axis Selected", axisSelected );

	CString callbackTitle = "Callback: " + getName();
	ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params, 
					     callbackTitle );
}

void 			/* virtual */
KiviatFU::init()
{
        /*
	 * Start by clearing up any 'leftovers' that will be around if
	 * init() isn't being called for the first time.  The goal is
	 * to start with a 'virgin' FU.
	 */
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( kiviatForm != NULL ) {
	    delete kiviatForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	/*
	 * Here's the real code to initialize the FU
	 */
	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 1 );
	inputPort->addTraits( FLOAT, 1 );
	inputPort->addTraits( DOUBLE, 1 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	kiviatForm = new KiviatFormWrapper( dialog, this, NullArgs, "Contents");
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );

	isConfigured = FALSE_;
}


Boolean_		/* virtual */
KiviatFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "\nUnable to open %s: %s", fileName.getValue(),
                                                errorString() );
        } else {
	    int	x, y, width, height;
            int labelSize;
	    char buf[LABEL_MAX_LEN];

	    // ----- Comment line and numeric parameters -----
            fscanf( fp, "%*[^\n]\n" );
   
	    if ( fscanf( fp, "%lf %lf %d %d %d %d %d %d\n",
                           &minValue, &maxValue, &numAxes,
			   &x, &y, &width, &height, &dimension ) != 8 ) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
		// ----- Comment Line; Horizontal Label Size and String -----
		labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                   strcpy( buf, "\0");
                }

                kiviatForm->setKiviatLabel( L_BOTTOM_HORIZ, buf );

                // ---- Comment Line; Vertical Label Size and String ----
                labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                   strcpy( buf, "\0" );
                }
                kiviatForm->setKiviatLabel( L_LEFT_VERT, buf );

                /*
                 * Recalculate the valueOffset and valueRange based on the
                 * widget's min and max values. Use loaded parameters to set
                 * other widget values.
                 */
		int wMin, wMax;
		kiviatForm->getKiviatAttr( XtNminValue, (XtArgVal) &wMin );
		kiviatForm->getKiviatAttr( XtNmaxValue, (XtArgVal) &wMax );
		valueOffset = wMin;
		valueRange = wMax - wMin;

		kiviatForm->setKiviatLabel( minValue, maxValue );
    		kiviatForm->setKiviatAttr( XtNnumaxes, numAxes );

		kiviatForm->setPerfWidgetPosition( x, y, width, height );
		isConfigured = TRUE_;
            } 
	    fclose( fp );
	}

	if ( ! isConfigured ) {
	    configure();
	}
	return isConfigured;
}

Boolean_ 		/* virtual */
KiviatFU::ready()
{
	return TRUE_;
}

void 			/* virtual */
KiviatFU::run(  Boolean_& /* errorFlag */ )
{
	Assert( inputPort->valueAvailable() );
	inputValue = inputPort->getValue();
 
	Array *aInputData = (Array *) inputValue;
	const int *aDimSizes = aInputData->getDimSizes();	 
	int inputElementCnt = aDimSizes[0];

        // Increase the number of kiviat axes if necessary.
	if ( inputElementCnt > numAxes ) {
	   numAxes = inputElementCnt;
        }

        // create vector and fill with scaled data values
        double tmpValueScaled;
        int *vals = new int[ numAxes ];

        for ( int i = 0; i < numAxes; i++ ) {
	    if ( i < inputElementCnt ) {
	        tmpValueScaled = warnings.fuDataRangeWarningCheck( this,
				      (double)aInputData->getCellValue( i ),
				      minValue, maxValue );
	        vals[i] = Round( Normalize( tmpValueScaled ) * valueRange );
	        vals[i] += valueOffset;
	    } else {
		vals[i] = valueOffset;
	    }
	}

        kiviatForm->setDisplayValues( numAxes, vals );
        delete[] vals;
}

Boolean_		/* virtual */
KiviatFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );
        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            int x, y, width, height;
	    const char *label;

	    kiviatForm->getPerfWidgetPosition( &x, &y, &width, &height );
            fprintf( fp, "# Min Max Axes X Y Width Height Dimension\n" );
            fprintf( fp, "%lf %lf %d %d %d %d %d %d\n", 
			 minValue, maxValue, numAxes,
			 x, y, width, height, dimension );

            kiviatForm->getKiviatLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

            kiviatForm->getKiviatLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

            fclose( fp );
            result = SUCCESS_;
        }

        return result;
}

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const KiviatFU::MY_CLASS = "KiviatFU";
